// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import 'chrome://resources/cr_components/localized_link/localized_link.js';
import 'chrome://resources/cr_elements/cr_dialog/cr_dialog.js';
import 'chrome://resources/cr_elements/cr_toolbar/cr_toolbar.js';
import '/strings.m.js';
import './app_content_item.js';
import './file_handling_item.js';
import './more_permissions_item.js';
import './run_on_os_login_item.js';
import './permission_item.js';
import './window_mode_item.js';
import './icons.html.js';
import './uninstall_button.js';
import './supported_links_item.js';
import './supported_links_overlapping_apps_dialog.js';
import './supported_links_dialog.js';
import { BrowserProxy } from 'chrome://resources/cr_components/app_management/browser_proxy.js';
import { getAppIcon } from 'chrome://resources/cr_components/app_management/util.js';
import { I18nMixinLit } from 'chrome://resources/cr_elements/i18n_mixin_lit.js';
import { assert } from 'chrome://resources/js/assert.js';
import { loadTimeData } from 'chrome://resources/js/load_time_data.js';
import { CrLitElement } from 'chrome://resources/lit/v3_0/lit.rollup.js';
import { getCss } from './app.css.js';
import { getHtml } from './app.html.js';
import { createDummyApp } from './web_app_settings_utils.js';
const AppElementBase = I18nMixinLit(CrLitElement);
// TODO(crbug.com/40213759): Investigate end-to-end WebAppSettings tests
export class AppElement extends AppElementBase {
    static get is() {
        return 'web-app-settings-app';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            app_: { type: Object },
            iconUrl_: { type: String },
            showSearch_: { type: Boolean },
            apps_: { type: Object },
        };
    }
    #app__accessor_storage = createDummyApp();
    get app_() { return this.#app__accessor_storage; }
    set app_(value) { this.#app__accessor_storage = value; }
    #apps__accessor_storage = {};
    get apps_() { return this.#apps__accessor_storage; }
    set apps_(value) { this.#apps__accessor_storage = value; }
    #iconUrl__accessor_storage = '';
    get iconUrl_() { return this.#iconUrl__accessor_storage; }
    set iconUrl_(value) { this.#iconUrl__accessor_storage = value; }
    #showSearch__accessor_storage = false;
    get showSearch_() { return this.#showSearch__accessor_storage; }
    set showSearch_(value) { this.#showSearch__accessor_storage = value; }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('app_')) {
            this.iconUrl_ = getAppIcon(this.app_);
        }
    }
    connectedCallback() {
        super.connectedCallback();
        this.apps_ = {};
        this.hidden = true;
        const urlPath = new URL(document.URL).pathname;
        if (urlPath.length <= 1) {
            return;
        }
        window.CrPolicyStrings = {
            controlledSettingPolicy: loadTimeData.getString('controlledSettingPolicy'),
        };
        const appId = urlPath.substring(1);
        BrowserProxy.getInstance().handler.getApp(appId).then((result) => {
            assert(result.app);
            this.app_ = result.app;
            this.hidden = false;
            if (this.isIsolatedWebApp_()) {
                // Launches an on-demand calculation of app size and data size; if one
                // of these two values ends up being different from the last known
                // value, informs this page via `onAppChanged`. This way we can quickly
                // render the cached data first without waiting for disk operations to
                // finish and then update the DOM later if necessary.
                BrowserProxy.getInstance().handler.updateAppSize(appId);
            }
        });
        BrowserProxy.getInstance().handler.getApps().then((result) => {
            for (const app of result.apps) {
                this.apps_[app.id] = app;
            }
        });
        // Listens to app update.
        const callbackRouter = BrowserProxy.getInstance().callbackRouter;
        callbackRouter.onAppChanged.addListener(this.onAppChanged_.bind(this));
        callbackRouter.onAppRemoved.addListener(this.onAppRemoved_.bind(this));
    }
    onAppChanged_(app) {
        if (app.id === this.app_.id) {
            this.app_ = app;
        }
        this.apps_ = { ...this.apps_, [app.id]: app };
    }
    onAppRemoved_(appId) {
        delete this.apps_[appId];
        this.apps_ = { ...this.apps_ };
    }
    getPermissionsHeader_() {
        return this.app_.formattedOrigin ?
            this.i18n('appManagementPermissionsWithOriginLabel', this.app_.formattedOrigin) :
            this.i18n('appManagementPermissionsLabel');
    }
    getTitle_() {
        return this.app_.title || '';
    }
    shouldShowSystemNotificationsSettingsLink_() {
        return this.app_.showSystemNotificationsSettingsLink;
    }
    isIsolatedWebApp_() {
        return this.app_.publisherId.startsWith('isolated-app://');
    }
    openNotificationsSystemSettings_(e) {
        // A place holder href with the value "#" is used to have a compliant link.
        // This prevents the browser from navigating the window to "#"
        e.detail.event.preventDefault();
        e.stopPropagation();
        // 
    }
}
customElements.define(AppElement.is, AppElement);
