// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview The 'certificate-manager' component is a newer way for
 * showing and managing TLS certificates. This is tied to the Chrome Root Store
 * and Chrome Cert Management Enterprise policies launch.
 */
import '/strings.m.js';
import './certificate_list.js';
import './certificate_confirmation_dialog.js';
import './certificate_info_dialog.js';
import './certificate_password_dialog.js';
import './certificate_subpage.js';
import './certificate_manager_icons.html.js';
import './certificate_manager_style.css.js';
import './crs_section.js';
import './local_certs_section.js';
// 
import '//resources/cr_elements/cr_icon/cr_icon.js';
import '//resources/cr_elements/cr_toast/cr_toast.js';
import '//resources/cr_elements/cr_toolbar/cr_toolbar.js';
import '//resources/cr_elements/cr_button/cr_button.js';
import '//resources/cr_elements/cr_link_row/cr_link_row.js';
import '//resources/cr_elements/cr_shared_style.css.js';
import '//resources/cr_elements/cr_shared_vars.css.js';
import '//resources/cr_elements/icons.html.js';
import '//resources/cr_elements/cr_page_selector/cr_page_selector.js';
import '//resources/cr_elements/cr_menu_selector/cr_menu_selector.js';
import '//resources/cr_elements/cr_nav_menu_item_style.css.js';
import '//resources/cr_elements/cr_page_host_style.css.js';
import { CrContainerShadowMixin } from '//resources/cr_elements/cr_container_shadow_mixin.js';
import { assert, assertNotReached } from '//resources/js/assert.js';
import { focusWithoutInk } from '//resources/js/focus_without_ink.js';
import { loadTimeData } from '//resources/js/load_time_data.js';
import { PluralStringProxyImpl } from '//resources/js/plural_string_proxy.js';
import { PromiseResolver } from '//resources/js/promise_resolver.js';
import { PolymerElement } from '//resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { getTemplate } from './certificate_manager.html.js';
import { CertificateSource } from './certificate_manager.mojom-webui.js';
import { CertificatesBrowserProxy } from './certificates_browser_proxy.js';
import { Page, RouteObserverMixin, Router } from './navigation.js';
const CertificateManagerElementBase = RouteObserverMixin(CrContainerShadowMixin(PolymerElement));
export class CertificateManagerElement extends CertificateManagerElementBase {
    static get is() {
        return 'certificate-manager';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            selectedPage_: String,
            enterpriseSubpageLists_: {
                type: (Array),
                value: () => {
                    return [
                        {
                            headerText: loadTimeData.getString('certificateManagerV2TrustedCertsList'),
                            certSource: CertificateSource.kEnterpriseTrustedCerts,
                        },
                        {
                            headerText: loadTimeData.getString('certificateManagerV2IntermediateCertsList'),
                            certSource: CertificateSource.kEnterpriseIntermediateCerts,
                            hideIfEmpty: true,
                        },
                        {
                            headerText: loadTimeData.getString('certificateManagerV2DistrustedCertsList'),
                            certSource: CertificateSource.kEnterpriseDistrustedCerts,
                            hideIfEmpty: true,
                        },
                    ];
                },
            },
            // 
            platformSubpageLists_: {
                type: (Array),
                value: () => {
                    return [
                        {
                            headerText: loadTimeData.getString('certificateManagerV2TrustedCertsList'),
                            certSource: CertificateSource.kPlatformUserTrustedCerts,
                        },
                        {
                            headerText: loadTimeData.getString('certificateManagerV2IntermediateCertsList'),
                            certSource: CertificateSource.kPlatformUserIntermediateCerts,
                            hideIfEmpty: true,
                        },
                        {
                            headerText: loadTimeData.getString('certificateManagerV2DistrustedCertsList'),
                            certSource: CertificateSource.kPlatformUserDistrustedCerts,
                            hideIfEmpty: true,
                        },
                    ];
                },
            },
            // 
            clientPlatformSubpageLists_: {
                type: (Array),
                // 
                // 
                computed: 'computeClientPlatformSubpageLists_()',
                // 
            },
            userSubpageLists_: {
                type: (Array),
                value: () => {
                    return [
                        {
                            headerText: loadTimeData.getString('certificateManagerV2TrustedCertsList'),
                            certSource: CertificateSource.kUserTrustedCerts,
                            showImport: loadTimeData.getBoolean('userCertsImportAllowed'),
                            certMetadataEditable: loadTimeData.getBoolean('userCertsImportAllowed'),
                        },
                        {
                            headerText: loadTimeData.getString('certificateManagerV2IntermediateCertsList'),
                            certSource: CertificateSource.kUserIntermediateCerts,
                            showImport: loadTimeData.getBoolean('userCertsImportAllowed'),
                            certMetadataEditable: loadTimeData.getBoolean('userCertsImportAllowed'),
                        },
                        {
                            headerText: loadTimeData.getString('certificateManagerV2DistrustedCertsList'),
                            certSource: CertificateSource.kUserDistrustedCerts,
                            showImport: loadTimeData.getBoolean('userCertsImportAllowed'),
                            certMetadataEditable: loadTimeData.getBoolean('userCertsImportAllowed'),
                        },
                    ];
                },
            },
            toastMessage_: String,
            showInfoDialog_: {
                type: Boolean,
                value: false,
            },
            infoDialogTitle_: String,
            infoDialogMessage_: String,
            showPasswordDialog_: {
                type: Boolean,
                value: false,
            },
            confirmationDialogTitle_: String,
            confirmationDialogMessage_: String,
            showConfirmationDialog_: {
                type: Boolean,
                value: false,
            },
            showSearch_: {
                type: Boolean,
                value: false,
            },
            // 
            certificateSourceEnum_: {
                type: Object,
                value: CertificateSource,
            },
            pageEnum_: {
                type: Object,
                value: Page,
            },
            numPlatformClientCertsString_: String,
        };
    }
    passwordEntryResolver_ = null;
    confirmationDialogResolver_ = null;
    ready() {
        super.ready();
        const proxy = CertificatesBrowserProxy.getInstance();
        proxy.callbackRouter.askForImportPassword.addListener(this.onAskForImportPassword_.bind(this));
        proxy.callbackRouter.askForConfirmation.addListener(this.onAskForConfirmation_.bind(this));
        proxy.callbackRouter.triggerReload.addListener(this.onTriggerReload_.bind(this));
        this.getClientCertCount_();
    }
    onTriggerReload_(certSources) {
        if (certSources.includes(CertificateSource.kPlatformClientCert)) {
            this.getClientCertCount_();
        }
    }
    getClientCertCount_() {
        CertificatesBrowserProxy.getInstance()
            .handler.getCertificates(CertificateSource.kPlatformClientCert)
            .then((results) => {
            PluralStringProxyImpl.getInstance()
                .getPluralString('certificateManagerV2NumCerts', results.certs.length)
                .then(label => {
                this.numPlatformClientCertsString_ = label;
            });
        });
    }
    onAskForImportPassword_() {
        this.showPasswordDialog_ = true;
        assert(this.passwordEntryResolver_ === null);
        this.passwordEntryResolver_ = new PromiseResolver();
        return this.passwordEntryResolver_.promise;
    }
    onPasswordDialogClose_() {
        const passwordDialog = this.shadowRoot.querySelector('#passwordDialog');
        assert(passwordDialog);
        assert(this.passwordEntryResolver_);
        this.passwordEntryResolver_.resolve({ password: passwordDialog.value() });
        this.passwordEntryResolver_ = null;
        this.showPasswordDialog_ = false;
    }
    onAskForConfirmation_(title, message) {
        this.confirmationDialogTitle_ = title;
        this.confirmationDialogMessage_ = message;
        this.showConfirmationDialog_ = true;
        assert(this.confirmationDialogResolver_ === null);
        this.confirmationDialogResolver_ =
            new PromiseResolver();
        return this.confirmationDialogResolver_.promise;
    }
    onConfirmationDialogClose_() {
        const confirmationDialog = this.shadowRoot.querySelector('#confirmationDialog');
        assert(confirmationDialog);
        assert(this.confirmationDialogResolver_);
        this.confirmationDialogResolver_.resolve({ confirmed: confirmationDialog.wasConfirmed() });
        this.confirmationDialogResolver_ = null;
        this.showConfirmationDialog_ = false;
    }
    onHashCopied_() {
        this.toastMessage_ =
            loadTimeData.getString('certificateManagerV2HashCopiedToast');
        this.$.toast.show();
    }
    // Prevent clicks on sidebar items from navigating and therefore reloading
    // the page.
    onMenuItemClick_(e) {
        e.preventDefault();
    }
    async currentRouteChanged(route, oldRoute) {
        this.selectedPage_ = route.page;
        if (route.isSubpage()) {
            // Sub-pages always show the top shadow, regardless of scroll position.
            this.enableScrollObservation(false);
            this.setForceDropShadows(true);
        }
        else {
            // Main page uses scroll position to determine whether a shadow should
            // be shown.
            this.enableScrollObservation(true);
            this.setForceDropShadows(false);
        }
        if (route.isSubpage()) {
            await this.$.main.updateComplete;
            switch (route.page) {
                case Page.ADMIN_CERTS:
                    this.$.adminCertsSection.setInitialFocus();
                    break;
                // 
                case Page.PLATFORM_CERTS:
                    this.$.platformCertsSection.setInitialFocus();
                    break;
                // 
                case Page.PLATFORM_CLIENT_CERTS:
                    this.$.platformClientCertsSection.setInitialFocus();
                    break;
                case Page.USER_CERTS:
                    this.$.userCertsSection.setInitialFocus();
                    break;
                default:
                    assertNotReached();
            }
        }
        else if (oldRoute.isSubpage()) {
            // If we're navigating back from a subpage, we may need to fiddle
            // with the focus element if we're going back to its parent page.
            switch (oldRoute.page) {
                case Page.ADMIN_CERTS:
                    if (route.page === Page.LOCAL_CERTS) {
                        await this.$.main.updateComplete;
                        this.$.localCertSection.setFocusToLinkRow(oldRoute.page);
                    }
                    break;
                // 
                case Page.PLATFORM_CERTS:
                    if (route.page === Page.LOCAL_CERTS) {
                        await this.$.main.updateComplete;
                        this.$.localCertSection.setFocusToLinkRow(oldRoute.page);
                    }
                    break;
                // 
                case Page.PLATFORM_CLIENT_CERTS:
                    if (route.page === Page.CLIENT_CERTS) {
                        await this.$.main.updateComplete;
                        focusWithoutInk(this.$.viewOsImportedClientCerts);
                    }
                    break;
                case Page.USER_CERTS:
                    if (route.page === Page.LOCAL_CERTS) {
                        await this.$.main.updateComplete;
                        this.$.localCertSection.setFocusToLinkRow(oldRoute.page);
                    }
                    break;
                default:
                    assertNotReached();
            }
        }
    }
    onMenuItemActivate_(e) {
        const page = e.detail.item.getAttribute('href');
        assert(page, 'Page is not available');
        Router.getInstance().navigateTo(page.substring(1));
    }
    getSelectedTopLevelHref_() {
        switch (this.selectedPage_) {
            // 
            case Page.PLATFORM_CERTS:
            // 
            case Page.ADMIN_CERTS:
            case Page.USER_CERTS:
                return this.generateHrefForPage_(Page.LOCAL_CERTS);
            case Page.PLATFORM_CLIENT_CERTS:
                return this.generateHrefForPage_(Page.CLIENT_CERTS);
            default:
                return this.generateHrefForPage_(this.selectedPage_);
        }
    }
    generateHrefForPage_(p) {
        return '/' + p;
    }
    onClientPlatformCertsLinkRowClick_(e) {
        e.preventDefault();
        Router.getInstance().navigateTo(Page.PLATFORM_CLIENT_CERTS);
    }
    onImportResult_(e) {
        const result = e.detail;
        if (result === null) {
            return;
        }
        if (result.error !== undefined) {
            this.infoDialogTitle_ =
                loadTimeData.getString('certificateManagerV2ImportErrorTitle');
            this.infoDialogMessage_ = result.error;
            this.showInfoDialog_ = true;
        }
    }
    onDeleteResult_(e) {
        const result = e.detail;
        if (result === null) {
            return;
        }
        if (result.error !== undefined) {
            this.infoDialogTitle_ =
                loadTimeData.getString('certificateManagerV2DeleteErrorTitle');
            this.infoDialogMessage_ = result.error;
            this.showInfoDialog_ = true;
        }
    }
    onInfoDialogClose_() {
        this.showInfoDialog_ = false;
    }
    computeClientPlatformSubpageLists_() {
        return [
            {
                headerText: loadTimeData.getString('certificateManagerV2ClientCertsFromPlatform'),
                certSource: CertificateSource.kPlatformClientCert,
                hideExport: true,
                // 
                // 
                showImport: true,
                hideHeader: false,
                // 
                // 
            },
        ];
    }
}
customElements.define(CertificateManagerElement.is, CertificateManagerElement);
