// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import './action_chips/action_chips.js';
import './iframe.js';
import './logo.js';
import '/strings.m.js';
import 'chrome://new-tab-page/shared/customize_buttons/customize_buttons.js';
import 'chrome://resources/cr_components/searchbox/searchbox.js';
import 'chrome://resources/cr_elements/cr_button/cr_button.js';
import 'chrome://resources/cr_components/composebox/composebox.js';
import { ColorChangeUpdater } from 'chrome://resources/cr_components/color_change_listener/colors_css_updater.js';
import { ComposeboxMode } from 'chrome://resources/cr_components/composebox/contextual_entrypoint_and_carousel.js';
import { HelpBubbleMixinLit } from 'chrome://resources/cr_components/help_bubble/help_bubble_mixin_lit.js';
import { assert } from 'chrome://resources/js/assert.js';
import { Command } from 'chrome://resources/js/browser_command.mojom-webui.js';
import { BrowserCommandProxy } from 'chrome://resources/js/browser_command/browser_command_proxy.js';
import { hexColorToSkColor, skColorToRgba } from 'chrome://resources/js/color_utils.js';
import { EventTracker } from 'chrome://resources/js/event_tracker.js';
import { FocusOutlineManager } from 'chrome://resources/js/focus_outline_manager.js';
import { loadTimeData } from 'chrome://resources/js/load_time_data.js';
import { getTrustedScriptURL } from 'chrome://resources/js/static_types.js';
import { CrLitElement } from 'chrome://resources/lit/v3_0/lit.rollup.js';
import { getCss } from './app.css.js';
import { getHtml } from './app.html.js';
import { BackgroundManager } from './background_manager.js';
import { SidePanelOpenTrigger } from './customize_buttons.mojom-webui.js';
import { CustomizeButtonsProxy } from './customize_buttons_proxy.js';
import { CustomizeChromeSection } from './customize_chrome.mojom-webui.js';
import { CustomizeDialogPage } from './customize_dialog_types.js';
import { recordBoolean, recordDuration, recordEnumeration, recordLinearValue, recordLoadDuration, recordSparseValueWithPersistentHash } from './metrics_utils.js';
import { ParentTrustedDocumentProxy } from './modules/microsoft_auth_frame_connector.js';
import { IphFeature, NtpBackgroundImageSource } from './new_tab_page.mojom-webui.js';
import { NewTabPageProxy } from './new_tab_page_proxy.js';
import { ShowNtpPromosResult } from './ntp_promo.mojom-webui.js';
import { $$ } from './utils.js';
import { Action as VoiceAction, recordVoiceAction } from './voice_search_overlay.js';
import { WindowProxy } from './window_proxy.js';
var ModuleLoadStatus;
(function (ModuleLoadStatus) {
    ModuleLoadStatus[ModuleLoadStatus["MODULE_LOAD_IN_PROGRESS"] = 0] = "MODULE_LOAD_IN_PROGRESS";
    ModuleLoadStatus[ModuleLoadStatus["MODULE_LOAD_NOT_ATTEMPTED"] = 1] = "MODULE_LOAD_NOT_ATTEMPTED";
    ModuleLoadStatus[ModuleLoadStatus["MODULE_LOAD_COMPLETE"] = 2] = "MODULE_LOAD_COMPLETE";
})(ModuleLoadStatus || (ModuleLoadStatus = {}));
/**
 * Elements on the NTP. This enum must match the numbering for NTPElement in
 * enums.xml. These values are persisted to logs. Entries should not be
 * renumbered, removed or reused.
 */
export var NtpElement;
(function (NtpElement) {
    NtpElement[NtpElement["OTHER"] = 0] = "OTHER";
    NtpElement[NtpElement["BACKGROUND"] = 1] = "BACKGROUND";
    NtpElement[NtpElement["ONE_GOOGLE_BAR"] = 2] = "ONE_GOOGLE_BAR";
    NtpElement[NtpElement["LOGO"] = 3] = "LOGO";
    NtpElement[NtpElement["REALBOX"] = 4] = "REALBOX";
    NtpElement[NtpElement["MOST_VISITED"] = 5] = "MOST_VISITED";
    NtpElement[NtpElement["MIDDLE_SLOT_PROMO"] = 6] = "MIDDLE_SLOT_PROMO";
    NtpElement[NtpElement["MODULE"] = 7] = "MODULE";
    NtpElement[NtpElement["CUSTOMIZE"] = 8] = "CUSTOMIZE";
    NtpElement[NtpElement["CUSTOMIZE_BUTTON"] = 9] = "CUSTOMIZE_BUTTON";
    NtpElement[NtpElement["CUSTOMIZE_DIALOG"] = 10] = "CUSTOMIZE_DIALOG";
    NtpElement[NtpElement["WALLPAPER_SEARCH_BUTTON"] = 11] = "WALLPAPER_SEARCH_BUTTON";
    NtpElement[NtpElement["MAX_VALUE"] = 11] = "MAX_VALUE";
})(NtpElement || (NtpElement = {}));
/**
 * Customize Chrome entry points. This enum must match the numbering for
 * NtpCustomizeChromeEntryPoint in enums.xml. These values are persisted to
 * logs. Entries should not be renumbered, removed or reused.
 */
export var NtpCustomizeChromeEntryPoint;
(function (NtpCustomizeChromeEntryPoint) {
    NtpCustomizeChromeEntryPoint[NtpCustomizeChromeEntryPoint["CUSTOMIZE_BUTTON"] = 0] = "CUSTOMIZE_BUTTON";
    NtpCustomizeChromeEntryPoint[NtpCustomizeChromeEntryPoint["MODULE"] = 1] = "MODULE";
    NtpCustomizeChromeEntryPoint[NtpCustomizeChromeEntryPoint["URL"] = 2] = "URL";
    NtpCustomizeChromeEntryPoint[NtpCustomizeChromeEntryPoint["WALLPAPER_SEARCH_BUTTON"] = 3] = "WALLPAPER_SEARCH_BUTTON";
    NtpCustomizeChromeEntryPoint[NtpCustomizeChromeEntryPoint["MAX_VALUE"] = 3] = "MAX_VALUE";
})(NtpCustomizeChromeEntryPoint || (NtpCustomizeChromeEntryPoint = {}));
/**
 * Defines the conditions that hide the wallpaper search button on the New Tab
 * Page.
 */
var NtpWallpaperSearchButtonHideCondition;
(function (NtpWallpaperSearchButtonHideCondition) {
    NtpWallpaperSearchButtonHideCondition[NtpWallpaperSearchButtonHideCondition["NONE"] = 0] = "NONE";
    NtpWallpaperSearchButtonHideCondition[NtpWallpaperSearchButtonHideCondition["BACKGROUND_IMAGE_SET"] = 1] = "BACKGROUND_IMAGE_SET";
    NtpWallpaperSearchButtonHideCondition[NtpWallpaperSearchButtonHideCondition["THEME_SET"] = 2] = "THEME_SET";
    NtpWallpaperSearchButtonHideCondition[NtpWallpaperSearchButtonHideCondition["MAX_VALUE"] = 2] = "MAX_VALUE";
})(NtpWallpaperSearchButtonHideCondition || (NtpWallpaperSearchButtonHideCondition = {}));
const CUSTOMIZE_URL_PARAM = 'customize';
const OGB_IFRAME_ORIGIN = 'chrome-untrusted://new-tab-page';
const MSAL_IFRAME_ORIGIN = 'chrome-untrusted://ntp-microsoft-auth';
export const CUSTOMIZE_CHROME_BUTTON_ELEMENT_ID = 'CustomizeButtonsHandler::kCustomizeChromeButtonElementId';
// 900px ~= 561px (max value for --ntp-search-box-width) * 1.5 + some margin.
const realboxCanShowSecondarySideMediaQueryList = window.matchMedia('(min-width: 900px)');
function recordClick(element) {
    recordEnumeration('NewTabPage.Click', element, NtpElement.MAX_VALUE + 1);
}
function recordCustomizeChromeOpen(element) {
    recordEnumeration('NewTabPage.CustomizeChromeOpened', element, NtpCustomizeChromeEntryPoint.MAX_VALUE + 1);
}
// Adds a <script> tag that holds the lazy loaded code.
function ensureLazyLoaded() {
    const script = document.createElement('script');
    script.type = 'module';
    script.src = getTrustedScriptURL `./lazy_load.js`;
    document.body.appendChild(script);
}
function recordShowBrowserPromosResult(result) {
    recordEnumeration('UserEducation.NtpPromos.ShowResult', result, ShowNtpPromosResult.MAX_VALUE + 1);
}
const AppElementBase = HelpBubbleMixinLit(CrLitElement);
export class AppElement extends AppElementBase {
    static get is() {
        return 'ntp-app';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            oneGoogleBarIframeOrigin_: { type: String },
            oneGoogleBarIframePath_: { type: String },
            oneGoogleBarLoaded_: { type: Boolean },
            theme_: { type: Object },
            showCustomize_: { type: Boolean },
            showCustomizeChromeText_: { type: Boolean },
            showWallpaperSearch_: { type: Boolean },
            isFooterVisible_: { type: Boolean },
            selectedCustomizeDialogPage_: { type: String },
            showVoiceSearchOverlay_: { type: Boolean },
            showBackgroundImage_: {
                reflect: true,
                type: Boolean,
            },
            backgroundImageAttribution1_: { type: String },
            backgroundImageAttribution2_: { type: String },
            backgroundImageAttributionUrl_: { type: String },
            // Used in cr-searchbox component via host-context.
            colorSourceIsBaseline: { type: Boolean },
            logoColor_: { type: String },
            singleColoredLogo_: { type: Boolean },
            /**
             * Whether the secondary side can be shown based on the feature state and
             * the width available to the dropdown for the ntp searchbox.
             */
            realboxCanShowSecondarySide: {
                type: Boolean,
                reflect: true,
            },
            /**
             * Whether the searchbox secondary side was at any point available to
             * be shown.
             */
            realboxHadSecondarySide: {
                type: Boolean,
                reflect: true,
                notify: true,
            },
            composeboxCloseByClickOutside_: { type: Boolean },
            composeboxEnabled: { type: Boolean },
            composeButtonEnabled: { type: Boolean },
            browserPromoType_: { type: String },
            browserPromoLimit_: { type: Number },
            browserPromoCompletedLimit_: { type: Number },
            showBrowserPromo_: { type: Boolean },
            realboxShown_: { type: Boolean },
            logoEnabled_: { type: Boolean },
            oneGoogleBarEnabled_: { type: Boolean },
            shortcutsEnabled_: { type: Boolean },
            middleSlotPromoEnabled_: { type: Boolean },
            modulesEnabled_: { type: Boolean },
            middleSlotPromoLoaded_: { type: Boolean },
            modulesLoadedStatus_: {
                type: Number,
                reflect: true,
            },
            modulesShownToUser: {
                type: Boolean,
                reflect: true,
            },
            microsoftModuleEnabled_: { type: Boolean },
            microsoftAuthIframePath_: { type: String },
            ntpRealboxNextEnabled_: {
                type: Boolean,
                reflect: true,
            },
            /**
             * In order to avoid flicker, the promo and modules are hidden until both
             * are loaded. If modules are disabled, the promo is shown as soon as it
             * is loaded.
             */
            promoAndModulesLoaded_: { type: Boolean },
            realboxLayoutMode_: {
                type: String,
                reflect: true,
            },
            searchboxCyclingPlaceholders_: {
                type: Boolean,
            },
            showComposebox_: {
                type: Boolean,
                reflect: true,
            },
            showLensUploadDialog_: { type: Boolean },
            /**
             * If true, renders additional elements that were not deemed crucial to
             * to show up immediately on load.
             */
            lazyRender_: { type: Boolean },
            scrolledToTop_: { type: Boolean },
            wallpaperSearchButtonAnimationEnabled_: { type: Boolean },
            wallpaperSearchButtonEnabled_: { type: Boolean },
            showWallpaperSearchButton_: { type: Boolean },
            /**
             * Whether the composebox has been opened at least once.
             */
            wasComposeboxOpened_: { type: Boolean },
            ntpNextFeaturesEnabled_: { type: Boolean },
            dropdownIsVisible_: { type: Boolean, reflect: true },
            searchboxInputFocused_: { type: Boolean },
            composeboxInputFocused_: { type: Boolean },
            /**
             * Whether the scrim is shown in Realbox Next.
             */
            showScrim_: { type: Boolean, reflect: true },
        };
    }
    #oneGoogleBarIframeOrigin__accessor_storage = OGB_IFRAME_ORIGIN;
    get oneGoogleBarIframeOrigin_() { return this.#oneGoogleBarIframeOrigin__accessor_storage; }
    set oneGoogleBarIframeOrigin_(value) { this.#oneGoogleBarIframeOrigin__accessor_storage = value; }
    #oneGoogleBarIframePath__accessor_storage;
    get oneGoogleBarIframePath_() { return this.#oneGoogleBarIframePath__accessor_storage; }
    set oneGoogleBarIframePath_(value) { this.#oneGoogleBarIframePath__accessor_storage = value; }
    #oneGoogleBarLoaded__accessor_storage = false;
    get oneGoogleBarLoaded_() { return this.#oneGoogleBarLoaded__accessor_storage; }
    set oneGoogleBarLoaded_(value) { this.#oneGoogleBarLoaded__accessor_storage = value; }
    #theme__accessor_storage = null;
    get theme_() { return this.#theme__accessor_storage; }
    set theme_(value) { this.#theme__accessor_storage = value; }
    #showCustomize__accessor_storage = false;
    get showCustomize_() { return this.#showCustomize__accessor_storage; }
    set showCustomize_(value) { this.#showCustomize__accessor_storage = value; }
    #showCustomizeChromeText__accessor_storage = false;
    get showCustomizeChromeText_() { return this.#showCustomizeChromeText__accessor_storage; }
    set showCustomizeChromeText_(value) { this.#showCustomizeChromeText__accessor_storage = value; }
    #showWallpaperSearch__accessor_storage = false;
    get showWallpaperSearch_() { return this.#showWallpaperSearch__accessor_storage; }
    set showWallpaperSearch_(value) { this.#showWallpaperSearch__accessor_storage = value; }
    #selectedCustomizeDialogPage__accessor_storage = null;
    get selectedCustomizeDialogPage_() { return this.#selectedCustomizeDialogPage__accessor_storage; }
    set selectedCustomizeDialogPage_(value) { this.#selectedCustomizeDialogPage__accessor_storage = value; }
    #showVoiceSearchOverlay__accessor_storage = false;
    get showVoiceSearchOverlay_() { return this.#showVoiceSearchOverlay__accessor_storage; }
    set showVoiceSearchOverlay_(value) { this.#showVoiceSearchOverlay__accessor_storage = value; }
    #showBackgroundImage__accessor_storage = false;
    get showBackgroundImage_() { return this.#showBackgroundImage__accessor_storage; }
    set showBackgroundImage_(value) { this.#showBackgroundImage__accessor_storage = value; }
    #backgroundImageAttribution1__accessor_storage = '';
    get backgroundImageAttribution1_() { return this.#backgroundImageAttribution1__accessor_storage; }
    set backgroundImageAttribution1_(value) { this.#backgroundImageAttribution1__accessor_storage = value; }
    #backgroundImageAttribution2__accessor_storage = '';
    get backgroundImageAttribution2_() { return this.#backgroundImageAttribution2__accessor_storage; }
    set backgroundImageAttribution2_(value) { this.#backgroundImageAttribution2__accessor_storage = value; }
    #backgroundImageAttributionUrl__accessor_storage = '';
    get backgroundImageAttributionUrl_() { return this.#backgroundImageAttributionUrl__accessor_storage; }
    set backgroundImageAttributionUrl_(value) { this.#backgroundImageAttributionUrl__accessor_storage = value; }
    #colorSourceIsBaseline_accessor_storage = false;
    get colorSourceIsBaseline() { return this.#colorSourceIsBaseline_accessor_storage; }
    set colorSourceIsBaseline(value) { this.#colorSourceIsBaseline_accessor_storage = value; }
    #logoColor__accessor_storage = null;
    get logoColor_() { return this.#logoColor__accessor_storage; }
    set logoColor_(value) { this.#logoColor__accessor_storage = value; }
    #singleColoredLogo__accessor_storage = false;
    get singleColoredLogo_() { return this.#singleColoredLogo__accessor_storage; }
    set singleColoredLogo_(value) { this.#singleColoredLogo__accessor_storage = value; }
    #realboxCanShowSecondarySide_accessor_storage = false;
    get realboxCanShowSecondarySide() { return this.#realboxCanShowSecondarySide_accessor_storage; }
    set realboxCanShowSecondarySide(value) { this.#realboxCanShowSecondarySide_accessor_storage = value; }
    #realboxHadSecondarySide_accessor_storage = false;
    get realboxHadSecondarySide() { return this.#realboxHadSecondarySide_accessor_storage; }
    set realboxHadSecondarySide(value) { this.#realboxHadSecondarySide_accessor_storage = value; }
    #realboxShown__accessor_storage = false;
    get realboxShown_() { return this.#realboxShown__accessor_storage; }
    set realboxShown_(value) { this.#realboxShown__accessor_storage = value; }
    #wasComposeboxOpened__accessor_storage = false;
    get wasComposeboxOpened_() { return this.#wasComposeboxOpened__accessor_storage; }
    set wasComposeboxOpened_(value) { this.#wasComposeboxOpened__accessor_storage = value; }
    #showLensUploadDialog__accessor_storage = false;
    get showLensUploadDialog_() { return this.#showLensUploadDialog__accessor_storage; }
    set showLensUploadDialog_(value) { this.#showLensUploadDialog__accessor_storage = value; }
    #showComposebox__accessor_storage = false;
    get showComposebox_() { return this.#showComposebox__accessor_storage; }
    set showComposebox_(value) { this.#showComposebox__accessor_storage = value; }
    #logoEnabled__accessor_storage = loadTimeData.getBoolean('logoEnabled');
    get logoEnabled_() { return this.#logoEnabled__accessor_storage; }
    set logoEnabled_(value) { this.#logoEnabled__accessor_storage = value; }
    #oneGoogleBarEnabled__accessor_storage = loadTimeData.getBoolean('oneGoogleBarEnabled');
    get oneGoogleBarEnabled_() { return this.#oneGoogleBarEnabled__accessor_storage; }
    set oneGoogleBarEnabled_(value) { this.#oneGoogleBarEnabled__accessor_storage = value; }
    #shortcutsEnabled__accessor_storage = loadTimeData.getBoolean('shortcutsEnabled');
    get shortcutsEnabled_() { return this.#shortcutsEnabled__accessor_storage; }
    set shortcutsEnabled_(value) { this.#shortcutsEnabled__accessor_storage = value; }
    #middleSlotPromoEnabled__accessor_storage = loadTimeData.getBoolean('middleSlotPromoEnabled');
    get middleSlotPromoEnabled_() { return this.#middleSlotPromoEnabled__accessor_storage; }
    set middleSlotPromoEnabled_(value) { this.#middleSlotPromoEnabled__accessor_storage = value; }
    #modulesEnabled__accessor_storage = loadTimeData.getBoolean('modulesEnabled');
    get modulesEnabled_() { return this.#modulesEnabled__accessor_storage; }
    set modulesEnabled_(value) { this.#modulesEnabled__accessor_storage = value; }
    #browserPromoType__accessor_storage = loadTimeData.getString('browserPromoType');
    get browserPromoType_() { return this.#browserPromoType__accessor_storage; }
    set browserPromoType_(value) { this.#browserPromoType__accessor_storage = value; }
    #browserPromoLimit__accessor_storage = loadTimeData.getInteger('browserPromoLimit');
    get browserPromoLimit_() { return this.#browserPromoLimit__accessor_storage; }
    set browserPromoLimit_(value) { this.#browserPromoLimit__accessor_storage = value; }
    #browserPromoCompletedLimit__accessor_storage = loadTimeData.getInteger('browserPromoCompletedLimit');
    get browserPromoCompletedLimit_() { return this.#browserPromoCompletedLimit__accessor_storage; }
    set browserPromoCompletedLimit_(value) { this.#browserPromoCompletedLimit__accessor_storage = value; }
    #showBrowserPromo__accessor_storage = false;
    get showBrowserPromo_() { return this.#showBrowserPromo__accessor_storage; }
    set showBrowserPromo_(value) { this.#showBrowserPromo__accessor_storage = value; }
    #middleSlotPromoLoaded__accessor_storage = false;
    get middleSlotPromoLoaded_() { return this.#middleSlotPromoLoaded__accessor_storage; }
    set middleSlotPromoLoaded_(value) { this.#middleSlotPromoLoaded__accessor_storage = value; }
    #modulesLoadedStatus__accessor_storage = ModuleLoadStatus.MODULE_LOAD_IN_PROGRESS;
    get modulesLoadedStatus_() { return this.#modulesLoadedStatus__accessor_storage; }
    set modulesLoadedStatus_(value) { this.#modulesLoadedStatus__accessor_storage = value; }
    #modulesShownToUser_accessor_storage = false;
    get modulesShownToUser() { return this.#modulesShownToUser_accessor_storage; }
    set modulesShownToUser(value) { this.#modulesShownToUser_accessor_storage = value; }
    #microsoftModuleEnabled__accessor_storage = loadTimeData.getBoolean('microsoftModuleEnabled');
    get microsoftModuleEnabled_() { return this.#microsoftModuleEnabled__accessor_storage; }
    set microsoftModuleEnabled_(value) { this.#microsoftModuleEnabled__accessor_storage = value; }
    #microsoftAuthIframePath__accessor_storage = MSAL_IFRAME_ORIGIN;
    get microsoftAuthIframePath_() { return this.#microsoftAuthIframePath__accessor_storage; }
    set microsoftAuthIframePath_(value) { this.#microsoftAuthIframePath__accessor_storage = value; }
    #promoAndModulesLoaded__accessor_storage = false;
    get promoAndModulesLoaded_() { return this.#promoAndModulesLoaded__accessor_storage; }
    set promoAndModulesLoaded_(value) { this.#promoAndModulesLoaded__accessor_storage = value; }
    #lazyRender__accessor_storage = false;
    get lazyRender_() { return this.#lazyRender__accessor_storage; }
    set lazyRender_(value) { this.#lazyRender__accessor_storage = value; }
    #scrolledToTop__accessor_storage = document.documentElement.scrollTop <= 0;
    get scrolledToTop_() { return this.#scrolledToTop__accessor_storage; }
    set scrolledToTop_(value) { this.#scrolledToTop__accessor_storage = value; }
    #wallpaperSearchButtonAnimationEnabled__accessor_storage = loadTimeData.getBoolean('wallpaperSearchButtonAnimationEnabled');
    get wallpaperSearchButtonAnimationEnabled_() { return this.#wallpaperSearchButtonAnimationEnabled__accessor_storage; }
    set wallpaperSearchButtonAnimationEnabled_(value) { this.#wallpaperSearchButtonAnimationEnabled__accessor_storage = value; }
    #wallpaperSearchButtonEnabled__accessor_storage = loadTimeData.getBoolean('wallpaperSearchButtonEnabled');
    get wallpaperSearchButtonEnabled_() { return this.#wallpaperSearchButtonEnabled__accessor_storage; }
    set wallpaperSearchButtonEnabled_(value) { this.#wallpaperSearchButtonEnabled__accessor_storage = value; }
    #showWallpaperSearchButton__accessor_storage = false;
    get showWallpaperSearchButton_() { return this.#showWallpaperSearchButton__accessor_storage; }
    set showWallpaperSearchButton_(value) { this.#showWallpaperSearchButton__accessor_storage = value; }
    #composeButtonEnabled_accessor_storage = loadTimeData.getBoolean('searchboxShowComposeEntrypoint');
    get composeButtonEnabled() { return this.#composeButtonEnabled_accessor_storage; }
    set composeButtonEnabled(value) { this.#composeButtonEnabled_accessor_storage = value; }
    #composeboxCloseByClickOutside__accessor_storage = loadTimeData.getBoolean('composeboxCloseByClickOutside');
    get composeboxCloseByClickOutside_() { return this.#composeboxCloseByClickOutside__accessor_storage; }
    set composeboxCloseByClickOutside_(value) { this.#composeboxCloseByClickOutside__accessor_storage = value; }
    #composeboxEnabled_accessor_storage = loadTimeData.getBoolean('searchboxShowComposebox');
    get composeboxEnabled() { return this.#composeboxEnabled_accessor_storage; }
    set composeboxEnabled(value) { this.#composeboxEnabled_accessor_storage = value; }
    #isFooterVisible__accessor_storage = false;
    get isFooterVisible_() { return this.#isFooterVisible__accessor_storage; }
    set isFooterVisible_(value) { this.#isFooterVisible__accessor_storage = value; }
    #ntpRealboxNextEnabled__accessor_storage = loadTimeData.getBoolean('ntpRealboxNextEnabled');
    get ntpRealboxNextEnabled_() { return this.#ntpRealboxNextEnabled__accessor_storage; }
    set ntpRealboxNextEnabled_(value) { this.#ntpRealboxNextEnabled__accessor_storage = value; }
    #realboxLayoutMode__accessor_storage = loadTimeData.getString('realboxLayoutMode');
    get realboxLayoutMode_() { return this.#realboxLayoutMode__accessor_storage; }
    set realboxLayoutMode_(value) { this.#realboxLayoutMode__accessor_storage = value; }
    #searchboxCyclingPlaceholders__accessor_storage = loadTimeData.getBoolean('searchboxCyclingPlaceholders');
    get searchboxCyclingPlaceholders_() { return this.#searchboxCyclingPlaceholders__accessor_storage; }
    set searchboxCyclingPlaceholders_(value) { this.#searchboxCyclingPlaceholders__accessor_storage = value; }
    #ntpNextFeaturesEnabled__accessor_storage = loadTimeData.getBoolean('ntpNextFeaturesEnabled');
    get ntpNextFeaturesEnabled_() { return this.#ntpNextFeaturesEnabled__accessor_storage; }
    set ntpNextFeaturesEnabled_(value) { this.#ntpNextFeaturesEnabled__accessor_storage = value; }
    #dropdownIsVisible__accessor_storage = false;
    get dropdownIsVisible_() { return this.#dropdownIsVisible__accessor_storage; }
    set dropdownIsVisible_(value) { this.#dropdownIsVisible__accessor_storage = value; }
    #searchboxInputFocused__accessor_storage = false;
    get searchboxInputFocused_() { return this.#searchboxInputFocused__accessor_storage; }
    set searchboxInputFocused_(value) { this.#searchboxInputFocused__accessor_storage = value; }
    #composeboxInputFocused__accessor_storage = false;
    get composeboxInputFocused_() { return this.#composeboxInputFocused__accessor_storage; }
    set composeboxInputFocused_(value) { this.#composeboxInputFocused__accessor_storage = value; }
    #showScrim__accessor_storage = false;
    get showScrim_() { return this.#showScrim__accessor_storage; }
    set showScrim_(value) { this.#showScrim__accessor_storage = value; }
    callbackRouter_;
    pageHandler_;
    customizeButtonsCallbackRouter_;
    customizeButtonsHandler_;
    backgroundManager_;
    connectMicrosoftAuthToParentDocumentListenerId_ = null;
    setThemeListenerId_ = null;
    setCustomizeChromeSidePanelVisibilityListener_ = null;
    setWallpaperSearchButtonVisibilityListener_ = null;
    footerVisibilityUpdatedListener_ = null;
    eventTracker_ = new EventTracker();
    shouldPrintPerformance_ = false;
    backgroundImageLoadStartEpoch_ = 0;
    backgroundImageLoadStart_ = 0;
    showWebstoreToastListenerId_ = null;
    pendingComposeboxContextFiles_ = [];
    pendingComposeboxText_ = '';
    pendingComposeboxMode_ = ComposeboxMode.DEFAULT;
    constructor() {
        performance.mark('app-creation-start');
        super();
        this.callbackRouter_ = NewTabPageProxy.getInstance().callbackRouter;
        this.pageHandler_ = NewTabPageProxy.getInstance().handler;
        this.customizeButtonsCallbackRouter_ =
            CustomizeButtonsProxy.getInstance().callbackRouter;
        this.customizeButtonsHandler_ = CustomizeButtonsProxy.getInstance().handler;
        this.backgroundManager_ = BackgroundManager.getInstance();
        this.shouldPrintPerformance_ =
            new URLSearchParams(location.search).has('print_perf');
        this.oneGoogleBarIframePath_ = (() => {
            const params = new URLSearchParams();
            params.set('paramsencoded', btoa(window.location.search.replace(/^[?]/, '&')));
            return `${OGB_IFRAME_ORIGIN}/one-google-bar?${params}`;
        })();
        this.showCustomize_ =
            WindowProxy.getInstance().url.searchParams.has(CUSTOMIZE_URL_PARAM);
        this.selectedCustomizeDialogPage_ =
            WindowProxy.getInstance().url.searchParams.get(CUSTOMIZE_URL_PARAM);
        this.realboxCanShowSecondarySide =
            realboxCanShowSecondarySideMediaQueryList.matches;
        /**
         * Initialized with the start of the performance timeline in case the
         * background image load is not triggered by JS.
         */
        this.backgroundImageLoadStartEpoch_ = performance.timeOrigin;
        recordLinearValue('NewTabPage.Height', 
        /*min=*/ 1, 
        /*max=*/ 1000, 
        /*buckets=*/ 200, 
        /*value=*/ Math.floor(window.innerHeight));
        recordLinearValue('NewTabPage.Width', 
        /*min=*/ 1, 
        /*max=*/ 1920, 
        /*buckets=*/ 384, 
        /*value=*/ Math.floor(window.innerWidth));
        ColorChangeUpdater.forDocument().start();
    }
    connectedCallback() {
        super.connectedCallback();
        realboxCanShowSecondarySideMediaQueryList.addEventListener('change', this.onRealboxCanShowSecondarySideChanged_.bind(this));
        // Listen for chrome-untrusted://ntp-microsoft-auth iframe trying to
        // connect to the NTP.
        this.connectMicrosoftAuthToParentDocumentListenerId_ =
            this.callbackRouter_.connectToParentDocument.addListener((childDocumentRemote) => {
                ParentTrustedDocumentProxy.setInstance(childDocumentRemote);
            });
        this.setThemeListenerId_ =
            this.callbackRouter_.setTheme.addListener((theme) => {
                if (!this.theme_) {
                    this.onThemeLoaded_(theme);
                }
                performance.measure('theme-set');
                this.theme_ = theme;
            });
        this.setCustomizeChromeSidePanelVisibilityListener_ =
            this.customizeButtonsCallbackRouter_
                .setCustomizeChromeSidePanelVisibility.addListener((visible) => {
                this.showCustomize_ = visible;
                if (!visible) {
                    this.showWallpaperSearch_ = false;
                }
            });
        this.showWebstoreToastListenerId_ =
            this.callbackRouter_.showWebstoreToast.addListener(() => {
                if (this.showCustomize_) {
                    const toast = $$(this, '#webstoreToast');
                    if (toast) {
                        toast.hidden = false;
                        toast.show();
                    }
                }
            });
        this.setWallpaperSearchButtonVisibilityListener_ =
            this.callbackRouter_.setWallpaperSearchButtonVisibility.addListener((visible) => {
                // Hides the wallpaper search button if the browser indicates that
                // it should be hidden.
                // Note: We don't resurface the button later even if the browser
                // says we should, to avoid issues if Customize Chrome doesn't
                // have the wallpaper search element yet.
                if (!visible) {
                    this.wallpaperSearchButtonEnabled_ = visible;
                }
            });
        this.footerVisibilityUpdatedListener_ =
            this.callbackRouter_.footerVisibilityUpdated.addListener((visible) => {
                this.isFooterVisible_ = visible;
            });
        this.pageHandler_.updateFooterVisibility();
        // Open Customize Chrome if there are Customize Chrome URL params.
        if (this.showCustomize_) {
            this.setCustomizeChromeSidePanelVisible_(this.showCustomize_);
            recordCustomizeChromeOpen(NtpCustomizeChromeEntryPoint.URL);
        }
        this.eventTracker_.add(window, 'message', (event) => {
            const data = event.data;
            // Something in OneGoogleBar is sending a message that is received here.
            // Need to ignore it.
            if (typeof data !== 'object') {
                return;
            }
            if ('frameType' in data && data.frameType === 'one-google-bar') {
                this.handleOneGoogleBarMessage_(event);
            }
        });
        this.eventTracker_.add(window, 'keydown', this.onWindowKeydown_.bind(this));
        this.eventTracker_.add(window, 'click', this.onWindowClick_.bind(this), /*capture=*/ true);
        this.eventTracker_.add(document, 'scroll', () => {
            this.scrolledToTop_ = document.documentElement.scrollTop <= 0;
        });
        if (loadTimeData.getString('backgroundImageUrl')) {
            this.backgroundManager_.getBackgroundImageLoadTime().then(time => {
                const duration = time - this.backgroundImageLoadStartEpoch_;
                recordDuration('NewTabPage.Images.ShownTime.BackgroundImage', duration);
                if (this.shouldPrintPerformance_) {
                    this.printPerformanceDatum_('background-image-load', this.backgroundImageLoadStart_, duration);
                    this.printPerformanceDatum_('background-image-loaded', this.backgroundImageLoadStart_ + duration);
                }
            }, () => {
                // Ignore. Failed to capture background image load time.
            });
        }
        FocusOutlineManager.forDocument(document);
        if (this.composeButtonEnabled) {
            recordBoolean('NewTabPage.ComposeEntrypoint.Shown', true);
            this.pageHandler_.incrementComposeButtonShownCount();
        }
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        realboxCanShowSecondarySideMediaQueryList.removeEventListener('change', this.onRealboxCanShowSecondarySideChanged_.bind(this));
        this.callbackRouter_.removeListener(this.connectMicrosoftAuthToParentDocumentListenerId_);
        this.callbackRouter_.removeListener(this.setThemeListenerId_);
        this.callbackRouter_.removeListener(this.showWebstoreToastListenerId_);
        this.callbackRouter_.removeListener(this.setWallpaperSearchButtonVisibilityListener_);
        this.customizeButtonsCallbackRouter_.removeListener(this.setCustomizeChromeSidePanelVisibilityListener_);
        this.callbackRouter_.removeListener(this.footerVisibilityUpdatedListener_);
        this.eventTracker_.removeAll();
    }
    firstUpdated() {
        this.pageHandler_.onAppRendered(WindowProxy.getInstance().now());
        // Let the browser breathe and then render remaining elements.
        WindowProxy.getInstance().waitForLazyRender().then(() => {
            ensureLazyLoaded();
            this.lazyRender_ = true;
        });
        this.printPerformance_();
        performance.measure('app-creation', 'app-creation-start');
        if (!this.modulesEnabled_) {
            this.recordBrowserPromoMetrics_();
        }
        this.pageHandler_.maybeTriggerAutomaticCustomizeChromePromo();
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('theme_')) {
            this.showBackgroundImage_ = this.computeShowBackgroundImage_();
            this.backgroundImageAttribution1_ =
                this.computeBackgroundImageAttribution1_();
            this.backgroundImageAttribution2_ =
                this.computeBackgroundImageAttribution2_();
            this.backgroundImageAttributionUrl_ =
                this.computeBackgroundImageAttributionUrl_();
            this.colorSourceIsBaseline = this.computeColorSourceIsBaseline();
            this.logoColor_ = this.computeLogoColor_();
            this.singleColoredLogo_ = this.computeSingleColoredLogo_();
        }
        // theme_, showLensUploadDialog_
        this.realboxShown_ = this.computeRealboxShown_();
        // middleSlotPromoLoaded_, modulesLoadedStatus_
        this.promoAndModulesLoaded_ = this.computePromoAndModulesLoaded_();
        // wallpaperSearchButtonEnabled_, showBackgroundImage_, backgroundColor_
        this.showWallpaperSearchButton_ = this.computeShowWallpaperSearchButton_();
        // showWallpaperSearchButton_, showBackgroundImage_
        this.showCustomizeChromeText_ = this.computeShowCustomizeChromeText_();
        // modulesEnabled_, modulesShownToUser, modulesLoadedStatus_
        this.showBrowserPromo_ = this.computeShowBrowserPromo_();
        if ((changedPrivateProperties.has('modulesLoadedStatus_') &&
            this.modulesLoadedStatus_ !==
                ModuleLoadStatus.MODULE_LOAD_IN_PROGRESS)) {
            this.recordBrowserPromoMetrics_();
        }
        if (this.ntpRealboxNextEnabled_ && [
            'showComposebox_',
            'searchboxInputFocused_',
            'composeboxInputFocused_',
        ].some((prop) => changedPrivateProperties.has(prop))) {
            /**
             * The current requirement is that the scrim should be shown when the
             * focus is placed on one of the input boxes and should be removed when
             * the focus moves outside.
             *
             * The additional OR operation with showComposebox_ is because the logic
             * does not close Composebox when a click outside is made while Composebox
             * is opened. What seems to be happening when showComposebox_ is used/not
             * used are as follows:
             * - Without it:
             *   1. A click outside is made.
             *   2. The focusout event first occurs.
             *   3. composeboxInputFocused_ is set to false.
             *   4. The scrim is removed.
             *   5. The click event fires.
             *   6. Since there is no scrim, the onclick handle of the scrim is not
             *      called.
             * - With it:
             *   1-3. same as above
             *   4. The scrim is kept since showComposebox_ is still true.
             *   5. The onclick handler of the scrim runs and sets showComposebox_ to
             *      false, and everything works as desired.
             */
            this.showScrim_ = this.showComposebox_ || this.searchboxInputFocused_ ||
                this.composeboxInputFocused_;
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('lazyRender_') && this.lazyRender_) {
            this.onLazyRendered_();
        }
        if (changedPrivateProperties.has('theme_')) {
            this.onThemeChange_();
        }
        if (changedPrivateProperties.has('isFooterVisible_') && this.lazyRender_) {
            this.maybeRegisterCustomizeButtonHelpBubble_();
        }
        if (changedPrivateProperties.has('logoColor_')) {
            this.style.setProperty('--ntp-logo-color', this.rgbaOrInherit_(this.logoColor_));
        }
        if (changedPrivateProperties.has('showBackgroundImage_')) {
            this.onShowBackgroundImageChange_();
        }
        if (changedPrivateProperties.has('promoAndModulesLoaded_')) {
            this.onPromoAndModulesLoadedChange_();
        }
        if (changedPrivateProperties.has('oneGoogleBarLoaded_') ||
            changedPrivateProperties.has('theme_') ||
            changedPrivateProperties.has('showComposebox_')) {
            this.updateOneGoogleBarAppearance_();
        }
    }
    // Called to update the OGB of relevant NTP state changes.
    updateOneGoogleBarAppearance_() {
        if (this.oneGoogleBarLoaded_) {
            let isNtpDarkTheme;
            if (this.showComposebox_) {
                isNtpDarkTheme = this.theme_ && this.theme_.isDark;
            }
            else {
                isNtpDarkTheme = this.theme_ &&
                    (!!this.theme_.backgroundImage || this.theme_.isDark);
            }
            $$(this, '#oneGoogleBar').postMessage({
                type: 'updateAppearance',
                // We should be using a light OGB for dark themes and vice versa.
                applyLightTheme: isNtpDarkTheme,
            });
        }
    }
    computeShowCustomizeChromeText_() {
        if (this.showWallpaperSearchButton_) {
            return false;
        }
        return !this.showBackgroundImage_;
    }
    computeBackgroundImageAttribution1_() {
        return this.theme_ && this.theme_.backgroundImageAttribution1 || '';
    }
    computeBackgroundImageAttribution2_() {
        return this.theme_ && this.theme_.backgroundImageAttribution2 || '';
    }
    computeBackgroundImageAttributionUrl_() {
        return this.theme_ && this.theme_.backgroundImageAttributionUrl ?
            this.theme_.backgroundImageAttributionUrl.url :
            '';
    }
    computeRealboxShown_() {
        // Do not show the realbox if the upload dialog is showing.
        return !!this.theme_ && !this.showLensUploadDialog_ &&
            !this.showComposebox_;
    }
    computePromoAndModulesLoaded_() {
        return (!loadTimeData.getBoolean('middleSlotPromoEnabled') ||
            this.middleSlotPromoLoaded_) &&
            (!loadTimeData.getBoolean('modulesEnabled') ||
                this.modulesLoadedStatus_ === ModuleLoadStatus.MODULE_LOAD_COMPLETE);
    }
    onRealboxCanShowSecondarySideChanged_(e) {
        this.realboxCanShowSecondarySide = e.matches;
    }
    onLazyRendered_() {
        // Integration tests use this attribute to determine when lazy load has
        // completed.
        document.documentElement.setAttribute('lazy-loaded', String(true));
        this.maybeRegisterCustomizeButtonHelpBubble_();
        if (this.showWallpaperSearchButton_) {
            this.customizeButtonsHandler_.incrementWallpaperSearchButtonShownCount();
        }
    }
    maybeRegisterCustomizeButtonHelpBubble_() {
        if (!this.isFooterVisible_) {
            this.registerHelpBubble(CUSTOMIZE_CHROME_BUTTON_ELEMENT_ID, ['ntp-customize-buttons', '#customizeButton'], { fixed: true });
            this.pageHandler_.maybeShowFeaturePromo(IphFeature.kCustomizeChrome);
        }
    }
    onComposeboxInitialized_(e) {
        e.detail.initializeComposeboxState(this.pendingComposeboxText_, this.pendingComposeboxContextFiles_, this.pendingComposeboxMode_);
        this.pendingComposeboxContextFiles_ = [];
        this.pendingComposeboxText_ = '';
        this.pendingComposeboxMode_ = ComposeboxMode.DEFAULT;
    }
    openComposebox_(e) {
        if (e.detail.searchboxText) {
            this.pendingComposeboxText_ = e.detail.searchboxText;
        }
        if (e.detail.contextFiles && e.detail.contextFiles.length > 0) {
            this.pendingComposeboxContextFiles_ = e.detail.contextFiles;
        }
        this.pendingComposeboxMode_ = e.detail.mode;
        this.toggleComposebox_();
    }
    toggleComposebox_() {
        this.showComposebox_ = !this.showComposebox_;
        if (!this.wasComposeboxOpened_) {
            recordLoadDuration('NewTabPage.Composebox.FromNTPLoadToSessionStart', WindowProxy.getInstance().now());
            this.wasComposeboxOpened_ = true;
        }
    }
    onComposeboxClickOutside_() {
        const composebox = this.shadowRoot.querySelector('#composebox');
        assert(composebox);
        const closeComposebox = new CustomEvent('closeComposebox', {
            detail: { composeboxText: composebox.getText() },
            bubbles: true,
            cancelable: true,
        });
        this.closeComposebox_(closeComposebox);
    }
    closeComposebox_(e) {
        const composeboxText = e.detail.composeboxText;
        if (composeboxText && composeboxText.trim()) {
            this.$.searchbox.setInputText(composeboxText);
        }
        const composebox = this.shadowRoot.querySelector('#composebox');
        assert(composebox);
        composebox.setText('');
        composebox.resetModes();
        if (this.ntpRealboxNextEnabled_) {
            composebox.closeDropdown();
        }
        this.toggleComposebox_();
        this.logoColor_ = this.computeLogoColor_();
        this.singleColoredLogo_ = this.computeSingleColoredLogo_();
        this.updateOneGoogleBarAppearance_();
    }
    onOpenVoiceSearch_() {
        this.showVoiceSearchOverlay_ = true;
        recordVoiceAction(VoiceAction.ACTIVATE_SEARCH_BOX);
    }
    onOpenLensSearch_() {
        this.showLensUploadDialog_ = true;
    }
    onCloseLensSearch_() {
        this.showLensUploadDialog_ = false;
    }
    onCustomizeClick_() {
        // Let side panel decide what page or section to show.
        this.selectedCustomizeDialogPage_ = null;
        this.setCustomizeChromeSidePanelVisible_(!this.showCustomize_);
        if (!this.showCustomize_) {
            this.customizeButtonsHandler_.incrementCustomizeChromeButtonOpenCount();
            recordCustomizeChromeOpen(NtpCustomizeChromeEntryPoint.CUSTOMIZE_BUTTON);
        }
    }
    computeShowWallpaperSearchButton_() {
        if (!this.wallpaperSearchButtonEnabled_) {
            return false;
        }
        switch (loadTimeData.getInteger('wallpaperSearchButtonHideCondition')) {
            case NtpWallpaperSearchButtonHideCondition.NONE:
                return true;
            case NtpWallpaperSearchButtonHideCondition.BACKGROUND_IMAGE_SET:
                return !this.showBackgroundImage_;
            case NtpWallpaperSearchButtonHideCondition.THEME_SET:
                return this.colorSourceIsBaseline && !this.showBackgroundImage_;
        }
        return false;
    }
    onWallpaperSearchClick_() {
        // Close the side panel if Wallpaper Search is open.
        if (this.showCustomize_ && this.showWallpaperSearch_) {
            this.selectedCustomizeDialogPage_ = null;
            this.setCustomizeChromeSidePanelVisible_(!this.showCustomize_);
            return;
        }
        // Open Wallpaper Search if the side panel is closed. Otherwise, navigate
        // the side panel to Wallpaper Search.
        this.selectedCustomizeDialogPage_ = CustomizeDialogPage.WALLPAPER_SEARCH;
        this.showWallpaperSearch_ = true;
        this.setCustomizeChromeSidePanelVisible_(this.showWallpaperSearch_);
        if (!this.showCustomize_) {
            this.customizeButtonsHandler_.incrementCustomizeChromeButtonOpenCount();
            recordCustomizeChromeOpen(NtpCustomizeChromeEntryPoint.WALLPAPER_SEARCH_BUTTON);
        }
    }
    onVoiceSearchOverlayClose_() {
        this.showVoiceSearchOverlay_ = false;
    }
    /**
     * Handles <CTRL> + <SHIFT> + <.> (also <CMD> + <SHIFT> + <.> on mac) to open
     * voice search.
     */
    onWindowKeydown_(e) {
        let ctrlKeyPressed = e.ctrlKey;
        // 
        if (ctrlKeyPressed && e.code === 'Period' && e.shiftKey) {
            this.showVoiceSearchOverlay_ = true;
            recordVoiceAction(VoiceAction.ACTIVATE_KEYBOARD);
        }
    }
    rgbaOrInherit_(skColor) {
        return skColor ? skColorToRgba(skColor) : 'inherit';
    }
    computeShowBackgroundImage_() {
        return !!this.theme_ && !!this.theme_.backgroundImage;
    }
    onShowBackgroundImageChange_() {
        this.backgroundManager_.setShowBackgroundImage(this.showBackgroundImage_);
    }
    onThemeChange_() {
        if (this.theme_) {
            this.backgroundManager_.setBackgroundColor(this.theme_.backgroundColor);
            this.style.setProperty('--color-new-tab-page-attribution-foreground', this.rgbaOrInherit_(this.theme_.textColor));
            this.style.setProperty('--color-new-tab-page-most-visited-foreground', this.rgbaOrInherit_(this.theme_.textColor));
        }
        this.updateBackgroundImagePath_();
    }
    onThemeLoaded_(theme) {
        recordSparseValueWithPersistentHash('NewTabPage.Collections.IdOnLoad', theme.backgroundImageCollectionId ?? '');
        if (!theme.backgroundImage) {
            recordEnumeration('NewTabPage.BackgroundImageSource', NtpBackgroundImageSource.kNoImage, NtpBackgroundImageSource.MAX_VALUE + 1);
        }
        else {
            recordEnumeration('NewTabPage.BackgroundImageSource', theme.backgroundImage.imageSource, NtpBackgroundImageSource.MAX_VALUE + 1);
        }
    }
    onPromoAndModulesLoadedChange_() {
        if (this.promoAndModulesLoaded_ &&
            loadTimeData.getBoolean('modulesEnabled')) {
            recordLoadDuration('NewTabPage.Modules.ShownTime', WindowProxy.getInstance().now());
        }
    }
    /**
     * Set the #backgroundImage |path| only when different and non-empty. Reset
     * the customize dialog background selection if the dialog is closed.
     *
     * The ntp-untrusted-iframe |path| is set directly. When using a data binding
     * instead, the quick updates to the |path| result in iframe loading an error
     * page.
     */
    updateBackgroundImagePath_() {
        const backgroundImage = this.theme_ && this.theme_.backgroundImage;
        if (!backgroundImage) {
            return;
        }
        this.backgroundManager_.setBackgroundImage(backgroundImage);
        if (this.wallpaperSearchButtonAnimationEnabled_ &&
            backgroundImage.imageSource ===
                NtpBackgroundImageSource.kWallpaperSearch ||
            backgroundImage.imageSource ===
                NtpBackgroundImageSource.kWallpaperSearchInspiration) {
            this.wallpaperSearchButtonAnimationEnabled_ = false;
        }
    }
    computeColorSourceIsBaseline() {
        return !!this.theme_ && this.theme_.isBaseline;
    }
    computeLogoColor_() {
        if (!this.theme_) {
            return null;
        }
        return this.theme_.logoColor ||
            (this.theme_.isDark ? hexColorToSkColor('#ffffff') : null);
    }
    computeSingleColoredLogo_() {
        return !!this.theme_ && (!!this.theme_.logoColor || this.theme_.isDark);
    }
    /**
     * Sends the command received from the given source and origin to the browser.
     * Relays the browser response to whether or not a promo containing the given
     * command can be shown back to the source promo frame. |commandSource| and
     * |commandOrigin| are used only to send the response back to the source promo
     * frame and should not be used for anything else.
     * @param  messageData Data received from the source promo frame.
     * @param commandSource Source promo frame.
     * @param commandOrigin Origin of the source promo frame.
     */
    canShowPromoWithBrowserCommand_(messageData, commandSource, commandOrigin) {
        // Make sure we don't send unsupported commands to the browser.
        /** @type {!Command} */
        const commandId = Object.values(Command).includes(messageData.commandId) ?
            messageData.commandId :
            Command.kUnknownCommand;
        BrowserCommandProxy.getInstance().handler.canExecuteCommand(commandId).then(({ canExecute }) => {
            const response = {
                messageType: messageData.messageType,
                [messageData.commandId]: canExecute,
            };
            commandSource.postMessage(response, commandOrigin);
        });
    }
    /**
     * Sends the command and the accompanying mouse click info received from the
     * promo of the given source and origin to the browser. Relays the execution
     * status response back to the source promo frame. |commandSource| and
     * |commandOrigin| are used only to send the execution status response back to
     * the source promo frame and should not be used for anything else.
     * @param commandData Command and mouse click info.
     * @param commandSource Source promo frame.
     * @param commandOrigin Origin of the source promo frame.
     */
    executePromoBrowserCommand_(commandData, commandSource, commandOrigin) {
        // Make sure we don't send unsupported commands to the browser.
        const commandId = Object.values(Command).includes(commandData.commandId) ?
            commandData.commandId :
            Command.kUnknownCommand;
        BrowserCommandProxy.getInstance()
            .handler.executeCommand(commandId, commandData.clickInfo)
            .then(({ commandExecuted }) => {
            commandSource.postMessage(commandExecuted, commandOrigin);
        });
    }
    /**
     * Handles messages from the OneGoogleBar iframe. The messages that are
     * handled include show bar on load and overlay updates.
     *
     * 'overlaysUpdated' message includes the updated array of overlay rects that
     * are shown.
     */
    handleOneGoogleBarMessage_(event) {
        const data = event.data;
        if (data.messageType === 'loaded') {
            const oneGoogleBar = $$(this, '#oneGoogleBar');
            oneGoogleBar.style.clipPath = 'url(#oneGoogleBarClipPath)';
            oneGoogleBar.style.zIndex = '1000';
            this.oneGoogleBarLoaded_ = true;
            this.pageHandler_.onOneGoogleBarRendered(WindowProxy.getInstance().now());
        }
        else if (data.messageType === 'overlaysUpdated') {
            this.$.oneGoogleBarClipPath.querySelectorAll('rect').forEach(el => {
                el.remove();
            });
            const overlayRects = data.data;
            overlayRects.forEach(({ x, y, width, height }) => {
                const rectElement = document.createElementNS('http://www.w3.org/2000/svg', 'rect');
                // Add 8px around every rect to ensure shadows are not cutoff.
                rectElement.setAttribute('x', `${x - 8}`);
                rectElement.setAttribute('y', `${y - 8}`);
                rectElement.setAttribute('width', `${width + 16}`);
                rectElement.setAttribute('height', `${height + 16}`);
                this.$.oneGoogleBarClipPath.appendChild(rectElement);
            });
        }
        else if (data.messageType === 'can-show-promo-with-browser-command') {
            this.canShowPromoWithBrowserCommand_(data, event.source, event.origin);
        }
        else if (data.messageType === 'execute-browser-command') {
            this.executePromoBrowserCommand_(data.data, event.source, event.origin);
        }
        else if (data.messageType === 'click') {
            recordClick(NtpElement.ONE_GOOGLE_BAR);
        }
    }
    onMiddleSlotPromoLoaded_() {
        this.middleSlotPromoLoaded_ = true;
    }
    onModulesLoaded_(e) {
        this.modulesLoadedStatus_ = e.detail ?
            ModuleLoadStatus.MODULE_LOAD_COMPLETE :
            ModuleLoadStatus.MODULE_LOAD_NOT_ATTEMPTED;
    }
    computeShowBrowserPromo_() {
        return !this.modulesEnabled_ ||
            (this.modulesLoadedStatus_ !==
                ModuleLoadStatus.MODULE_LOAD_IN_PROGRESS &&
                !this.modulesShownToUser);
    }
    recordBrowserPromoMetrics_() {
        if (!this.showBrowserPromo_) {
            recordShowBrowserPromosResult(ShowNtpPromosResult.kNotShownDueToPolicy);
            return;
        }
        switch (this.browserPromoType_) {
            case 'disabled':
                recordShowBrowserPromosResult(ShowNtpPromosResult.kNotShownDueToPolicy);
                break;
            case 'empty':
                recordShowBrowserPromosResult(ShowNtpPromosResult.kNotShownNoPromos);
                break;
            case 'simple':
            case 'setuplist':
                recordShowBrowserPromosResult(ShowNtpPromosResult.kShown);
                break;
            default:
                break;
        }
    }
    onCustomizeModule_() {
        this.showCustomize_ = true;
        this.selectedCustomizeDialogPage_ = CustomizeDialogPage.MODULES;
        recordCustomizeChromeOpen(NtpCustomizeChromeEntryPoint.MODULE);
        this.setCustomizeChromeSidePanelVisible_(this.showCustomize_);
    }
    setCustomizeChromeSidePanelVisible_(visible) {
        let section = CustomizeChromeSection.kUnspecified;
        switch (this.selectedCustomizeDialogPage_) {
            case CustomizeDialogPage.BACKGROUNDS:
            case CustomizeDialogPage.THEMES:
                section = CustomizeChromeSection.kAppearance;
                break;
            case CustomizeDialogPage.SHORTCUTS:
                section = CustomizeChromeSection.kShortcuts;
                break;
            case CustomizeDialogPage.MODULES:
                section = CustomizeChromeSection.kModules;
                break;
            case CustomizeDialogPage.WALLPAPER_SEARCH:
                section = CustomizeChromeSection.kWallpaperSearch;
                break;
        }
        this.customizeButtonsHandler_.setCustomizeChromeSidePanelVisible(visible, section, SidePanelOpenTrigger.kNewTabPage);
    }
    printPerformanceDatum_(name, time, auxTime = 0) {
        if (!this.shouldPrintPerformance_) {
            return;
        }
        console.info(!auxTime ? `${name}: ${time}` : `${name}: ${time} (${auxTime})`);
    }
    /**
     * Prints performance measurements to the console. Also, installs  performance
     * observer to continuously print performance measurements after.
     */
    printPerformance_() {
        if (!this.shouldPrintPerformance_) {
            return;
        }
        const entryTypes = ['paint', 'measure'];
        const log = (entry) => {
            this.printPerformanceDatum_(entry.name, entry.duration ? entry.duration : entry.startTime, entry.duration && entry.startTime ? entry.startTime : 0);
        };
        const observer = new PerformanceObserver(list => {
            list.getEntries().forEach((entry) => {
                log(entry);
            });
        });
        observer.observe({ entryTypes: entryTypes });
        performance.getEntries().forEach((entry) => {
            if (!entryTypes.includes(entry.entryType)) {
                return;
            }
            log(entry);
        });
    }
    onWebstoreToastButtonClick_() {
        window.location.assign(`https://chrome.google.com/webstore/category/collection/chrome_color_themes?hl=${window.navigator.language}`);
    }
    onWindowClick_(e) {
        if (e.composedPath() && e.composedPath()[0] === $$(this, '#content')) {
            recordClick(NtpElement.BACKGROUND);
            return;
        }
        for (const target of e.composedPath()) {
            switch (target) {
                case $$(this, 'ntp-logo'):
                    recordClick(NtpElement.LOGO);
                    return;
                case $$(this, 'cr-searchbox'):
                    recordClick(NtpElement.REALBOX);
                    return;
                case $$(this, 'cr-most-visited'):
                    recordClick(NtpElement.MOST_VISITED);
                    return;
                case $$(this, 'ntp-middle-slot-promo'):
                    recordClick(NtpElement.MIDDLE_SLOT_PROMO);
                    return;
                case $$(this, '#modules'):
                    recordClick(NtpElement.MODULE);
                    return;
            }
        }
        const customizeButtonsElement = this.shadowRoot.querySelector('ntp-customize-buttons');
        if (customizeButtonsElement) {
            for (const target of e.composedPath()) {
                switch (target) {
                    case $$(customizeButtonsElement, '#customizeButton'):
                        recordClick(NtpElement.CUSTOMIZE_BUTTON);
                        return;
                    case $$(customizeButtonsElement, '#wallpaperSearchButton'):
                        recordClick(NtpElement.WALLPAPER_SEARCH_BUTTON);
                        return;
                }
            }
        }
        recordClick(NtpElement.OTHER);
    }
    isThemeDark_() {
        return !!this.theme_ && this.theme_.isDark;
    }
    showThemeAttribution_() {
        return !!this.theme_?.backgroundImage?.attributionUrl;
    }
    onDropdownVisibleChanged_(e) {
        this.dropdownIsVisible_ = e.detail.value;
    }
    onInputFocusChanged_(e) {
        switch (e.type) {
            case 'searchbox-input-focus-changed':
                this.searchboxInputFocused_ = e.detail.value;
                break;
            case 'composebox-input-focus-changed':
                this.composeboxInputFocused_ = e.detail.value;
                break;
        }
    }
    onRealboxHadSecondarySideChanged_(e) {
        this.realboxHadSecondarySide = e.detail.value;
    }
    onModulesShownToUserChanged_(e) {
        this.modulesShownToUser = e.detail.value;
    }
}
customElements.define(AppElement.is, AppElement);
