// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import '/strings.m.js';
import { I18nMixinLit } from 'chrome://resources/cr_elements/i18n_mixin_lit.js';
import { WebUiListenerMixinLit } from 'chrome://resources/cr_elements/web_ui_listener_mixin_lit.js';
import { assert } from 'chrome://resources/js/assert.js';
import { CrLitElement } from 'chrome://resources/lit/v3_0/lit.rollup.js';
import { Coordinate2d } from '../data/coordinate2d.js';
import { CustomMarginsOrientation } from '../data/margins.js';
import { Size } from '../data/size.js';
import { InputMixin } from './input_mixin.js';
import { getCss } from './margin_control.css.js';
import { getHtml } from './margin_control.html.js';
/**
 * Radius of the margin control in pixels. Padding of control + 1 for border.
 */
const RADIUS_PX = 9;
const PrintPreviewMarginControlElementBase = I18nMixinLit(WebUiListenerMixinLit(InputMixin(CrLitElement)));
export class PrintPreviewMarginControlElement extends PrintPreviewMarginControlElementBase {
    static get is() {
        return 'print-preview-margin-control';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            disabled: {
                type: Boolean,
                reflect: true,
            },
            side: {
                type: String,
                reflect: true,
            },
            invalid: {
                type: Boolean,
                reflect: true,
            },
            invisible: {
                type: Boolean,
                reflect: true,
            },
            measurementSystem: { type: Object },
            focused_: {
                type: Boolean,
                reflect: true,
            },
            positionInPts_: { type: Number },
            scaleTransform: { type: Number },
            translateTransform: { type: Object },
            pageSize: { type: Object },
            clipSize: { type: Object },
        };
    }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #side_accessor_storage = CustomMarginsOrientation.TOP;
    get side() { return this.#side_accessor_storage; }
    set side(value) { this.#side_accessor_storage = value; }
    #invalid_accessor_storage = false;
    get invalid() { return this.#invalid_accessor_storage; }
    set invalid(value) { this.#invalid_accessor_storage = value; }
    #invisible_accessor_storage = false;
    get invisible() { return this.#invisible_accessor_storage; }
    set invisible(value) { this.#invisible_accessor_storage = value; }
    #measurementSystem_accessor_storage = null;
    get measurementSystem() { return this.#measurementSystem_accessor_storage; }
    set measurementSystem(value) { this.#measurementSystem_accessor_storage = value; }
    #scaleTransform_accessor_storage = 1;
    get scaleTransform() { return this.#scaleTransform_accessor_storage; }
    set scaleTransform(value) { this.#scaleTransform_accessor_storage = value; }
    #translateTransform_accessor_storage = new Coordinate2d(0, 0);
    get translateTransform() { return this.#translateTransform_accessor_storage; }
    set translateTransform(value) { this.#translateTransform_accessor_storage = value; }
    #pageSize_accessor_storage = new Size(612, 792);
    get pageSize() { return this.#pageSize_accessor_storage; }
    set pageSize(value) { this.#pageSize_accessor_storage = value; }
    #clipSize_accessor_storage = null;
    get clipSize() { return this.#clipSize_accessor_storage; }
    set clipSize(value) { this.#clipSize_accessor_storage = value; }
    #focused__accessor_storage = false;
    get focused_() { return this.#focused__accessor_storage; }
    set focused_(value) { this.#focused__accessor_storage = value; }
    #positionInPts__accessor_storage = 0;
    get positionInPts_() { return this.#positionInPts__accessor_storage; }
    set positionInPts_(value) { this.#positionInPts__accessor_storage = value; }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('disabled')) {
            if (this.disabled) {
                this.focused_ = false;
            }
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedProperties.has('clipSize') ||
            changedProperties.has('invisible')) {
            this.onClipSizeChange_();
        }
        if (changedPrivateProperties.has('positionInPts_') ||
            changedProperties.has('scaleTransform') ||
            changedProperties.has('translateTransform') ||
            changedProperties.has('pageSize') || changedProperties.has('side')) {
            this.updatePosition_();
        }
    }
    firstUpdated() {
        this.addEventListener('input-change', e => this.onInputChange_(e));
    }
    /** @return The input element for InputBehavior. */
    getInput() {
        return this.$.input;
    }
    /**
     * @param valueInPts New value of the margin control's textbox in pts.
     */
    setTextboxValue(valueInPts) {
        const textbox = this.$.input;
        const pts = textbox.value ? this.parseValueToPts_(textbox.value) : null;
        if (pts !== null && valueInPts === Math.round(pts)) {
            // If the textbox's value represents the same value in pts as the new one,
            // don't reset. This allows the "undo" command to work as expected, see
            // https://crbug.com/452844.
            return;
        }
        textbox.value = this.serializeValueFromPts_(valueInPts);
        this.resetString();
    }
    /** @return The current position of the margin control. */
    getPositionInPts() {
        return this.positionInPts_;
    }
    /** @param position The new position for the margin control. */
    setPositionInPts(position) {
        this.positionInPts_ = position;
    }
    /**
     * @return 'true' or 'false', indicating whether the input should be
     *     aria-hidden.
     */
    getAriaHidden_() {
        return this.invisible.toString();
    }
    /**
     * Converts a value in pixels to points.
     * @param pixels Pixel value to convert.
     * @return Given value expressed in points.
     */
    convertPixelsToPts(pixels) {
        let pts;
        const Orientation = CustomMarginsOrientation;
        if (this.side === Orientation.TOP) {
            pts = pixels - this.translateTransform.y + RADIUS_PX;
            pts /= this.scaleTransform;
        }
        else if (this.side === Orientation.RIGHT) {
            pts = pixels - this.translateTransform.x + RADIUS_PX;
            pts /= this.scaleTransform;
            pts = this.pageSize.width - pts;
        }
        else if (this.side === Orientation.BOTTOM) {
            pts = pixels - this.translateTransform.y + RADIUS_PX;
            pts /= this.scaleTransform;
            pts = this.pageSize.height - pts;
        }
        else {
            assert(this.side === Orientation.LEFT);
            pts = pixels - this.translateTransform.x + RADIUS_PX;
            pts /= this.scaleTransform;
        }
        return pts;
    }
    /**
     * @param event A pointerdown event triggered by this element.
     * @return Whether the margin should start being dragged.
     */
    shouldDrag(event) {
        return !this.disabled && event.button === 0 &&
            (event.composedPath()[0] === this.$.lineContainer ||
                event.composedPath()[0] === this.$.line);
    }
    /**
     * @param value Value to parse to points. E.g. '3.40' or '200'.
     * @return Value in points represented by the input value.
     */
    parseValueToPts_(value) {
        value = value.trim();
        if (value.length === 0) {
            return null;
        }
        assert(this.measurementSystem);
        const decimal = this.measurementSystem.decimalDelimiter;
        const thousands = this.measurementSystem.thousandsDelimiter;
        const whole = `(?:0|[1-9]\\d*|[1-9]\\d{0,2}(?:[${thousands}]\\d{3})*)`;
        const fractional = `(?:[${decimal}]\\d+)`;
        const wholeDecimal = `(?:${whole}[${decimal}])`;
        const validationRegex = new RegExp(`^-?(?:${whole}${fractional}?|${fractional}|${wholeDecimal})$`);
        if (validationRegex.test(value)) {
            // Removing thousands delimiters and replacing the decimal delimiter with
            // the dot symbol in order to use parseFloat() properly.
            value = value.replace(new RegExp(`\\${thousands}`, 'g'), '')
                .replace(decimal, '.');
            return this.measurementSystem.convertToPoints(parseFloat(value));
        }
        return null;
    }
    /**
     * @param value Value in points to serialize.
     * @return String representation of the value in the system's local units.
     */
    serializeValueFromPts_(value) {
        assert(this.measurementSystem);
        value = this.measurementSystem.convertFromPoints(value);
        value = this.measurementSystem.roundValue(value);
        // Convert the dot symbol to the decimal delimiter for the locale.
        return value.toString().replace('.', this.measurementSystem.decimalDelimiter);
    }
    fire_(eventName, detail) {
        this.dispatchEvent(new CustomEvent(eventName, { bubbles: true, composed: true, detail }));
    }
    /**
     * @param e Contains the new value of the input.
     */
    onInputChange_(e) {
        if (e.detail === '') {
            return;
        }
        const value = this.parseValueToPts_(e.detail);
        if (value === null) {
            this.invalid = true;
            return;
        }
        this.fire_('text-change', value);
    }
    onBlur_() {
        this.focused_ = false;
        this.resetAndUpdate();
        this.fire_('text-blur', this.invalid || !this.$.input.value);
    }
    onFocus_() {
        this.focused_ = true;
        this.fire_('text-focus');
    }
    updatePosition_() {
        if (!this.translateTransform || !this.scaleTransform ||
            !this.measurementSystem) {
            return;
        }
        const Orientation = CustomMarginsOrientation;
        let x = this.translateTransform.x;
        let y = this.translateTransform.y;
        let width = null;
        let height = null;
        if (this.side === Orientation.TOP) {
            y = this.scaleTransform * this.positionInPts_ +
                this.translateTransform.y - RADIUS_PX;
            width = this.scaleTransform * this.pageSize.width;
        }
        else if (this.side === Orientation.RIGHT) {
            x = this.scaleTransform * (this.pageSize.width - this.positionInPts_) +
                this.translateTransform.x - RADIUS_PX;
            height = this.scaleTransform * this.pageSize.height;
        }
        else if (this.side === Orientation.BOTTOM) {
            y = this.scaleTransform * (this.pageSize.height - this.positionInPts_) +
                this.translateTransform.y - RADIUS_PX;
            width = this.scaleTransform * this.pageSize.width;
        }
        else {
            x = this.scaleTransform * this.positionInPts_ +
                this.translateTransform.x - RADIUS_PX;
            height = this.scaleTransform * this.pageSize.height;
        }
        window.requestAnimationFrame(() => {
            this.style.left = Math.round(x) + 'px';
            this.style.top = Math.round(y) + 'px';
            if (width !== null) {
                this.style.width = Math.round(width) + 'px';
            }
            if (height !== null) {
                this.style.height = Math.round(height) + 'px';
            }
        });
        this.onClipSizeChange_();
    }
    onClipSizeChange_() {
        if (!this.clipSize) {
            return;
        }
        window.requestAnimationFrame(() => {
            const offsetLeft = this.offsetLeft;
            const offsetTop = this.offsetTop;
            this.style.clip = 'rect(' + (-offsetTop) + 'px, ' +
                (this.clipSize.width - offsetLeft) + 'px, ' +
                (this.clipSize.height - offsetTop) + 'px, ' + (-offsetLeft) + 'px)';
        });
    }
}
customElements.define(PrintPreviewMarginControlElement.is, PrintPreviewMarginControlElement);
