// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import '//resources/cr_elements/cr_collapse/cr_collapse.js';
import '//resources/cr_elements/cr_icon/cr_icon.js';
import '//resources/cr_elements/cr_expand_button/cr_expand_button.js';
import './site_favicon.js';
import { CrLitElement, html } from '//resources/lit/v3_0/lit.rollup.js';
import { getCss } from './list_item.css.js';
import { getHtml } from './list_item.html.js';
import { SiteType } from './related_website_sets.mojom-webui.js';
export class RelatedWebsiteSetsListItemElement extends CrLitElement {
    static get is() {
        return 'related-website-sets-list-item';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            expanded: { type: Boolean },
            primarySite: { type: String },
            memberSites: { type: Array },
            managedByEnterprise: { type: Boolean },
            query: { type: String },
        };
    }
    #expanded_accessor_storage = false;
    get expanded() { return this.#expanded_accessor_storage; }
    set expanded(value) { this.#expanded_accessor_storage = value; }
    #primarySite_accessor_storage = '';
    get primarySite() { return this.#primarySite_accessor_storage; }
    set primarySite(value) { this.#primarySite_accessor_storage = value; }
    #memberSites_accessor_storage = [];
    get memberSites() { return this.#memberSites_accessor_storage; }
    set memberSites(value) { this.#memberSites_accessor_storage = value; }
    #managedByEnterprise_accessor_storage = false;
    get managedByEnterprise() { return this.#managedByEnterprise_accessor_storage; }
    set managedByEnterprise(value) { this.#managedByEnterprise_accessor_storage = value; }
    #query_accessor_storage = '';
    get query() { return this.#query_accessor_storage; }
    set query(value) { this.#query_accessor_storage = value; }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('expanded')) {
            this.dispatchEvent(new CustomEvent('expanded-toggled', { detail: { id: this.primarySite, expanded: this.expanded } }));
        }
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('query')) {
            if (this.query) {
                const memberSitesWithMatch = this.memberSites.filter(member => member.site.toLowerCase().includes(this.query.toLowerCase()));
                this.expanded = memberSitesWithMatch.length > 0;
            }
            else {
                this.expanded = false;
            }
        }
    }
    onExpandedChanged_(e) {
        this.expanded = e.detail.value;
    }
    getSiteType_(type) {
        switch (type) {
            case SiteType.kPrimary:
                return 'Primary';
            case SiteType.kAssociated:
                return 'Associated';
            case SiteType.kService:
                return 'Service';
            default:
                throw new Error(`Unexpected site type ${type}`);
        }
    }
    isEnterpriseIconHidden_() {
        return !this.managedByEnterprise;
    }
    boldQuery_(url) {
        const domain = url.includes('://') ? url.split('://')[1] : url;
        if (!this.query) {
            return domain;
        }
        const queryLower = this.query.toLowerCase();
        const parts = domain.split(new RegExp(`(${this.query})`, 'gi'));
        return parts.map(part => part.toLowerCase() === queryLower ? html `<b>${part}</b>` : part);
    }
    getIconImageUrl_(site) {
        return `${site}/favicon.ico`;
    }
}
customElements.define(RelatedWebsiteSetsListItemElement.is, RelatedWebsiteSetsListItemElement);
