// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview
 * 'settings-a11y-page' is the small section of advanced settings with
 * a link to the web store accessibility page on most platforms, and
 * a subpage with lots of other settings on Chrome OS.
 */
import 'chrome://resources/cr_elements/cr_link_row/cr_link_row.js';
import '../controls/settings_toggle_button.js';
import '../settings_page/settings_section.js';
import '../settings_shared.css.js';
// clang-format off
// 
import './ax_annotations_section.js';
// 
// 
// clang-format on
import { PrefsMixin } from '/shared/settings/prefs/prefs_mixin.js';
import { WebUiListenerMixin } from 'chrome://resources/cr_elements/web_ui_listener_mixin.js';
import { assert } from 'chrome://resources/js/assert.js';
import { PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { BaseMixin } from '../base_mixin.js';
import { loadTimeData } from '../i18n_setup.js';
import { routes } from '../route.js';
import { Router } from '../router.js';
import { SettingsViewMixin } from '../settings_page/settings_view_mixin.js';
import { AccessibilityBrowserProxyImpl } from './a11y_browser_proxy.js';
import { getTemplate } from './a11y_page.html.js';
/**
 * Must be kept in sync with the C++ enum of the same name in
 * chrome/browser/ui/toasts/toast_metrics.h.
 */
export var ToastAlertLevel;
(function (ToastAlertLevel) {
    ToastAlertLevel[ToastAlertLevel["ALL"] = 0] = "ALL";
    ToastAlertLevel[ToastAlertLevel["ACTIONABLE"] = 1] = "ACTIONABLE";
    // Must be last.
    ToastAlertLevel[ToastAlertLevel["COUNT"] = 1] = "COUNT";
})(ToastAlertLevel || (ToastAlertLevel = {}));
const SettingsA11yPageElementBase = SettingsViewMixin(PrefsMixin(WebUiListenerMixin(BaseMixin(PolymerElement))));
export class SettingsA11yPageElement extends SettingsA11yPageElementBase {
    static get is() {
        return 'settings-a11y-page';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            // 
            enableLiveCaption_: {
                type: Boolean,
                value: function () {
                    return loadTimeData.getBoolean('enableLiveCaption');
                },
            },
            // 
            /**
             * Indicate whether a screen reader is enabled. Also, determine whether
             * to show accessibility labels settings.
             */
            hasScreenReader_: {
                type: Boolean,
                value: false,
            },
            /**
             * Whether to show the AxTreeFixing subpage.
             */
            showAxTreeFixingSection_: {
                type: Boolean,
                value: () => {
                    return loadTimeData.getBoolean('axTreeFixingEnabled');
                },
            },
            // 
            /**
             * Whether to show the AxAnnotations subpage.
             */
            showAxAnnotationsSection_: {
                type: Boolean,
                computed: 'computeShowAxAnnotationsSection_(hasScreenReader_)',
            },
            // 
            // 
            /** Valid toast alert level option. */
            toastAlertLevelEnum_: {
                type: Object,
                value: ToastAlertLevel,
            },
            numericUncheckedToastAlertValues_: {
                type: Array,
                value: () => [ToastAlertLevel.ACTIONABLE],
            },
            // 
        };
    }
    browserProxy_ = AccessibilityBrowserProxyImpl.getInstance();
    // 
    connectedCallback() {
        super.connectedCallback();
        const updateScreenReaderState = (hasScreenReader) => {
            this.hasScreenReader_ = hasScreenReader;
        };
        this.browserProxy_.getScreenReaderState().then(updateScreenReaderState);
        this.addWebUiListener('screen-reader-state-changed', updateScreenReaderState);
    }
    onA11yCaretBrowsingChange_(event) {
        if (event.target.checked) {
            chrome.metricsPrivate.recordUserAction('Accessibility.CaretBrowsing.EnableWithSettings');
        }
        else {
            chrome.metricsPrivate.recordUserAction('Accessibility.CaretBrowsing.DisableWithSettings');
        }
    }
    onA11yImageLabelsChange_(event) {
        const a11yImageLabelsOn = event.target.checked;
        if (a11yImageLabelsOn) {
            chrome.send('confirmA11yImageLabels');
        }
    }
    // 
    /**
     * Return whether to show the AxAnnotations subpage based on:
     *    1. If any annotation's feature flag is enabled.
     *    2. Whether a screen reader is enabled.
     * Note: on ChromeOS, the AxAnnotations subpage is shown on a different
     * settings page; i.e. Settings > Accessibility > Text-to-Speech.
     */
    computeShowAxAnnotationsSection_() {
        const anyAxAnnotationsFeatureEnabled = loadTimeData.getBoolean('mainNodeAnnotationsEnabled');
        return anyAxAnnotationsFeatureEnabled && this.hasScreenReader_;
    }
    onCaptionsClick_() {
        // 
        // 
        Router.getInstance().navigateTo(routes.CAPTIONS);
        // 
    }
    // 
    // 
    onFocusHighlightChange_(event) {
        chrome.metricsPrivate.recordBoolean('Accessibility.FocusHighlight.ToggleEnabled', event.target.checked);
    }
    // 
    // 
    /** private */
    onMoreFeaturesLinkClick_() {
        window.open('https://chrome.google.com/webstore/category/collection/3p_accessibility_extensions');
    }
    // 
    onOverscrollHistoryNavigationChange_(event) {
        const enabled = event.target.checked;
        this.browserProxy_.recordOverscrollHistoryNavigationChanged(enabled);
    }
    // 
    // 
    // 
    onToastAlertLevelChange_() {
        chrome.metricsPrivate.recordEnumerationValue('Toast.FrequencyPrefChanged', this.getPref('settings.toast.alert_level').value, ToastAlertLevel.COUNT);
    }
    // 
    // 
    // SettingsViewMixin implementation.
    getFocusConfig() {
        const map = new Map();
        if (routes.CAPTIONS) {
            map.set(routes.CAPTIONS.path, '#captions');
        }
        return map;
    }
    // 
    // SettingsViewMixin implementation.
    getAssociatedControlFor(childViewId) {
        assert(childViewId === 'captions');
        const control = this.shadowRoot.querySelector('#captions');
        assert(control);
        return control;
    }
}
customElements.define(SettingsA11yPageElement.is, SettingsA11yPageElement);
