// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'settings-translate-page' is the settings page
 * translate settings.
 */
import 'chrome://resources/cr_elements/cr_button/cr_button.js';
import 'chrome://resources/cr_elements/cr_collapse/cr_collapse.js';
import 'chrome://resources/cr_elements/cr_icon_button/cr_icon_button.js';
import 'chrome://resources/cr_elements/icons.html.js';
import 'chrome://resources/cr_elements/cr_shared_style.css.js';
import 'chrome://resources/cr_elements/md_select.css.js';
import './add_languages_dialog.js';
import './languages.js';
import '../controls/settings_toggle_button.js';
import '../icons.html.js';
import '../settings_shared.css.js';
import { PrefsMixin } from '/shared/settings/prefs/prefs_mixin.js';
import { I18nMixin } from 'chrome://resources/cr_elements/i18n_mixin.js';
import { assert } from 'chrome://resources/js/assert.js';
import { focusWithoutInk } from 'chrome://resources/js/focus_without_ink.js';
import { PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { getLanguageHelperInstance } from './languages.js';
import { LanguageSettingsActionType, LanguageSettingsMetricsProxyImpl } from './languages_settings_metrics_proxy.js';
import { convertLanguageCodeForChrome, getFullName, isTranslateBaseLanguage } from './languages_util.js';
import { getTemplate } from './translate_page.html.js';
const SettingsTranslatePageElementBase = PrefsMixin(I18nMixin(PolymerElement));
export class SettingsTranslatePageElement extends SettingsTranslatePageElementBase {
    static get is() {
        return 'settings-translate-page';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            /**
             * Read-only reference to the languages model provided by the
             * 'settings-languages' instance.
             */
            languages: Object,
            showAddAlwaysTranslateDialog_: Boolean,
            showAddNeverTranslateDialog_: Boolean,
            addLanguagesDialogLanguages_: Array,
        };
    }
    languageHelper_;
    languageSettingsMetricsProxy_ = LanguageSettingsMetricsProxyImpl.getInstance();
    connectedCallback() {
        super.connectedCallback();
        this.languageHelper_ = getLanguageHelperInstance();
    }
    onTargetLanguageChange_() {
        this.languageHelper_.setTranslateTargetLanguage(this.shadowRoot.querySelector('#targetLanguage').value);
        this.languageSettingsMetricsProxy_.recordSettingsMetric(LanguageSettingsActionType.CHANGE_TRANSLATE_TARGET);
    }
    /**
     * Helper function to get the text to display in the target language drop down
     * list. Returns the display name in the current UI language and the native
     * name of the language.
     */
    getTargetLanguageDisplayOption_(item) {
        return getFullName(item);
    }
    /**
     * Checks if a Chrome language code is equal to the translate language code.
     * Used in the translate language selector. If the item matches the translate
     * target language, it will set that item as selected.
     */
    translateLanguageEqual_(chromeItemCode, translateTarget) {
        return chromeItemCode === convertLanguageCodeForChrome(translateTarget);
    }
    /**
     * A function used for sorting languages alphabetically by display name.
     */
    alphabeticalSort_(first, second) {
        return first.displayName.localeCompare(second.displayName);
    }
    /**
     * A filter function to return true if language is not undefined and has a
     * displayName.
     */
    hasDisplayName_(language) {
        return !!language && !!language.displayName;
    }
    /**
     * Stamps and opens the Add Languages dialog, registering a listener to
     * disable the dialog's dom-if again on close.
     */
    onAddAlwaysTranslateLanguagesClick_(e) {
        e.preventDefault();
        const translatableLanguages = this.getTranslatableLanguages_();
        this.addLanguagesDialogLanguages_ = translatableLanguages.filter(language => !this.languages.alwaysTranslate.includes(language));
        this.showAddAlwaysTranslateDialog_ = true;
    }
    onAlwaysTranslateDialogClose_() {
        this.showAddAlwaysTranslateDialog_ = false;
        this.addLanguagesDialogLanguages_ = null;
        const toFocus = this.shadowRoot.querySelector('#addAlwaysTranslate');
        assert(toFocus);
        focusWithoutInk(toFocus);
    }
    /**
     * Helper function fired by the add dialog's on-languages-added event. Adds
     * selected languages to the always-translate languages list.
     */
    onAlwaysTranslateLanguagesAdded_(e) {
        const languagesToAdd = e.detail;
        languagesToAdd.forEach(languageCode => {
            this.languageHelper_.setLanguageAlwaysTranslateState(languageCode, true);
            this.languageSettingsMetricsProxy_.recordSettingsMetric(LanguageSettingsActionType.ADD_TO_ALWAYS_TRANSLATE);
        });
    }
    /**
     * Removes a language from the always translate languages list.
     */
    onRemoveAlwaysTranslateLanguageClick_(e) {
        const languageCode = e.model.item.code;
        this.languageHelper_.setLanguageAlwaysTranslateState(languageCode, false);
        this.languageSettingsMetricsProxy_.recordSettingsMetric(LanguageSettingsActionType.REMOVE_FROM_ALWAYS_TRANSLATE);
    }
    /**
     * Stamps and opens the Add Languages dialog, registering a listener to
     * disable the dialog's dom-if again on close.
     */
    onAddNeverTranslateLanguagesClick_(e) {
        e.preventDefault();
        const translatableLanguages = this.getTranslatableLanguages_();
        this.addLanguagesDialogLanguages_ = translatableLanguages.filter(language => !this.languages.neverTranslate.includes(language));
        this.showAddNeverTranslateDialog_ = true;
    }
    onNeverTranslateDialogClose_() {
        this.showAddNeverTranslateDialog_ = false;
        this.addLanguagesDialogLanguages_ = null;
        const toFocus = this.shadowRoot.querySelector('#addNeverTranslate');
        assert(toFocus);
        focusWithoutInk(toFocus);
    }
    onNeverTranslateLanguagesAdded_(e) {
        const languagesToAdd = e.detail;
        languagesToAdd.forEach(languageCode => {
            this.languageHelper_.disableTranslateLanguage(languageCode);
            this.languageSettingsMetricsProxy_.recordSettingsMetric(LanguageSettingsActionType.ADD_TO_NEVER_TRANSLATE);
        });
    }
    /**
     * Removes a language from the never translate languages list.
     */
    onRemoveNeverTranslateLanguageClick_(e) {
        const languageCode = e.model.item.code;
        this.languageHelper_.enableTranslateLanguage(languageCode);
        this.languageSettingsMetricsProxy_.recordSettingsMetric(LanguageSettingsActionType.REMOVE_FROM_NEVER_TRANSLATE);
    }
    onTranslateToggleChange_(e) {
        this.languageSettingsMetricsProxy_.recordSettingsMetric(e.target.checked ?
            LanguageSettingsActionType.ENABLE_TRANSLATE_GLOBALLY :
            LanguageSettingsActionType.DISABLE_TRANSLATE_GLOBALLY);
    }
    /**
     * @return Whether the list has any items.
     */
    hasSome_(list) {
        return !!list.length;
    }
    /**
     * @return Whether the list is has the given length.
     */
    hasLength_(list, length) {
        return list.length === length;
    }
    /**
     * Gets the list of languages that chrome can translate
     */
    getTranslatableLanguages_() {
        return this.languages.supported.filter(language => {
            return this.isTranslateSupported_(language);
        });
    }
    /**
     * Filters only for translate supported languages
     */
    isTranslateSupported_(language) {
        return isTranslateBaseLanguage(language);
    }
}
customElements.define(SettingsTranslatePageElement.is, SettingsTranslatePageElement);
