// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import 'chrome://resources/cr_elements/cr_checkbox/cr_checkbox.js';
import 'chrome://resources/cr_elements/cr_icon_button/cr_icon_button.js';
import 'chrome://resources/cr_elements/cr_expand_button/cr_expand_button.js';
import 'chrome://resources/cr_elements/cr_input/cr_input.js';
import 'chrome://resources/cr_elements/cr_url_list_item/cr_url_list_item.js';
import 'chrome://resources/cr_elements/cr_icon/cr_icon.js';
import { PriceTrackingBrowserProxyImpl } from '//resources/cr_components/commerce/price_tracking_browser_proxy.js';
import { CrLitElement } from '//resources/lit/v3_0/lit.rollup.js';
import { CrUrlListItemSize } from 'chrome://resources/cr_elements/cr_url_list_item/cr_url_list_item.js';
import { loadTimeData } from 'chrome://resources/js/load_time_data.js';
import { isRTL } from 'chrome://resources/js/util.js';
import { KeyArrowNavigationService } from './keyboard_arrow_navigation_service.js';
import { getCss } from './power_bookmark_row.css.js';
import { getHtml } from './power_bookmark_row.html.js';
import { PowerBookmarksService } from './power_bookmarks_service.js';
import { getFolderLabel } from './power_bookmarks_utils.js';
export const NESTED_BOOKMARKS_BASE_MARGIN = 28;
export const NESTED_BOOKMARKS_MARGIN_PER_DEPTH = 12;
export const BOOKMARK_ROW_LOAD_EVENT = 'bookmark-row-connected-event';
export class PowerBookmarkRowElement extends CrLitElement {
    static get is() {
        return 'power-bookmark-row';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            bookmark: { type: Object },
            compact: { type: Boolean },
            bookmarksTreeViewEnabled: { type: Boolean },
            contextMenuBookmark: { type: Object },
            depth: {
                type: Number,
                reflect: true,
            },
            hasCheckbox: {
                type: Boolean,
                reflect: true,
            },
            selectedBookmarks: { type: Array },
            renamingId: { type: String },
            imageUrls: { type: Object },
            isPriceTracked: { type: Boolean },
            searchQuery: { type: String },
            shoppingCollectionFolderId: { type: String },
            rowAriaDescription: { type: String },
            trailingIconTooltip: { type: String },
            listItemSize: { type: String },
            toggleExpand: { type: Boolean },
            isSelected: { type: Boolean },
            updatedElementIds: { type: Array },
            canDrag: { type: Boolean },
            activeFolderPath: { type: Array },
            hasFolders: { type: Boolean, reflect: true },
        };
    }
    #bookmark_accessor_storage = {
        id: '',
        parentId: '',
        index: 0,
        title: '',
        url: null,
        dateAdded: null,
        dateLastUsed: null,
        unmodifiable: false,
        children: null,
    };
    get bookmark() { return this.#bookmark_accessor_storage; }
    set bookmark(value) { this.#bookmark_accessor_storage = value; }
    #compact_accessor_storage = false;
    get compact() { return this.#compact_accessor_storage; }
    set compact(value) { this.#compact_accessor_storage = value; }
    #contextMenuBookmark_accessor_storage;
    get contextMenuBookmark() { return this.#contextMenuBookmark_accessor_storage; }
    set contextMenuBookmark(value) { this.#contextMenuBookmark_accessor_storage = value; }
    #bookmarksTreeViewEnabled_accessor_storage = loadTimeData.getBoolean('bookmarksTreeViewEnabled');
    get bookmarksTreeViewEnabled() { return this.#bookmarksTreeViewEnabled_accessor_storage; }
    set bookmarksTreeViewEnabled(value) { this.#bookmarksTreeViewEnabled_accessor_storage = value; }
    #depth_accessor_storage = 0;
    get depth() { return this.#depth_accessor_storage; }
    set depth(value) { this.#depth_accessor_storage = value; }
    #hasCheckbox_accessor_storage = false;
    get hasCheckbox() { return this.#hasCheckbox_accessor_storage; }
    set hasCheckbox(value) { this.#hasCheckbox_accessor_storage = value; }
    #selectedBookmarks_accessor_storage = [];
    get selectedBookmarks() { return this.#selectedBookmarks_accessor_storage; }
    set selectedBookmarks(value) { this.#selectedBookmarks_accessor_storage = value; }
    #renamingId_accessor_storage = '';
    get renamingId() { return this.#renamingId_accessor_storage; }
    set renamingId(value) { this.#renamingId_accessor_storage = value; }
    #searchQuery_accessor_storage;
    get searchQuery() { return this.#searchQuery_accessor_storage; }
    set searchQuery(value) { this.#searchQuery_accessor_storage = value; }
    #shoppingCollectionFolderId_accessor_storage = '';
    get shoppingCollectionFolderId() { return this.#shoppingCollectionFolderId_accessor_storage; }
    set shoppingCollectionFolderId(value) { this.#shoppingCollectionFolderId_accessor_storage = value; }
    #rowAriaDescription_accessor_storage = '';
    get rowAriaDescription() { return this.#rowAriaDescription_accessor_storage; }
    set rowAriaDescription(value) { this.#rowAriaDescription_accessor_storage = value; }
    #trailingIconTooltip_accessor_storage = '';
    get trailingIconTooltip() { return this.#trailingIconTooltip_accessor_storage; }
    set trailingIconTooltip(value) { this.#trailingIconTooltip_accessor_storage = value; }
    #toggleExpand_accessor_storage = false;
    get toggleExpand() { return this.#toggleExpand_accessor_storage; }
    set toggleExpand(value) { this.#toggleExpand_accessor_storage = value; }
    #isSelected_accessor_storage = false;
    get isSelected() { return this.#isSelected_accessor_storage; }
    set isSelected(value) { this.#isSelected_accessor_storage = value; }
    #imageUrls_accessor_storage = {};
    get imageUrls() { return this.#imageUrls_accessor_storage; }
    set imageUrls(value) { this.#imageUrls_accessor_storage = value; }
    #updatedElementIds_accessor_storage = [];
    get updatedElementIds() { return this.#updatedElementIds_accessor_storage; }
    set updatedElementIds(value) { this.#updatedElementIds_accessor_storage = value; }
    #isPriceTracked_accessor_storage = false;
    get isPriceTracked() { return this.#isPriceTracked_accessor_storage; }
    set isPriceTracked(value) { this.#isPriceTracked_accessor_storage = value; }
    #canDrag_accessor_storage = true;
    get canDrag() { return this.#canDrag_accessor_storage; }
    set canDrag(value) { this.#canDrag_accessor_storage = value; }
    #activeFolderPath_accessor_storage = [];
    get activeFolderPath() { return this.#activeFolderPath_accessor_storage; }
    set activeFolderPath(value) { this.#activeFolderPath_accessor_storage = value; }
    #hasFolders_accessor_storage = false;
    get hasFolders() { return this.#hasFolders_accessor_storage; }
    set hasFolders(value) { this.#hasFolders_accessor_storage = value; }
    #listItemSize_accessor_storage = CrUrlListItemSize.COMPACT;
    get listItemSize() { return this.#listItemSize_accessor_storage; }
    set listItemSize(value) { this.#listItemSize_accessor_storage = value; }
    bookmarksService_ = PowerBookmarksService.getInstance();
    priceTrackingProxy_ = PriceTrackingBrowserProxyImpl.getInstance();
    shoppingListenerIds_ = [];
    keyArrowNavigationService_ = KeyArrowNavigationService.getInstance();
    connectedCallback() {
        super.connectedCallback();
        this.onInputDisplayChange_();
        this.addEventListener('keydown', this.onKeydown_);
        this.addEventListener('focus', this.onFocus_);
        this.isPriceTracked = this.isPriceTracked_();
        const callbackRouter = this.priceTrackingProxy_.getCallbackRouter();
        this.shoppingListenerIds_.push(callbackRouter.priceTrackedForBookmark.addListener((product) => this.handleBookmarkSubscriptionChange_(product, true)), callbackRouter.priceUntrackedForBookmark.addListener((product) => this.handleBookmarkSubscriptionChange_(product, false)));
        this.fire(BOOKMARK_ROW_LOAD_EVENT);
    }
    disconnectedCallback() {
        this.shoppingListenerIds_.forEach(id => this.priceTrackingProxy_.getCallbackRouter().removeListener(id));
    }
    getUrl_() {
        return this.bookmark.url || undefined;
    }
    handleBookmarkSubscriptionChange_(product, subscribed) {
        if (product.bookmarkId.toString() === this.bookmark.id) {
            this.isPriceTracked = subscribed;
        }
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('bookmark') &&
            this.bookmark.id !== changedProperties.get('bookmark')?.id) {
            this.toggleExpand = false;
        }
        if (changedProperties.has('activeFolderPath')) {
            this.isSelected = this.activeFolderPath?.length > 0 &&
                this.activeFolderPath[this.activeFolderPath.length - 1].id ===
                    this.bookmark.id;
        }
        if (changedProperties.has('compact')) {
            this.listItemSize =
                this.compact ? CrUrlListItemSize.COMPACT : CrUrlListItemSize.LARGE;
            if (this.bookmarksTreeViewEnabled && this.compact) {
                // Set custom margins for nested bookmarks in tree view.
                this.style.setProperty('--base-margin', `${NESTED_BOOKMARKS_BASE_MARGIN}px`);
                this.style.setProperty('--margin-per-depth', `${NESTED_BOOKMARKS_MARGIN_PER_DEPTH}px`);
            }
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('renamingId') ||
            changedProperties.has('bookmark')) {
            if (this.renamingId === this.bookmark?.id) {
                this.onInputDisplayChange_();
            }
        }
        if (changedProperties.has('listItemSize')) {
            this.handleListItemSizeChanged_();
        }
        if (changedProperties.has('depth')) {
            this.style.setProperty('--depth', `${this.depth}`);
        }
    }
    shouldUpdate(changedProperties) {
        if (changedProperties.has('updatedElementIds')) {
            const updatedElementIds = changedProperties.get('updatedElementIds');
            if (updatedElementIds?.includes(this.bookmark?.id)) {
                return true;
            }
            changedProperties.delete('updatedElementIds');
        }
        return super.shouldUpdate(changedProperties);
    }
    async getUpdateComplete() {
        // Wait for all children to update before marking as complete.
        const result = await super.getUpdateComplete();
        const children = [...this.shadowRoot.querySelectorAll('power-bookmark-row')];
        await Promise.all(children.map(el => el.updateComplete));
        return result;
    }
    focus() {
        this.currentUrlListItem_.focus();
    }
    setExpanded_(expanded, event) {
        if (!this.isFolder_() || this.toggleExpand === expanded) {
            return;
        }
        this.toggleExpand = expanded;
        if (!this.toggleExpand) {
            this.keyArrowNavigationService_.removeElementsWithin(this);
        }
        this.dispatchEvent(new CustomEvent('power-bookmark-toggle', {
            bubbles: true,
            composed: true,
            detail: {
                bookmark: this.bookmark,
                expanded: this.toggleExpand,
                event: event,
            },
        }));
    }
    onKeydown_(e) {
        if (this.shadowRoot.activeElement !== this.currentUrlListItem_) {
            return;
        }
        const isRtl = isRTL();
        const forwardKey = isRtl ? 'ArrowLeft' : 'ArrowRight';
        const backwardKey = isRtl ? 'ArrowRight' : 'ArrowLeft';
        if (e.key === forwardKey) {
            if (this.isFolder_()) {
                if (!this.toggleExpand) {
                    this.setExpanded_(true);
                }
                else if (this.isFolderWithChildren_()) {
                    this.keyArrowNavigationService_.moveFocus(1);
                }
            }
            e.stopPropagation();
            return;
        }
        if (e.key === backwardKey) {
            if (this.isFolder_() && this.toggleExpand) {
                this.setExpanded_(false);
            }
            else {
                const parentRow = this.getRootNode()?.host;
                parentRow.focus();
                this.keyArrowNavigationService_.setCurrentFocusIndex(parentRow);
            }
            e.stopPropagation();
            return;
        }
        if (e.shiftKey && e.key === 'Tab') {
            // Hitting shift tab from CrUrlListItem to traverse focus backwards will
            // attempt to move focus to this element, which is responsible for
            // delegating focus but should itself not be focusable. So when the user
            // hits shift tab, immediately hijack focus onto itself so that the
            // browser moves focus to the focusable element before it once it
            // processes the shift tab.
            super.focus();
        }
        else if (e.key === 'Enter') {
            // Prevent iron-list from moving focus.
            e.stopPropagation();
        }
    }
    getBookmarkDescriptionForTests(bookmark) {
        return this.getBookmarkDescription_(bookmark);
    }
    onFocus_(e) {
        if (e.composedPath()[0] === this && this.matches(':focus-visible')) {
            // If trying to directly focus on this row, move the focus to the
            // <cr-url-list-item>. Otherwise, UI might be trying to directly focus on
            // a specific child (eg. the input).
            // This should only be done when focusing via keyboard, to avoid blocking
            // drag interactions.
            this.currentUrlListItem_.focus();
        }
    }
    get currentUrlListItem_() {
        return this.shadowRoot.querySelector('#crUrlListItem');
    }
    async handleListItemSizeChanged_() {
        await this.currentUrlListItem_.updateComplete;
        this.dispatchEvent(new CustomEvent('list-item-size-changed', {
            bubbles: true,
            composed: true,
        }));
    }
    isRenamingItem_() {
        return this.bookmark.id === this.renamingId;
    }
    isCheckboxChecked_() {
        return this.selectedBookmarks.includes(this.bookmark);
    }
    isBookmarksBar_() {
        return this.bookmark?.id === loadTimeData.getString('bookmarksBarId');
    }
    showTrailingIcon_() {
        return !this.isRenamingItem_() && !this.hasCheckbox;
    }
    onExpandedChanged_(event) {
        event.preventDefault();
        event.stopPropagation();
        this.setExpanded_(event.detail.value, event);
    }
    onInputDisplayChange_() {
        const input = this.shadowRoot.querySelector('#input');
        if (input) {
            input.select();
        }
    }
    /**
     * Dispatches a custom click event when the user clicks anywhere on the row.
     */
    onRowClicked_(event) {
        // Ignore clicks on the row when it has an input, to ensure the row doesn't
        // eat input clicks. Also ignore clicks if the row has no associated
        // bookmark, or if the event is a right-click.
        if (this.isRenamingItem_() || !this.bookmark || event.button === 2) {
            return;
        }
        event.preventDefault();
        event.stopPropagation();
        if (this.hasCheckbox && this.canEdit_()) {
            // Clicking the row should trigger a checkbox click rather than a
            // standard row click.
            const checkbox = this.shadowRoot.querySelector('#checkbox');
            checkbox.checked = !checkbox.checked;
            return;
        }
        this.dispatchEvent(new CustomEvent('row-clicked', {
            bubbles: true,
            composed: true,
            detail: {
                bookmark: this.bookmark,
                event: event,
            },
        }));
    }
    /**
     * Dispatches a custom click event when the user right-clicks anywhere on the
     * row.
     */
    onContextMenu_(event) {
        event.preventDefault();
        event.stopPropagation();
        this.dispatchEvent(new CustomEvent('context-menu', {
            bubbles: true,
            composed: true,
            detail: {
                bookmark: this.bookmark,
                event: event,
            },
        }));
    }
    /**
     * Dispatches a custom click event when the user clicks anywhere on the
     * trailing icon button.
     */
    onTrailingIconClicked_(event) {
        event.preventDefault();
        event.stopPropagation();
        this.dispatchEvent(new CustomEvent('trailing-icon-clicked', {
            bubbles: true,
            composed: true,
            detail: {
                bookmark: this.bookmark,
                event: event,
            },
        }));
    }
    /**
     * Dispatches a custom click event when the user clicks on the checkbox.
     */
    onCheckboxChange_(event) {
        event.preventDefault();
        event.stopPropagation();
        this.dispatchEvent(new CustomEvent('checkbox-change', {
            bubbles: true,
            composed: true,
            detail: {
                bookmark: this.bookmark,
                checked: event.target.checked,
            },
        }));
    }
    /**
     * Triggers an input change event on enter. Extends default input behavior
     * which only triggers a change event if the value of the input has changed.
     */
    onInputKeyDown_(event) {
        if (event.key === 'Enter') {
            event.stopPropagation();
            this.onInputChange_(event);
        }
    }
    createInputChangeEvent_(value) {
        return new CustomEvent('input-change', {
            bubbles: true,
            composed: true,
            detail: {
                bookmark: this.bookmark,
                value: value,
            },
        });
    }
    /**
     * Triggers a custom input change event when the user hits enter or the input
     * loses focus.
     */
    onInputChange_(event) {
        event.preventDefault();
        event.stopPropagation();
        const inputElement = this.shadowRoot.querySelector('#input');
        if (inputElement) {
            this.dispatchEvent(this.createInputChangeEvent_(inputElement.value));
        }
    }
    onInputBlur_(event) {
        event.preventDefault();
        event.stopPropagation();
        this.dispatchEvent(this.createInputChangeEvent_(null));
    }
    isPriceTracked_() {
        return !!this.bookmarksService_.getPriceTrackedInfo(this.bookmark);
    }
    /**
     * Whether the given price-tracked bookmark should display as if discounted.
     */
    showDiscountedPrice_() {
        const bookmarkProductInfo = this.bookmarksService_.getPriceTrackedInfo(this.bookmark);
        if (bookmarkProductInfo) {
            return bookmarkProductInfo.info.previousPrice.length > 0;
        }
        return false;
    }
    getCurrentPrice_(bookmark) {
        const bookmarkProductInfo = this.bookmarksService_.getPriceTrackedInfo(bookmark);
        if (bookmarkProductInfo) {
            return bookmarkProductInfo.info.currentPrice;
        }
        else {
            return '';
        }
    }
    getPreviousPrice_(bookmark) {
        const bookmarkProductInfo = this.bookmarksService_.getPriceTrackedInfo(bookmark);
        if (bookmarkProductInfo) {
            return bookmarkProductInfo.info.previousPrice;
        }
        else {
            return '';
        }
    }
    getBookmarkForceHover_() {
        return this.bookmark === this.contextMenuBookmark;
    }
    shouldExpand_() {
        return this.bookmark?.children && this.bookmarksTreeViewEnabled &&
            this.compact;
    }
    canEdit_() {
        return this.bookmark?.id !== loadTimeData.getString('bookmarksBarId') &&
            this.bookmark?.id !==
                loadTimeData.getString('managedBookmarksFolderId');
    }
    isShoppingCollection_() {
        return this.bookmark?.id === this.shoppingCollectionFolderId;
    }
    isFolder_() {
        return !this.bookmark.url;
    }
    isFolderWithChildren_() {
        return this.isFolder_() && !!this.bookmark.children?.length;
    }
    getBookmarkDescription_(bookmark) {
        if (this.compact) {
            if (bookmark?.url) {
                return undefined;
            }
            const count = bookmark?.children ? bookmark?.children.length : 0;
            return loadTimeData.getStringF('bookmarkFolderChildCount', count);
        }
        else {
            let urlString;
            if (bookmark?.url) {
                const url = new URL(bookmark?.url);
                // Show chrome:// if it's a chrome internal url
                if (url.protocol === 'chrome:') {
                    urlString = 'chrome://' + url.hostname;
                }
                urlString = url.hostname;
            }
            if (urlString && this.searchQuery && bookmark?.parentId) {
                const parentFolder = this.bookmarksService_.findBookmarkWithId(bookmark?.parentId);
                const folderLabel = getFolderLabel(parentFolder);
                return loadTimeData.getStringF('urlFolderDescription', urlString, folderLabel);
            }
            return urlString;
        }
    }
    getBookmarkImageUrls_() {
        const imageUrls = [];
        if (this.bookmark?.url) {
            const imageUrl = Object.entries(this.imageUrls)
                .find(([key, _val]) => key === this.bookmark.id)?.[1];
            if (imageUrl) {
                imageUrls.push(imageUrl);
            }
        }
        else if (this.canEdit_() && this.bookmark?.children &&
            !this.isShoppingCollection_()) {
            this.bookmark?.children.forEach(child => {
                const childImageUrl = Object.entries(this.imageUrls)
                    .find(([key, _val]) => key === child.id)?.[1];
                if (childImageUrl) {
                    imageUrls.push(childImageUrl);
                }
            });
        }
        return imageUrls;
    }
    getBookmarkMenuA11yLabel_() {
        return loadTimeData.getStringF(this.bookmark.url ? 'bookmarkMenuLabel' : 'folderMenuLabel', this.bookmark.title);
    }
    getBookmarkA11yLabel_() {
        if (this.hasCheckbox) {
            if (this.isCheckboxChecked_()) {
                return loadTimeData.getStringF(this.bookmark.url ? 'deselectBookmarkLabel' : 'deselectFolderLabel', this.bookmark.title);
            }
            return loadTimeData.getStringF(this.bookmark.url ? 'selectBookmarkLabel' : 'selectFolderLabel', this.bookmark.title);
        }
        return loadTimeData.getStringF(this.bookmark.url ? 'openBookmarkLabel' : 'openFolderLabel', this.bookmark.title);
    }
    getBookmarkA11yDescription_() {
        const bookmark = this.bookmark;
        let description = '';
        if (this.bookmarksService_.getPriceTrackedInfo(bookmark)) {
            description += loadTimeData.getStringF('a11yDescriptionPriceTracking', this.getCurrentPrice_(bookmark));
            const previousPrice = this.getPreviousPrice_(bookmark);
            if (previousPrice) {
                description += ' ' +
                    loadTimeData.getStringF('a11yDescriptionPriceChange', previousPrice);
            }
        }
        return description;
    }
    getBookmarkDescriptionMeta_() {
        // If there is a price available for the product and it isn't being
        // tracked, return the current price which will be added to the description
        // meta section.
        const productInfo = this.bookmarksService_.getAvailableProductInfo(this.bookmark);
        if (productInfo && productInfo.info.currentPrice &&
            !this.isPriceTracked_()) {
            return productInfo.info.currentPrice;
        }
        return '';
    }
}
customElements.define(PowerBookmarkRowElement.is, PowerBookmarkRowElement);
