"use strict";
import * as Common from "../common/common.js";
import * as i18n from "../i18n/i18n.js";
import * as Root from "../root/root.js";
import * as EnhancedTraces from "./EnhancedTracesParser.js";
import { TraceObject } from "./TraceObject.js";
const UIStrings = {
  /**
   * @description Text that appears when no source text is available for the given script
   */
  noSourceText: "No source text available",
  /**
   * @description Text to indicate rehydrating connection cannot find host window
   */
  noHostWindow: "Can not find host window",
  /**
   * @description Text to indicate that there is an error loading the log
   */
  errorLoadingLog: "Error loading log"
};
const str_ = i18n.i18n.registerUIStrings("core/sdk/RehydratingConnection.ts", UIStrings);
const i18nString = i18n.i18n.getLocalizedString.bind(void 0, str_);
export var RehydratingConnectionState = /* @__PURE__ */ ((RehydratingConnectionState2) => {
  RehydratingConnectionState2[RehydratingConnectionState2["UNINITIALIZED"] = 1] = "UNINITIALIZED";
  RehydratingConnectionState2[RehydratingConnectionState2["INITIALIZED"] = 2] = "INITIALIZED";
  RehydratingConnectionState2[RehydratingConnectionState2["REHYDRATED"] = 3] = "REHYDRATED";
  return RehydratingConnectionState2;
})(RehydratingConnectionState || {});
export class RehydratingConnection {
  rehydratingConnectionState = 1 /* UNINITIALIZED */;
  onDisconnect = null;
  onMessage = null;
  trace = null;
  sessions = /* @__PURE__ */ new Map();
  #onConnectionLost;
  #rehydratingWindow = window;
  #onReceiveHostWindowPayloadBound = this.onReceiveHostWindowPayload.bind(this);
  constructor(onConnectionLost) {
    this.#onConnectionLost = onConnectionLost;
    if (!this.#maybeHandleLoadingFromUrl()) {
      this.#setupMessagePassing();
    }
  }
  /** Returns true if found a trace URL. */
  #maybeHandleLoadingFromUrl() {
    let traceUrl = Root.Runtime.Runtime.queryParam("traceURL");
    if (!traceUrl) {
      const timelineUrl = Root.Runtime.Runtime.queryParam("loadTimelineFromURL");
      if (timelineUrl) {
        traceUrl = decodeURIComponent(timelineUrl);
      }
    }
    if (traceUrl) {
      void fetch(traceUrl).then((r) => r.arrayBuffer()).then((b) => Common.Gzip.arrayBufferToString(b)).then((traceJson) => {
        const trace = new TraceObject(JSON.parse(traceJson));
        void this.startHydration(trace);
      });
      return true;
    }
    return false;
  }
  #setupMessagePassing() {
    this.#rehydratingWindow.addEventListener("message", this.#onReceiveHostWindowPayloadBound);
    if (this.#rehydratingWindow.opener) {
      this.#rehydratingWindow.opener.postMessage({ type: "REHYDRATING_WINDOW_READY" });
    } else if (this.#rehydratingWindow !== window.top) {
      this.#rehydratingWindow.parent.postMessage({ type: "REHYDRATING_IFRAME_READY" });
    } else {
      this.#onConnectionLost(i18nString(UIStrings.noHostWindow));
    }
  }
  /**
   * This is a callback for rehydrated session to receive payload from host window. Payload includes but not limited to
   * the trace event and all necessary data to power a rehydrated session.
   */
  onReceiveHostWindowPayload(event) {
    if (event.data.type === "REHYDRATING_TRACE_FILE") {
      const traceJson = event.data.traceJson;
      let trace;
      try {
        trace = new TraceObject(JSON.parse(traceJson));
      } catch {
        this.#onConnectionLost(i18nString(UIStrings.errorLoadingLog));
        return;
      }
      void this.startHydration(trace);
    }
    this.#rehydratingWindow.removeEventListener("message", this.#onReceiveHostWindowPayloadBound);
  }
  async startHydration(trace) {
    if (!this.onMessage || this.rehydratingConnectionState !== 2 /* INITIALIZED */) {
      return false;
    }
    if (!("traceEvents" in trace)) {
      console.error("RehydratingConnection failed to initialize due to missing trace events in payload");
      return false;
    }
    this.trace = trace;
    const enhancedTracesParser = new EnhancedTraces.EnhancedTracesParser(trace);
    const hydratingData = enhancedTracesParser.data();
    let sessionId = 0;
    this.sessions.set(sessionId, new RehydratingSessionBase(this));
    for (const hydratingDataPerTarget of hydratingData) {
      const target = hydratingDataPerTarget.target;
      const executionContexts = hydratingDataPerTarget.executionContexts;
      const scripts = hydratingDataPerTarget.scripts;
      this.postToFrontend({
        method: "Target.targetCreated",
        params: {
          targetInfo: {
            targetId: target.targetId,
            type: target.type,
            title: target.url,
            url: target.url,
            attached: false,
            canAccessOpener: false
          }
        }
      });
      sessionId += 1;
      const session = new RehydratingSession(sessionId, target, executionContexts, scripts, this);
      this.sessions.set(sessionId, session);
      session.declareSessionAttachedToTarget();
    }
    await this.#onRehydrated();
    return true;
  }
  async #onRehydrated() {
    if (!this.trace) {
      return;
    }
    this.rehydratingConnectionState = 3 /* REHYDRATED */;
    await Common.Revealer.reveal(this.trace);
  }
  setOnMessage(onMessage) {
    this.onMessage = onMessage;
    this.rehydratingConnectionState = 2 /* INITIALIZED */;
  }
  setOnDisconnect(onDisconnect) {
    this.onDisconnect = onDisconnect;
  }
  // The function "sendRawMessage" is typically devtools front-end
  // sending message to the backend via CDP. In this case, given that Rehydrating
  // connection is an emulation of devtool back-end, sendRawMessage here
  // is in fact rehydrating connection directly handling and acting on the
  // receieved message.
  sendRawMessage(message) {
    if (typeof message === "string") {
      message = JSON.parse(message);
    }
    const data = message;
    if (typeof data.sessionId !== "undefined") {
      const session = this.sessions.get(data.sessionId);
      if (session) {
        session.handleFrontendMessageAsFakeCDPAgent(data);
      } else {
        console.error("Invalid SessionId: " + data.sessionId);
      }
    } else {
      this.sessions.get(0)?.handleFrontendMessageAsFakeCDPAgent(data);
    }
  }
  // Posting rehydrating connection's message/response
  // to devtools frontend through debugger protocol.
  postToFrontend(arg) {
    if (this.onMessage) {
      this.onMessage(arg);
    } else {
      console.error("onMessage was not initialized");
    }
  }
  disconnect() {
    return Promise.reject();
  }
}
class RehydratingSessionBase {
  connection = null;
  constructor(connection) {
    this.connection = connection;
  }
  sendMessageToFrontend(payload) {
    setTimeout(() => {
      this.connection?.postToFrontend(payload);
    });
  }
  handleFrontendMessageAsFakeCDPAgent(data) {
    this.sendMessageToFrontend({
      id: data.id,
      result: {}
    });
  }
}
export class RehydratingSession extends RehydratingSessionBase {
  sessionId;
  target;
  executionContexts = [];
  scripts = [];
  constructor(sessionId, target, executionContexts, scripts, connection) {
    super(connection);
    this.sessionId = sessionId;
    this.target = target;
    this.executionContexts = executionContexts;
    this.scripts = scripts;
  }
  sendMessageToFrontend(payload, attachSessionId = true) {
    if (this.sessionId !== 0 && attachSessionId) {
      payload.sessionId = this.sessionId;
    }
    super.sendMessageToFrontend(payload);
  }
  handleFrontendMessageAsFakeCDPAgent(data) {
    switch (data.method) {
      case "Runtime.enable":
        this.handleRuntimeEnabled(data.id);
        break;
      case "Debugger.enable":
        this.handleDebuggerEnable(data.id);
        break;
      case "Debugger.getScriptSource":
        if (data.params) {
          const params = data.params;
          this.handleDebuggerGetScriptSource(data.id, params.scriptId);
        }
        break;
      default:
        this.sendMessageToFrontend({
          id: data.id,
          result: {}
        });
        break;
    }
  }
  declareSessionAttachedToTarget() {
    this.sendMessageToFrontend(
      {
        method: "Target.attachedToTarget",
        params: {
          sessionId: this.sessionId,
          waitingForDebugger: false,
          targetInfo: {
            targetId: this.target.targetId,
            type: this.target.type,
            title: this.target.url,
            url: this.target.url,
            attached: true,
            canAccessOpener: false
          }
        }
      },
      /* attachSessionId */
      false
    );
  }
  // Runtime.Enable indicates that Runtime domain is flushing the event to communicate
  // the current state with the backend. In rehydrating connection, we made up the artificial
  // execution context to support the rehydrated session.
  handleRuntimeEnabled(id) {
    for (const executionContext of this.executionContexts) {
      executionContext.name = executionContext.origin;
      this.sendMessageToFrontend({
        method: "Runtime.executionContextCreated",
        params: {
          context: executionContext
        }
      });
    }
    this.sendMessageToFrontend({
      id,
      result: {}
    });
  }
  handleDebuggerGetScriptSource(id, scriptId) {
    const script = this.scripts.find((script2) => script2.scriptId === scriptId);
    if (!script) {
      console.error("No script for id: " + scriptId);
      return;
    }
    this.sendMessageToFrontend({
      id,
      result: {
        scriptSource: typeof script.sourceText === "undefined" ? i18nString(UIStrings.noSourceText) : script.sourceText
      }
    });
  }
  // Debugger.Enable indicates that Debugger domain is flushing the event to communicate
  // the current state with the backend. In rehydrating connection, we made up the artificial
  // script parsed event to communicate the current script state and respond with a mock
  // debugger id.
  handleDebuggerEnable(id) {
    for (const script of this.scripts) {
      this.sendMessageToFrontend({
        method: "Debugger.scriptParsed",
        params: script
      });
    }
    const mockDebuggerId = "7777777777777777777.8888888888888888888";
    this.sendMessageToFrontend({
      id,
      result: {
        debuggerId: mockDebuggerId
      }
    });
  }
}
//# sourceMappingURL=RehydratingConnection.js.map
