"use strict";
import * as SDK from "../../core/sdk/sdk.js";
import * as StackTrace from "./stack_trace.js";
import { AsyncFragmentImpl, FragmentImpl, FrameImpl, StackTraceImpl } from "./StackTraceImpl.js";
import { Trie } from "./Trie.js";
export class StackTraceModel extends SDK.SDKModel.SDKModel {
  #trie = new Trie();
  /** @returns the {@link StackTraceModel} for the target, or the model for the primaryPageTarget when passing null/undefined */
  static #modelForTarget(target) {
    const model = (target ?? SDK.TargetManager.TargetManager.instance().primaryPageTarget())?.model(StackTraceModel);
    if (!model) {
      throw new Error("Unable to find StackTraceModel");
    }
    return model;
  }
  async createFromProtocolRuntime(stackTrace, rawFramesToUIFrames) {
    const translatePromises = [];
    const fragment = this.#createFragment(stackTrace.callFrames);
    translatePromises.push(this.#translateFragment(fragment, rawFramesToUIFrames));
    const asyncFragments = [];
    const debuggerModel = this.target().model(SDK.DebuggerModel.DebuggerModel);
    if (debuggerModel) {
      for await (const { stackTrace: asyncStackTrace, target } of debuggerModel.iterateAsyncParents(stackTrace)) {
        const model = StackTraceModel.#modelForTarget(target);
        const asyncFragment = model.#createFragment(asyncStackTrace.callFrames);
        translatePromises.push(model.#translateFragment(asyncFragment, rawFramesToUIFrames));
        asyncFragments.push(new AsyncFragmentImpl(asyncStackTrace.description ?? "", asyncFragment));
      }
    }
    await Promise.all(translatePromises);
    return new StackTraceImpl(fragment, asyncFragments);
  }
  /** Trigger re-translation of all fragments with the provide script in their call stack */
  async scriptInfoChanged(script, translateRawFrames) {
    const translatePromises = [];
    let stackTracesToUpdate = /* @__PURE__ */ new Set();
    for (const fragment of this.#affectedFragments(script)) {
      if (fragment.node.children.length === 0) {
        translatePromises.push(this.#translateFragment(fragment, translateRawFrames));
      }
      stackTracesToUpdate = stackTracesToUpdate.union(fragment.stackTraces);
    }
    await Promise.all(translatePromises);
    for (const stackTrace of stackTracesToUpdate) {
      stackTrace.dispatchEventToListeners(StackTrace.StackTrace.Events.UPDATED);
    }
  }
  #createFragment(frames) {
    return FragmentImpl.getOrCreate(this.#trie.insert(frames));
  }
  async #translateFragment(fragment, rawFramesToUIFrames) {
    const rawFrames = fragment.node.getCallStack().map((node) => node.rawFrame).toArray();
    const uiFrames = await rawFramesToUIFrames(rawFrames, this.target());
    console.assert(rawFrames.length === uiFrames.length, "Broken rawFramesToUIFrames implementation");
    let i = 0;
    for (const node of fragment.node.getCallStack()) {
      node.frames = uiFrames[i++].map(
        (frame) => new FrameImpl(
          frame.url,
          frame.uiSourceCode,
          frame.name,
          frame.line,
          frame.column,
          frame.missingDebugInfo
        )
      );
    }
  }
  #affectedFragments(script) {
    const affectedBranches = /* @__PURE__ */ new Set();
    this.#trie.walk(null, (node) => {
      if (node.rawFrame.scriptId === script.scriptId || !node.rawFrame.scriptId && node.rawFrame.url === script.sourceURL) {
        affectedBranches.add(node);
        return false;
      }
      return true;
    });
    const fragments = /* @__PURE__ */ new Set();
    for (const branch of affectedBranches) {
      this.#trie.walk(branch, (node) => {
        if (node.fragment) {
          fragments.add(node.fragment);
        }
        return true;
      });
    }
    return fragments;
  }
}
SDK.SDKModel.SDKModel.register(StackTraceModel, { capabilities: SDK.Target.Capability.NONE, autostart: false });
//# sourceMappingURL=StackTraceModel.js.map
