"use strict";
import * as Common from "../../core/common/common.js";
import * as Host from "../../core/host/host.js";
import * as i18n from "../../core/i18n/i18n.js";
import * as Platform from "../../core/platform/platform.js";
import * as Root from "../../core/root/root.js";
import * as SDK from "../../core/sdk/sdk.js";
import * as PanelCommon from "../../panels/common/common.js";
import * as Buttons from "../../ui/components/buttons/buttons.js";
import * as TreeOutline from "../../ui/components/tree_outline/tree_outline.js";
import * as UI from "../../ui/legacy/legacy.js";
import * as VisualLogging from "../../ui/visual_logging/visual_logging.js";
import { AccessibilityTreeView } from "./AccessibilityTreeView.js";
import { ColorSwatchPopoverIcon } from "./ColorSwatchPopoverIcon.js";
import * as ElementsComponents from "./components/components.js";
import { ComputedStyleModel } from "./ComputedStyleModel.js";
import { ComputedStyleWidget } from "./ComputedStyleWidget.js";
import elementsPanelStyles from "./elementsPanel.css.js";
import { DOMTreeWidget } from "./ElementsTreeOutline.js";
import { LayoutPane } from "./LayoutPane.js";
import { MetricsSidebarPane } from "./MetricsSidebarPane.js";
import {
  Events as StylesSidebarPaneEvents,
  StylesSidebarPane
} from "./StylesSidebarPane.js";
const UIStrings = {
  /**
   * @description Placeholder text for the search box the Elements Panel. Selector refers to CSS
   * selectors.
   */
  findByStringSelectorOrXpath: "Find by string, selector, or `XPath`",
  /**
   * @description Button text for a button that takes the user to the Accessibility Tree View from the
   * DOM tree view, in the Elements panel.
   */
  switchToAccessibilityTreeView: "Switch to Accessibility Tree view",
  /**
   * @description Button text for a button that takes the user to the DOM tree view from the
   * Accessibility Tree View, in the Elements panel.
   */
  switchToDomTreeView: "Switch to DOM Tree view",
  /**
   * @description Tooltip for the the Computed Styles sidebar toggle in the Styles pane. Command to
   * open/show the sidebar.
   */
  showComputedStylesSidebar: "Show Computed Styles sidebar",
  /**
   * @description Tooltip for the the Computed Styles sidebar toggle in the Styles pane. Command to
   * close/hide the sidebar.
   */
  hideComputedStylesSidebar: "Hide Computed Styles sidebar",
  /**
   * @description Screen reader announcement when the computed styles sidebar is shown in the Elements panel.
   */
  computedStylesShown: "Computed Styles sidebar shown",
  /**
   * @description Screen reader announcement when the computed styles sidebar is hidden in the Elements panel.
   */
  computedStylesHidden: "Computed Styles sidebar hidden",
  /**
   * @description Title of a pane in the Elements panel that shows computed styles for the selected
   * HTML element. Computed styles are the final, actual styles of the element, including all
   * implicit and specified styles.
   */
  computed: "Computed",
  /**
   * @description Title of a pane in the Elements panel that shows the CSS styles for the selected
   * HTML element.
   */
  styles: "Styles",
  /**
   * @description A context menu item to reveal a node in the DOM tree of the Elements Panel
   */
  openInElementsPanel: "Open in Elements panel",
  /**
   * @description Warning/error text displayed when a node cannot be found in the current page.
   */
  nodeCannotBeFoundInTheCurrent: "Node cannot be found in the current page.",
  /**
   * @description Console warning when a user tries to reveal a non-node type Remote Object. A remote
   * object is a JavaScript object that is not stored in DevTools, that DevTools has a connection to.
   * It should correspond to a local node.
   */
  theRemoteObjectCouldNotBe: "The remote object could not be resolved to a valid node.",
  /**
   * @description Console warning when the user tries to reveal a deferred DOM Node that resolves as
   * null. A deferred DOM node is a node we know about but have not yet fetched from the backend (we
   * defer the work until later).
   */
  theDeferredDomNodeCouldNotBe: "The deferred `DOM` Node could not be resolved to a valid node.",
  /**
   * @description Text in Elements Panel of the Elements panel. Shows the current CSS Pseudo-classes
   * applicable to the selected HTML element.
   * @example {::after, ::before} PH1
   */
  elementStateS: "Element state: {PH1}",
  /**
   * @description Accessible name for side panel toolbar.
   */
  sidePanelToolbar: "Side panel toolbar",
  /**
   * @description Accessible name for side panel contents.
   */
  sidePanelContent: "Side panel content",
  /**
   * @description Accessible name for the DOM tree explorer view.
   */
  domTreeExplorer: "DOM tree explorer",
  /**
   * @description A context menu item to reveal a submenu with badge settings.
   */
  adornerSettings: "Badge settings"
};
const str_ = i18n.i18n.registerUIStrings("panels/elements/ElementsPanel.ts", UIStrings);
const i18nString = i18n.i18n.getLocalizedString.bind(void 0, str_);
export var SidebarPaneTabId = /* @__PURE__ */ ((SidebarPaneTabId2) => {
  SidebarPaneTabId2["COMPUTED"] = "computed";
  SidebarPaneTabId2["STYLES"] = "styles";
  return SidebarPaneTabId2;
})(SidebarPaneTabId || {});
const createAccessibilityTreeToggleButton = (isActive) => {
  const button = new Buttons.Button.Button();
  const title = isActive ? i18nString(UIStrings.switchToDomTreeView) : i18nString(UIStrings.switchToAccessibilityTreeView);
  button.data = {
    active: isActive,
    variant: Buttons.Button.Variant.TOOLBAR,
    iconName: "person",
    title,
    jslogContext: "toggle-accessibility-tree"
  };
  button.tabIndex = 0;
  button.classList.add("axtree-button");
  if (isActive) {
    button.classList.add("active");
  }
  return button;
};
let elementsPanelInstance;
export class ElementsPanel extends UI.Panel.Panel {
  splitWidget;
  #searchableView;
  mainContainer;
  domTreeContainer;
  splitMode;
  accessibilityTreeView;
  breadcrumbs;
  stylesWidget;
  computedStyleWidget;
  metricsWidget;
  searchResults;
  currentSearchResultIndex;
  pendingNodeReveal;
  adornerManager;
  adornersByName;
  accessibilityTreeButton;
  domTreeButton;
  selectedNodeOnReset;
  hasNonDefaultSelectedNode;
  searchConfig;
  omitDefaultSelection;
  notFirstInspectElement;
  sidebarPaneView;
  stylesViewToReveal;
  nodeInsertedTaskRunner = {
    queue: Promise.resolve(),
    run(task) {
      this.queue = this.queue.then(task);
    }
  };
  cssStyleTrackerByCSSModel;
  #domTreeWidget;
  getTreeOutlineForTesting() {
    return this.#domTreeWidget.getTreeOutlineForTesting();
  }
  constructor() {
    super("elements");
    this.registerRequiredCSS(elementsPanelStyles);
    this.splitWidget = new UI.SplitWidget.SplitWidget(true, true, "elements-panel-split-view-state", 325, 325);
    this.splitWidget.addEventListener(
      UI.SplitWidget.Events.SIDEBAR_SIZE_CHANGED,
      this.updateTreeOutlineVisibleWidth.bind(this)
    );
    this.splitWidget.show(this.element);
    this.#searchableView = new UI.SearchableView.SearchableView(this, null);
    this.#searchableView.setMinimalSearchQuerySize(0);
    this.#searchableView.setMinimumSize(25, 28);
    this.#searchableView.setPlaceholder(i18nString(UIStrings.findByStringSelectorOrXpath));
    const stackElement = this.#searchableView.element;
    this.mainContainer = document.createElement("div");
    this.domTreeContainer = document.createElement("div");
    const crumbsContainer = document.createElement("div");
    if (Root.Runtime.experiments.isEnabled("full-accessibility-tree")) {
      this.initializeFullAccessibilityTreeView();
    }
    this.mainContainer.appendChild(this.domTreeContainer);
    stackElement.appendChild(this.mainContainer);
    stackElement.appendChild(crumbsContainer);
    UI.ARIAUtils.markAsMain(this.domTreeContainer);
    UI.ARIAUtils.setLabel(this.domTreeContainer, i18nString(UIStrings.domTreeExplorer));
    this.splitWidget.setMainWidget(this.#searchableView);
    this.splitMode = null;
    this.mainContainer.id = "main-content";
    this.domTreeContainer.id = "elements-content";
    this.domTreeContainer.tabIndex = -1;
    if (Common.Settings.Settings.instance().moduleSetting("dom-word-wrap").get()) {
      this.domTreeContainer.classList.add("elements-wrap");
    }
    Common.Settings.Settings.instance().moduleSetting("dom-word-wrap").addChangeListener(this.domWordWrapSettingChanged.bind(this));
    crumbsContainer.id = "elements-crumbs";
    if (this.domTreeButton) {
      this.accessibilityTreeView = new AccessibilityTreeView(this.domTreeButton, new TreeOutline.TreeOutline.TreeOutline());
    }
    this.breadcrumbs = new ElementsComponents.ElementsBreadcrumbs.ElementsBreadcrumbs();
    this.breadcrumbs.addEventListener("breadcrumbsnodeselected", (event) => {
      this.crumbNodeSelected(event);
    });
    crumbsContainer.appendChild(this.breadcrumbs);
    const computedStyleModel = new ComputedStyleModel();
    this.stylesWidget = new StylesSidebarPane(computedStyleModel);
    this.computedStyleWidget = new ComputedStyleWidget(computedStyleModel);
    this.metricsWidget = new MetricsSidebarPane(computedStyleModel);
    Common.Settings.Settings.instance().moduleSetting("sidebar-position").addChangeListener(this.updateSidebarPosition.bind(this));
    this.updateSidebarPosition();
    this.cssStyleTrackerByCSSModel = /* @__PURE__ */ new Map();
    this.currentSearchResultIndex = -1;
    this.pendingNodeReveal = false;
    this.adornerManager = new ElementsComponents.AdornerManager.AdornerManager(
      Common.Settings.Settings.instance().moduleSetting("adorner-settings")
    );
    this.adornersByName = /* @__PURE__ */ new Map();
    this.#domTreeWidget = new DOMTreeWidget();
    this.#domTreeWidget.omitRootDOMNode = true;
    this.#domTreeWidget.selectEnabled = true;
    this.#domTreeWidget.onSelectedNodeChanged = this.selectedNodeChanged.bind(this);
    this.#domTreeWidget.onElementsTreeUpdated = this.updateBreadcrumbIfNeeded.bind(this);
    this.#domTreeWidget.onDocumentUpdated = this.documentUpdated.bind(this);
    this.#domTreeWidget.setWordWrap(Common.Settings.Settings.instance().moduleSetting("dom-word-wrap").get());
    SDK.TargetManager.TargetManager.instance().observeModels(SDK.DOMModel.DOMModel, this, { scoped: true });
    SDK.TargetManager.TargetManager.instance().addEventListener(
      SDK.TargetManager.Events.NAME_CHANGED,
      (event) => this.targetNameChanged(event.data)
    );
    Common.Settings.Settings.instance().moduleSetting("show-ua-shadow-dom").addChangeListener(this.showUAShadowDOMChanged.bind(this));
    PanelCommon.ExtensionServer.ExtensionServer.instance().addEventListener(
      PanelCommon.ExtensionServer.Events.SidebarPaneAdded,
      this.extensionSidebarPaneAdded,
      this
    );
  }
  initializeFullAccessibilityTreeView() {
    this.accessibilityTreeButton = createAccessibilityTreeToggleButton(false);
    this.accessibilityTreeButton.addEventListener("click", this.showAccessibilityTree.bind(this));
    this.domTreeButton = createAccessibilityTreeToggleButton(true);
    this.domTreeButton.addEventListener("click", this.showDOMTree.bind(this));
    this.mainContainer.appendChild(this.accessibilityTreeButton);
  }
  showAccessibilityTree() {
    if (this.accessibilityTreeView) {
      this.splitWidget.setMainWidget(this.accessibilityTreeView);
    }
  }
  showDOMTree() {
    this.splitWidget.setMainWidget(this.#searchableView);
    const selectedNode = this.selectedDOMNode();
    if (!selectedNode) {
      return;
    }
    this.#domTreeWidget.selectDOMNodeWithoutReveal(selectedNode);
  }
  toggleAccessibilityTree() {
    if (!this.domTreeButton) {
      return;
    }
    if (this.splitWidget.mainWidget() === this.accessibilityTreeView) {
      this.showDOMTree();
    } else {
      this.showAccessibilityTree();
    }
  }
  static instance(opts = { forceNew: null }) {
    const { forceNew } = opts;
    if (!elementsPanelInstance || forceNew) {
      elementsPanelInstance = new ElementsPanel();
    }
    return elementsPanelInstance;
  }
  revealProperty(cssProperty) {
    if (!this.sidebarPaneView || !this.stylesViewToReveal) {
      return Promise.resolve();
    }
    return this.sidebarPaneView.showView(this.stylesViewToReveal).then(() => {
      this.stylesWidget.revealProperty(cssProperty);
    });
  }
  resolveLocation(_locationName) {
    return this.sidebarPaneView || null;
  }
  showToolbarPane(widget, toggle) {
    this.stylesWidget.showToolbarPane(widget, toggle);
  }
  modelAdded(domModel) {
    this.setupStyleTracking(domModel.cssModel());
    this.#domTreeWidget.modelAdded(domModel);
    if (this.isShowing()) {
      this.wasShown();
    }
    if (this.domTreeContainer.hasFocus()) {
      this.#domTreeWidget.focus();
    }
    domModel.addEventListener(SDK.DOMModel.Events.DocumentUpdated, this.documentUpdatedEvent, this);
    domModel.addEventListener(SDK.DOMModel.Events.NodeInserted, this.handleNodeInserted, this);
  }
  modelRemoved(domModel) {
    domModel.removeEventListener(SDK.DOMModel.Events.DocumentUpdated, this.documentUpdatedEvent, this);
    domModel.removeEventListener(SDK.DOMModel.Events.NodeInserted, this.handleNodeInserted, this);
    this.#domTreeWidget.modelRemoved(domModel);
    if (!domModel.parentModel()) {
      this.#domTreeWidget.detach();
    }
    this.removeStyleTracking(domModel.cssModel());
  }
  handleNodeInserted(event) {
    this.nodeInsertedTaskRunner.run(async () => {
      const node = event.data;
      if (!node.isViewTransitionPseudoNode()) {
        return;
      }
      const cssModel = node.domModel().cssModel();
      const styleSheetHeader = await cssModel.requestViaInspectorStylesheet(node.frameId());
      if (!styleSheetHeader) {
        return;
      }
      const cssText = await cssModel.getStyleSheetText(styleSheetHeader.id);
      if (cssText?.includes(`${node.simpleSelector()} {`)) {
        return;
      }
      await cssModel.setStyleSheetText(styleSheetHeader.id, `${cssText}
${node.simpleSelector()} {}`, false);
    });
  }
  targetNameChanged(target) {
    const domModel = target.model(SDK.DOMModel.DOMModel);
    if (!domModel) {
      return;
    }
  }
  updateTreeOutlineVisibleWidth() {
    let width = this.splitWidget.element.offsetWidth;
    if (this.splitWidget.isVertical()) {
      width -= this.splitWidget.sidebarSize();
    }
    this.#domTreeWidget.visibleWidth = width;
  }
  focus() {
    if (this.#domTreeWidget.empty()) {
      this.domTreeContainer.focus();
    } else {
      this.#domTreeWidget.focus();
    }
  }
  searchableView() {
    return this.#searchableView;
  }
  wasShown() {
    super.wasShown();
    UI.Context.Context.instance().setFlavor(ElementsPanel, this);
    this.#domTreeWidget.show(this.domTreeContainer);
  }
  willHide() {
    SDK.OverlayModel.OverlayModel.hideDOMNodeHighlight();
    this.#domTreeWidget.detach();
    super.willHide();
    UI.Context.Context.instance().setFlavor(ElementsPanel, null);
  }
  onResize() {
    this.element.window().requestAnimationFrame(this.updateSidebarPosition.bind(this));
    this.updateTreeOutlineVisibleWidth();
  }
  selectedNodeChanged(event) {
    let selectedNode = event.data.node;
    if (selectedNode?.pseudoType() && !selectedNode.parentNode) {
      selectedNode = null;
    }
    const { focus: focus2 } = event.data;
    if (!selectedNode) {
      this.#domTreeWidget.selectDOMNode(null);
    }
    if (selectedNode) {
      const activeNode = ElementsComponents.Helper.legacyNodeToElementsComponentsNode(selectedNode);
      const crumbs = [activeNode];
      for (let current = selectedNode.parentNode; current; current = current.parentNode) {
        crumbs.push(ElementsComponents.Helper.legacyNodeToElementsComponentsNode(current));
      }
      this.breadcrumbs.data = {
        crumbs,
        selectedNode: ElementsComponents.Helper.legacyNodeToElementsComponentsNode(selectedNode)
      };
      if (this.accessibilityTreeView) {
        void this.accessibilityTreeView.selectedNodeChanged(selectedNode);
      }
    } else {
      this.breadcrumbs.data = { crumbs: [], selectedNode: null };
    }
    UI.Context.Context.instance().setFlavor(SDK.DOMModel.DOMNode, selectedNode);
    if (!selectedNode) {
      return;
    }
    void selectedNode.setAsInspectedNode();
    if (focus2) {
      this.selectedNodeOnReset = selectedNode;
      this.hasNonDefaultSelectedNode = true;
    }
    const executionContexts = selectedNode.domModel().runtimeModel().executionContexts();
    const nodeFrameId = selectedNode.frameId();
    for (const context of executionContexts) {
      if (context.frameId === nodeFrameId) {
        UI.Context.Context.instance().setFlavor(SDK.RuntimeModel.ExecutionContext, context);
        break;
      }
    }
  }
  documentUpdatedEvent(event) {
    const domModel = event.data;
    this.documentUpdated(domModel);
    this.removeStyleTracking(domModel.cssModel());
    this.setupStyleTracking(domModel.cssModel());
  }
  documentUpdated(domModel) {
    this.#searchableView.cancelSearch();
    if (!domModel.existingDocument()) {
      if (this.isShowing()) {
        void domModel.requestDocument();
      }
      return;
    }
    this.hasNonDefaultSelectedNode = false;
    if (this.omitDefaultSelection) {
      return;
    }
    const savedSelectedNodeOnReset = this.selectedNodeOnReset;
    void restoreNode.call(this, domModel, this.selectedNodeOnReset || null);
    async function restoreNode(domModel2, staleNode) {
      const nodePath = staleNode ? staleNode.path() : null;
      const restoredNodeId = nodePath ? await domModel2.pushNodeByPathToFrontend(nodePath) : null;
      if (savedSelectedNodeOnReset !== this.selectedNodeOnReset) {
        return;
      }
      let node = domModel2.nodeForId(restoredNodeId);
      if (!node) {
        const inspectedDocument = domModel2.existingDocument();
        node = inspectedDocument ? inspectedDocument.body || inspectedDocument.documentElement : null;
      }
      if (node) {
        this.setDefaultSelectedNode(node);
        this.lastSelectedNodeSelectedForTest();
      }
    }
  }
  lastSelectedNodeSelectedForTest() {
  }
  setDefaultSelectedNode(node) {
    if (!node || this.hasNonDefaultSelectedNode || this.pendingNodeReveal) {
      return;
    }
    this.selectDOMNode(node);
    this.#domTreeWidget.expand();
  }
  onSearchClosed() {
    const selectedNode = this.selectedDOMNode();
    if (!selectedNode) {
      return;
    }
    this.#domTreeWidget.selectDOMNodeWithoutReveal(selectedNode);
  }
  onSearchCanceled() {
    this.searchConfig = void 0;
    this.hideSearchHighlights();
    this.#searchableView.updateSearchMatchesCount(0);
    this.currentSearchResultIndex = -1;
    delete this.searchResults;
    SDK.DOMModel.DOMModel.cancelSearch();
  }
  performSearch(searchConfig, shouldJump, jumpBackwards) {
    const query = searchConfig.query;
    const whitespaceTrimmedQuery = query.trim();
    if (!whitespaceTrimmedQuery.length) {
      return;
    }
    if (!this.searchConfig || this.searchConfig.query !== query) {
      this.onSearchCanceled();
    } else {
      this.hideSearchHighlights();
    }
    this.searchConfig = searchConfig;
    const showUAShadowDOM = Common.Settings.Settings.instance().moduleSetting("show-ua-shadow-dom").get();
    const domModels = SDK.TargetManager.TargetManager.instance().models(SDK.DOMModel.DOMModel, { scoped: true });
    const promises = domModels.map((domModel) => domModel.performSearch(whitespaceTrimmedQuery, showUAShadowDOM));
    void Promise.all(promises).then((resultCounts) => {
      this.searchResults = [];
      for (let i = 0; i < resultCounts.length; ++i) {
        const resultCount = resultCounts[i];
        for (let j = 0; j < resultCount; ++j) {
          this.searchResults.push({ domModel: domModels[i], index: j, node: void 0 });
        }
      }
      this.#searchableView.updateSearchMatchesCount(this.searchResults.length);
      if (!this.searchResults.length) {
        return;
      }
      if (this.currentSearchResultIndex >= this.searchResults.length) {
        this.currentSearchResultIndex = -1;
      }
      let index = this.currentSearchResultIndex;
      if (shouldJump) {
        if (this.currentSearchResultIndex === -1) {
          index = jumpBackwards ? -1 : 0;
        } else {
          index = jumpBackwards ? index - 1 : index + 1;
        }
        this.jumpToSearchResult(index);
      }
    });
  }
  domWordWrapSettingChanged(event) {
    this.domTreeContainer.classList.toggle("elements-wrap", event.data);
    this.#domTreeWidget.setWordWrap(event.data);
  }
  jumpToSearchResult(index) {
    if (!this.searchResults) {
      return;
    }
    this.currentSearchResultIndex = (index + this.searchResults.length) % this.searchResults.length;
    this.highlightCurrentSearchResult();
  }
  jumpToNextSearchResult() {
    if (!this.searchResults || !this.searchConfig) {
      return;
    }
    this.performSearch(this.searchConfig, true);
  }
  jumpToPreviousSearchResult() {
    if (!this.searchResults || !this.searchConfig) {
      return;
    }
    this.performSearch(this.searchConfig, true, true);
  }
  supportsCaseSensitiveSearch() {
    return false;
  }
  supportsWholeWordSearch() {
    return false;
  }
  supportsRegexSearch() {
    return false;
  }
  highlightCurrentSearchResult() {
    const index = this.currentSearchResultIndex;
    const searchResults = this.searchResults;
    if (!searchResults) {
      return;
    }
    const searchResult = searchResults[index];
    this.#searchableView.updateCurrentMatchIndex(index);
    if (searchResult.node === null) {
      return;
    }
    if (typeof searchResult.node === "undefined") {
      void searchResult.domModel.searchResult(searchResult.index).then((node) => {
        searchResult.node = node;
        const highlightRequestValid = this.searchConfig && this.searchResults && this.currentSearchResultIndex !== -1;
        if (highlightRequestValid) {
          this.highlightCurrentSearchResult();
        }
      });
      return;
    }
    void searchResult.node.scrollIntoView();
    if (searchResult.node) {
      this.#domTreeWidget.highlightMatch(searchResult.node, this.searchConfig?.query);
    }
  }
  hideSearchHighlights() {
    if (!this.searchResults?.length || this.currentSearchResultIndex === -1) {
      return;
    }
    const searchResult = this.searchResults[this.currentSearchResultIndex];
    if (!searchResult.node) {
      return;
    }
    this.#domTreeWidget.hideMatchHighlights(searchResult.node);
  }
  selectedDOMNode() {
    return this.#domTreeWidget.selectedDOMNode();
  }
  selectDOMNode(node, focus2) {
    this.#domTreeWidget.selectDOMNode(node, focus2);
  }
  highlightNodeAttribute(node, attribute) {
    this.#domTreeWidget.highlightNodeAttribute(node, attribute);
  }
  selectAndShowSidebarTab(tabId) {
    if (!this.sidebarPaneView) {
      return;
    }
    this.sidebarPaneView.tabbedPane().selectTab(tabId);
    if (!this.isShowing()) {
      void UI.ViewManager.ViewManager.instance().showView("elements");
    }
  }
  updateBreadcrumbIfNeeded(event) {
    const nodes = event.data;
    const selectedNode = this.selectedDOMNode();
    if (!selectedNode) {
      this.breadcrumbs.data = {
        crumbs: [],
        selectedNode: null
      };
      return;
    }
    const activeNode = ElementsComponents.Helper.legacyNodeToElementsComponentsNode(selectedNode);
    const existingCrumbs = [activeNode];
    for (let current = selectedNode.parentNode; current; current = current.parentNode) {
      existingCrumbs.push(ElementsComponents.Helper.legacyNodeToElementsComponentsNode(current));
    }
    const newNodes = nodes.map(ElementsComponents.Helper.legacyNodeToElementsComponentsNode);
    const nodesThatHaveChangedMap = /* @__PURE__ */ new Map();
    newNodes.forEach((crumb) => nodesThatHaveChangedMap.set(crumb.id, crumb));
    const newSetOfCrumbs = existingCrumbs.map((crumb) => {
      const replacement = nodesThatHaveChangedMap.get(crumb.id);
      return replacement || crumb;
    });
    this.breadcrumbs.data = {
      crumbs: newSetOfCrumbs,
      selectedNode: activeNode
    };
  }
  crumbNodeSelected(event) {
    this.selectDOMNode(event.legacyDomNode, true);
  }
  leaveUserAgentShadowDOM(node) {
    let userAgentShadowRoot;
    while ((userAgentShadowRoot = node.ancestorUserAgentShadowRoot()) && userAgentShadowRoot.parentNode) {
      node = userAgentShadowRoot.parentNode;
    }
    return node;
  }
  async revealAndSelectNode(nodeToReveal, opts) {
    const { showPanel = true, focusNode = false, highlightInOverlay = true } = opts ?? {};
    this.omitDefaultSelection = true;
    const node = Common.Settings.Settings.instance().moduleSetting("show-ua-shadow-dom").get() ? nodeToReveal : this.leaveUserAgentShadowDOM(nodeToReveal);
    if (highlightInOverlay) {
      node.highlightForTwoSeconds();
    }
    if (this.accessibilityTreeView) {
      void this.accessibilityTreeView.revealAndSelectNode(nodeToReveal);
    }
    if (showPanel) {
      await UI.ViewManager.ViewManager.instance().showView("elements", false, !focus);
    }
    this.selectDOMNode(node, focusNode);
    delete this.omitDefaultSelection;
    if (!this.notFirstInspectElement) {
      ElementsPanel.firstInspectElementNodeNameForTest = node.nodeName();
      ElementsPanel.firstInspectElementCompletedForTest();
      Host.InspectorFrontendHost.InspectorFrontendHostInstance.inspectElementCompleted();
    }
    this.notFirstInspectElement = true;
  }
  showUAShadowDOMChanged() {
    this.#domTreeWidget.reload();
  }
  setupTextSelectionHack(stylePaneWrapperElement) {
    const uninstallHackBound = uninstallHack.bind(this);
    const uninstallHackOnMousemove = (event) => {
      if (event.buttons === 0) {
        uninstallHack.call(this);
      }
    };
    stylePaneWrapperElement.addEventListener("mousedown", (event) => {
      if (event.button !== 0) {
        return;
      }
      this.splitWidget.element.classList.add("disable-resizer-for-elements-hack");
      stylePaneWrapperElement.style.setProperty("height", `${stylePaneWrapperElement.offsetHeight}px`);
      const largeLength = 1e6;
      stylePaneWrapperElement.style.setProperty("left", `${-1 * largeLength}px`);
      stylePaneWrapperElement.style.setProperty("padding-left", `${largeLength}px`);
      stylePaneWrapperElement.style.setProperty("width", `calc(100% + ${largeLength}px)`);
      stylePaneWrapperElement.style.setProperty("position", "fixed");
      stylePaneWrapperElement.window().addEventListener("blur", uninstallHackBound);
      stylePaneWrapperElement.window().addEventListener("contextmenu", uninstallHackBound, true);
      stylePaneWrapperElement.window().addEventListener("dragstart", uninstallHackBound, true);
      stylePaneWrapperElement.window().addEventListener("mousemove", uninstallHackOnMousemove, true);
      stylePaneWrapperElement.window().addEventListener("mouseup", uninstallHackBound, true);
      stylePaneWrapperElement.window().addEventListener("visibilitychange", uninstallHackBound);
    }, true);
    function uninstallHack() {
      this.splitWidget.element.classList.remove("disable-resizer-for-elements-hack");
      stylePaneWrapperElement.style.removeProperty("left");
      stylePaneWrapperElement.style.removeProperty("padding-left");
      stylePaneWrapperElement.style.removeProperty("width");
      stylePaneWrapperElement.style.removeProperty("position");
      stylePaneWrapperElement.window().removeEventListener("blur", uninstallHackBound);
      stylePaneWrapperElement.window().removeEventListener("contextmenu", uninstallHackBound, true);
      stylePaneWrapperElement.window().removeEventListener("dragstart", uninstallHackBound, true);
      stylePaneWrapperElement.window().removeEventListener("mousemove", uninstallHackOnMousemove, true);
      stylePaneWrapperElement.window().removeEventListener("mouseup", uninstallHackBound, true);
      stylePaneWrapperElement.window().removeEventListener("visibilitychange", uninstallHackBound);
    }
  }
  initializeSidebarPanes(splitMode) {
    this.splitWidget.setVertical(splitMode === "Vertical" /* VERTICAL */);
    this.showToolbarPane(
      null,
      null
      /* toggle */
    );
    const matchedStylePanesWrapper = new UI.Widget.VBox();
    matchedStylePanesWrapper.element.classList.add("style-panes-wrapper");
    matchedStylePanesWrapper.element.setAttribute("jslog", `${VisualLogging.pane("styles").track({ resize: true })}`);
    this.stylesWidget.show(matchedStylePanesWrapper.element);
    this.setupTextSelectionHack(matchedStylePanesWrapper.element);
    const computedStylePanesWrapper = new UI.Widget.VBox();
    computedStylePanesWrapper.element.classList.add("style-panes-wrapper");
    computedStylePanesWrapper.element.setAttribute("jslog", `${VisualLogging.pane("computed").track({ resize: true })}`);
    this.computedStyleWidget.show(computedStylePanesWrapper.element);
    const stylesSplitWidget = new UI.SplitWidget.SplitWidget(
      true,
      true,
      "elements.styles.sidebar.width",
      100
    );
    stylesSplitWidget.setMainWidget(matchedStylePanesWrapper);
    stylesSplitWidget.hideSidebar();
    stylesSplitWidget.enableShowModeSaving();
    stylesSplitWidget.addEventListener(UI.SplitWidget.Events.SHOW_MODE_CHANGED, () => {
      showMetricsWidgetInStylesPane();
    });
    this.stylesWidget.addEventListener(StylesSidebarPaneEvents.INITIAL_UPDATE_COMPLETED, () => {
      this.stylesWidget.appendToolbarItem(stylesSplitWidget.createShowHideSidebarButton(
        i18nString(UIStrings.showComputedStylesSidebar),
        i18nString(UIStrings.hideComputedStylesSidebar),
        i18nString(UIStrings.computedStylesShown),
        i18nString(UIStrings.computedStylesHidden),
        "computed-styles"
      ));
    });
    const showMetricsWidgetInComputedPane = () => {
      this.metricsWidget.show(computedStylePanesWrapper.element, this.computedStyleWidget.element);
      this.metricsWidget.toggleVisibility(
        true
        /* visible */
      );
      this.stylesWidget.removeEventListener(StylesSidebarPaneEvents.STYLES_UPDATE_COMPLETED, toggleMetricsWidget);
    };
    const showMetricsWidgetInStylesPane = () => {
      const showMergedComputedPane = stylesSplitWidget.showMode() === UI.SplitWidget.ShowMode.BOTH;
      if (showMergedComputedPane) {
        showMetricsWidgetInComputedPane();
      } else {
        this.metricsWidget.show(matchedStylePanesWrapper.element);
        if (!this.stylesWidget.hasMatchedStyles) {
          this.metricsWidget.toggleVisibility(
            false
            /* invisible */
          );
        }
        this.stylesWidget.addEventListener(StylesSidebarPaneEvents.STYLES_UPDATE_COMPLETED, toggleMetricsWidget);
      }
    };
    const toggleMetricsWidget = (event) => {
      this.metricsWidget.toggleVisibility(event.data.hasMatchedStyles);
    };
    const tabSelected = (event) => {
      const { tabId } = event.data;
      if (tabId === "computed" /* COMPUTED */) {
        computedStylePanesWrapper.show(computedView.element);
        showMetricsWidgetInComputedPane();
      } else if (tabId === "styles" /* STYLES */) {
        stylesSplitWidget.setSidebarWidget(computedStylePanesWrapper);
        showMetricsWidgetInStylesPane();
      }
    };
    this.sidebarPaneView = UI.ViewManager.ViewManager.instance().createTabbedLocation(
      () => UI.ViewManager.ViewManager.instance().showView("elements"),
      "styles-pane-sidebar",
      true,
      true
    );
    const tabbedPane = this.sidebarPaneView.tabbedPane();
    tabbedPane.headerElement().setAttribute(
      "jslog",
      `${VisualLogging.toolbar("sidebar").track({ keydown: "ArrowUp|ArrowLeft|ArrowDown|ArrowRight|Enter|Space" })}`
    );
    if (this.splitMode !== "Vertical" /* VERTICAL */) {
      this.splitWidget.installResizer(tabbedPane.headerElement());
    }
    const headerElement = tabbedPane.headerElement();
    UI.ARIAUtils.markAsNavigation(headerElement);
    UI.ARIAUtils.setLabel(headerElement, i18nString(UIStrings.sidePanelToolbar));
    const contentElement = tabbedPane.tabbedPaneContentElement();
    UI.ARIAUtils.markAsComplementary(contentElement);
    UI.ARIAUtils.setLabel(contentElement, i18nString(UIStrings.sidePanelContent));
    const stylesView = new UI.View.SimpleView({
      title: i18nString(UIStrings.styles),
      viewId: "styles" /* STYLES */
    });
    this.sidebarPaneView.appendView(stylesView);
    stylesView.element.classList.add("flex-auto");
    stylesSplitWidget.show(stylesView.element);
    const computedView = new UI.View.SimpleView({
      title: i18nString(UIStrings.computed),
      viewId: "computed" /* COMPUTED */
    });
    computedView.element.classList.add("composite", "fill");
    tabbedPane.addEventListener(UI.TabbedPane.Events.TabSelected, tabSelected, this);
    this.sidebarPaneView.appendView(computedView);
    this.stylesViewToReveal = stylesView;
    this.sidebarPaneView.appendApplicableItems("elements-sidebar");
    const extensionSidebarPanes = PanelCommon.ExtensionServer.ExtensionServer.instance().sidebarPanes();
    for (let i = 0; i < extensionSidebarPanes.length; ++i) {
      this.addExtensionSidebarPane(extensionSidebarPanes[i]);
    }
    this.splitWidget.setSidebarWidget(this.sidebarPaneView.tabbedPane());
  }
  updateSidebarPosition() {
    if (this.sidebarPaneView?.tabbedPane().shouldHideOnDetach()) {
      return;
    }
    const position = Common.Settings.Settings.instance().moduleSetting("sidebar-position").get();
    let splitMode = "Horizontal" /* HORIZONTAL */;
    if (position === "right" || position === "auto" && this.splitWidget.element.offsetWidth > 680) {
      splitMode = "Vertical" /* VERTICAL */;
    }
    if (!this.sidebarPaneView) {
      this.initializeSidebarPanes(splitMode);
      return;
    }
    if (splitMode === this.splitMode) {
      return;
    }
    this.splitMode = splitMode;
    const tabbedPane = this.sidebarPaneView.tabbedPane();
    this.splitWidget.uninstallResizer(tabbedPane.headerElement());
    this.splitWidget.setVertical(this.splitMode === "Vertical" /* VERTICAL */);
    this.showToolbarPane(
      null,
      null
      /* toggle */
    );
    if (this.splitMode !== "Vertical" /* VERTICAL */) {
      this.splitWidget.installResizer(tabbedPane.headerElement());
    }
  }
  extensionSidebarPaneAdded(event) {
    this.addExtensionSidebarPane(event.data);
  }
  addExtensionSidebarPane(pane) {
    if (this.sidebarPaneView && pane.panelName() === this.name) {
      this.sidebarPaneView.appendView(pane);
    }
  }
  getComputedStyleWidget() {
    return this.computedStyleWidget;
  }
  setupStyleTracking(cssModel) {
    const cssPropertyTracker = cssModel.createCSSPropertyTracker(TrackedCSSProperties);
    cssPropertyTracker.start();
    this.cssStyleTrackerByCSSModel.set(cssModel, cssPropertyTracker);
    cssPropertyTracker.addEventListener(
      SDK.CSSModel.CSSPropertyTrackerEvents.TRACKED_CSS_PROPERTIES_UPDATED,
      this.trackedCSSPropertiesUpdated,
      this
    );
  }
  removeStyleTracking(cssModel) {
    const cssPropertyTracker = this.cssStyleTrackerByCSSModel.get(cssModel);
    if (!cssPropertyTracker) {
      return;
    }
    cssPropertyTracker.stop();
    this.cssStyleTrackerByCSSModel.delete(cssModel);
    cssPropertyTracker.removeEventListener(
      SDK.CSSModel.CSSPropertyTrackerEvents.TRACKED_CSS_PROPERTIES_UPDATED,
      this.trackedCSSPropertiesUpdated,
      this
    );
  }
  trackedCSSPropertiesUpdated({ data: domNodes }) {
    for (const domNode of domNodes) {
      if (!domNode) {
        continue;
      }
      this.#domTreeWidget.updateNodeAdorners(domNode);
    }
    LayoutPane.instance().requestUpdate();
  }
  populateAdornerSettingsContextMenu(contextMenu) {
    const adornerSubMenu = contextMenu.viewSection().appendSubMenuItem(
      i18nString(UIStrings.adornerSettings),
      false,
      "show-adorner-settings"
    );
    const adornerSettings = this.adornerManager.getSettings();
    for (const [adorner, isEnabled] of adornerSettings) {
      adornerSubMenu.defaultSection().appendCheckboxItem(adorner, () => {
        const updatedIsEnabled = !isEnabled;
        const adornersToUpdate = this.adornersByName.get(adorner);
        if (adornersToUpdate) {
          for (const adornerToUpdate of adornersToUpdate) {
            updatedIsEnabled ? adornerToUpdate.show() : adornerToUpdate.hide();
          }
        }
        this.adornerManager.getSettings().set(adorner, updatedIsEnabled);
        this.adornerManager.updateSettings(adornerSettings);
      }, { checked: isEnabled, jslogContext: adorner });
    }
  }
  isAdornerEnabled(adornerText) {
    return this.adornerManager.isAdornerEnabled(adornerText);
  }
  registerAdorner(adorner) {
    let adornerSet = this.adornersByName.get(adorner.name);
    if (!adornerSet) {
      adornerSet = /* @__PURE__ */ new Set();
      this.adornersByName.set(adorner.name, adornerSet);
    }
    adornerSet.add(adorner);
    if (!this.isAdornerEnabled(adorner.name)) {
      adorner.hide();
    }
  }
  deregisterAdorner(adorner) {
    const adornerSet = this.adornersByName.get(adorner.name);
    if (!adornerSet) {
      return;
    }
    adornerSet.delete(adorner);
  }
  toggleHideElement(node) {
    this.#domTreeWidget.toggleHideElement(node);
  }
  toggleEditAsHTML(node) {
    this.#domTreeWidget.toggleEditAsHTML(node);
  }
  duplicateNode(node) {
    this.#domTreeWidget.duplicateNode(node);
  }
  copyStyles(node) {
    this.#domTreeWidget.copyStyles(node);
  }
  static firstInspectElementCompletedForTest = function() {
  };
  static firstInspectElementNodeNameForTest = "";
}
globalThis.Elements = globalThis.Elements || {};
globalThis.Elements.ElementsPanel = ElementsPanel;
var SplitMode = /* @__PURE__ */ ((SplitMode2) => {
  SplitMode2["VERTICAL"] = "Vertical";
  SplitMode2["HORIZONTAL"] = "Horizontal";
  return SplitMode2;
})(SplitMode || {});
const TrackedCSSProperties = [
  {
    name: "display",
    value: "grid"
  },
  {
    name: "display",
    value: "inline-grid"
  },
  {
    name: "display",
    value: "flex"
  },
  {
    name: "display",
    value: "inline-flex"
  },
  {
    name: "container-type",
    value: "inline-size"
  },
  {
    name: "container-type",
    value: "block-size"
  },
  {
    name: "container-type",
    value: "size"
  }
];
export class ContextMenuProvider {
  appendApplicableItems(event, contextMenu, object) {
    if (object instanceof SDK.RemoteObject.RemoteObject && !object.isNode()) {
      return;
    }
    if (ElementsPanel.instance().element.isAncestor(event.target)) {
      return;
    }
    contextMenu.revealSection().appendItem(
      i18nString(UIStrings.openInElementsPanel),
      () => Common.Revealer.reveal(object),
      { jslogContext: "elements.reveal-node" }
    );
  }
}
export class DOMNodeRevealer {
  reveal(node, omitFocus) {
    const panel = ElementsPanel.instance();
    panel.pendingNodeReveal = true;
    return new Promise(revealPromise).catch((reason) => {
      let message;
      if (Platform.UserVisibleError.isUserVisibleError(reason)) {
        message = reason.message;
      } else {
        message = i18nString(UIStrings.nodeCannotBeFoundInTheCurrent);
      }
      Common.Console.Console.instance().warn(message);
      throw reason;
    });
    function revealPromise(resolve, reject) {
      if (node instanceof SDK.DOMModel.DOMNode) {
        onNodeResolved(node);
      } else if (node instanceof SDK.DOMModel.DeferredDOMNode) {
        node.resolve(checkDeferredDOMNodeThenReveal);
      } else {
        const domModel = node.runtimeModel().target().model(SDK.DOMModel.DOMModel);
        if (domModel) {
          void domModel.pushObjectAsNodeToFrontend(node).then(checkRemoteObjectThenReveal);
        } else {
          const msg = i18nString(UIStrings.nodeCannotBeFoundInTheCurrent);
          reject(new Platform.UserVisibleError.UserVisibleError(msg));
        }
      }
      function onNodeResolved(resolvedNode) {
        panel.pendingNodeReveal = false;
        let currentNode = resolvedNode;
        while (currentNode.parentNode) {
          currentNode = currentNode.parentNode;
        }
        const isDetached = !(currentNode instanceof SDK.DOMModel.DOMDocument);
        const isDocument = node instanceof SDK.DOMModel.DOMDocument;
        if (!isDocument && isDetached) {
          const msg2 = i18nString(UIStrings.nodeCannotBeFoundInTheCurrent);
          reject(new Platform.UserVisibleError.UserVisibleError(msg2));
          return;
        }
        if (resolvedNode) {
          void panel.revealAndSelectNode(resolvedNode, { showPanel: true, focusNode: !omitFocus }).then(resolve);
          return;
        }
        const msg = i18nString(UIStrings.nodeCannotBeFoundInTheCurrent);
        reject(new Platform.UserVisibleError.UserVisibleError(msg));
      }
      function checkRemoteObjectThenReveal(resolvedNode) {
        if (!resolvedNode) {
          const msg = i18nString(UIStrings.theRemoteObjectCouldNotBe);
          reject(new Platform.UserVisibleError.UserVisibleError(msg));
          return;
        }
        onNodeResolved(resolvedNode);
      }
      function checkDeferredDOMNodeThenReveal(resolvedNode) {
        if (!resolvedNode) {
          const msg = i18nString(UIStrings.theDeferredDomNodeCouldNotBe);
          reject(new Platform.UserVisibleError.UserVisibleError(msg));
          return;
        }
        onNodeResolved(resolvedNode);
      }
    }
  }
}
export class CSSPropertyRevealer {
  reveal(property) {
    const panel = ElementsPanel.instance();
    return panel.revealProperty(property);
  }
}
export class ElementsActionDelegate {
  handleAction(context, actionId) {
    const node = context.flavor(SDK.DOMModel.DOMNode);
    if (!node) {
      return true;
    }
    switch (actionId) {
      case "elements.hide-element":
        ElementsPanel.instance().toggleHideElement(node);
        return true;
      case "elements.edit-as-html":
        ElementsPanel.instance().toggleEditAsHTML(node);
        return true;
      case "elements.duplicate-element":
        ElementsPanel.instance().duplicateNode(node);
        return true;
      case "elements.copy-styles":
        ElementsPanel.instance().copyStyles(node);
        return true;
      case "elements.undo":
        void SDK.DOMModel.DOMModelUndoStack.instance().undo();
        ElementsPanel.instance().stylesWidget.forceUpdate();
        return true;
      case "elements.redo":
        void SDK.DOMModel.DOMModelUndoStack.instance().redo();
        ElementsPanel.instance().stylesWidget.forceUpdate();
        return true;
      case "elements.toggle-a11y-tree":
        ElementsPanel.instance().toggleAccessibilityTree();
        return true;
      case "elements.toggle-word-wrap": {
        const setting = Common.Settings.Settings.instance().moduleSetting("dom-word-wrap");
        setting.set(!setting.get());
        return true;
      }
      case "elements.show-styles":
        ElementsPanel.instance().selectAndShowSidebarTab("styles" /* STYLES */);
        return true;
      case "elements.show-computed":
        ElementsPanel.instance().selectAndShowSidebarTab("computed" /* COMPUTED */);
        return true;
      case "elements.toggle-eye-dropper": {
        const colorSwatchPopoverIcon = UI.Context.Context.instance().flavor(ColorSwatchPopoverIcon);
        if (!colorSwatchPopoverIcon) {
          return false;
        }
        void colorSwatchPopoverIcon.toggleEyeDropper();
      }
    }
    return false;
  }
}
let pseudoStateMarkerDecoratorInstance;
export class PseudoStateMarkerDecorator {
  static instance(opts = { forceNew: null }) {
    const { forceNew } = opts;
    if (!pseudoStateMarkerDecoratorInstance || forceNew) {
      pseudoStateMarkerDecoratorInstance = new PseudoStateMarkerDecorator();
    }
    return pseudoStateMarkerDecoratorInstance;
  }
  decorate(node) {
    const pseudoState = node.domModel().cssModel().pseudoState(node);
    if (!pseudoState) {
      return null;
    }
    return {
      color: "--sys-color-orange-bright",
      title: i18nString(UIStrings.elementStateS, { PH1: ":" + pseudoState.join(", :") })
    };
  }
}
//# sourceMappingURL=ElementsPanel.js.map
