"use strict";
import "./SidebarSingleInsightSet.js";
import * as Trace from "../../../models/trace/trace.js";
import * as Buttons from "../../../ui/components/buttons/buttons.js";
import * as ComponentHelpers from "../../../ui/components/helpers/helpers.js";
import * as Lit from "../../../ui/lit/lit.js";
import * as Utils from "../utils/utils.js";
import * as Insights from "./insights/insights.js";
import sidebarInsightsTabStyles from "./sidebarInsightsTab.css.js";
const { html } = Lit;
export class SidebarInsightsTab extends HTMLElement {
  #shadow = this.attachShadow({ mode: "open" });
  #parsedTrace = null;
  #activeInsight = null;
  #selectedCategory = Trace.Insights.Types.InsightCategory.ALL;
  /**
   * When a trace has sets of insights, we show an accordion with each
   * set within. A set can be specific to a single navigation, or include the
   * beginning of the trace up to the first navigation.
   * You can only have one of these open at any time, and we track it via this ID.
   */
  #selectedInsightSetKey = null;
  // TODO(paulirish): add back a disconnectedCallback() to avoid memory leaks that doesn't cause b/372943062
  set parsedTrace(data) {
    if (data === this.#parsedTrace) {
      return;
    }
    this.#parsedTrace = data;
    this.#selectedInsightSetKey = null;
    if (this.#parsedTrace?.insights) {
      this.#selectedInsightSetKey = [...this.#parsedTrace.insights.keys()].at(0) ?? null;
    }
    void ComponentHelpers.ScheduledRender.scheduleRender(this, this.#render);
  }
  get activeInsight() {
    return this.#activeInsight;
  }
  set activeInsight(active) {
    if (active === this.#activeInsight) {
      return;
    }
    this.#activeInsight = active;
    if (this.#activeInsight) {
      this.#selectedInsightSetKey = this.#activeInsight.insightSetKey;
    }
    void ComponentHelpers.ScheduledRender.scheduleRender(this, this.#render);
  }
  #insightSetToggled(id) {
    this.#selectedInsightSetKey = this.#selectedInsightSetKey === id ? null : id;
    if (this.#selectedInsightSetKey !== this.#activeInsight?.insightSetKey) {
      this.dispatchEvent(new Insights.SidebarInsight.InsightDeactivated());
    }
    void ComponentHelpers.ScheduledRender.scheduleRender(this, this.#render);
  }
  #insightSetHovered(id) {
    const data = this.#parsedTrace?.insights?.get(id);
    data && this.dispatchEvent(new Insights.SidebarInsight.InsightSetHovered(data.bounds));
  }
  #insightSetUnhovered() {
    this.dispatchEvent(new Insights.SidebarInsight.InsightSetHovered());
  }
  #onZoomClick(event, id) {
    event.stopPropagation();
    const data = this.#parsedTrace?.insights?.get(id);
    if (!data) {
      return;
    }
    this.dispatchEvent(new Insights.SidebarInsight.InsightSetZoom(data.bounds));
  }
  #renderZoomButton(insightSetToggled) {
    const classes = Lit.Directives.classMap({
      "zoom-icon": true,
      active: insightSetToggled
    });
    return html`
    <div class=${classes}>
        <devtools-button .data=${{
      variant: Buttons.Button.Variant.ICON,
      iconName: "center-focus-weak",
      size: Buttons.Button.Size.SMALL
    }}
      ></devtools-button></div>`;
  }
  #renderDropdownIcon(insightSetToggled) {
    const containerClasses = Lit.Directives.classMap({
      "dropdown-icon": true,
      active: insightSetToggled
    });
    return html`
      <div class=${containerClasses}>
        <devtools-button .data=${{
      variant: Buttons.Button.Variant.ICON,
      iconName: "chevron-right",
      size: Buttons.Button.Size.SMALL
    }}
      ></devtools-button></div>
    `;
  }
  highlightActiveInsight() {
    if (!this.#activeInsight) {
      return;
    }
    const set = this.#shadow?.querySelector(
      `devtools-performance-sidebar-single-navigation[data-insight-set-key="${this.#activeInsight.insightSetKey}"]`
    );
    if (!set) {
      return;
    }
    set.highlightActiveInsight();
  }
  #render() {
    if (!this.#parsedTrace?.insights) {
      Lit.render(Lit.nothing, this.#shadow, { host: this });
      return;
    }
    const insights = this.#parsedTrace.insights;
    const hasMultipleInsightSets = insights.size > 1;
    const labels = Utils.Helpers.createUrlLabels([...insights.values()].map(({ url }) => url));
    const contents = (
      // clang-format off
      html`
      <style>${sidebarInsightsTabStyles}</style>
      <div class="insight-sets-wrapper">
        ${[...insights.values()].map(({ id, url }, index) => {
        const data = {
          insightSetKey: id,
          activeCategory: this.#selectedCategory,
          activeInsight: this.#activeInsight,
          parsedTrace: this.#parsedTrace
        };
        const contents2 = html`
            <devtools-performance-sidebar-single-navigation
              data-insight-set-key=${id}
              .data=${data}>
            </devtools-performance-sidebar-single-navigation>
          `;
        if (hasMultipleInsightSets) {
          return html`<details
              ?open=${id === this.#selectedInsightSetKey}
            >
              <summary
                @click=${() => this.#insightSetToggled(id)}
                @mouseenter=${() => this.#insightSetHovered(id)}
                @mouseleave=${() => this.#insightSetUnhovered()}
                title=${url.href}>
                ${this.#renderDropdownIcon(id === this.#selectedInsightSetKey)}
                <span>${labels[index]}</span>
                <span class='zoom-button' @click=${(event) => this.#onZoomClick(event, id)}>${this.#renderZoomButton(id === this.#selectedInsightSetKey)}</span>
              </summary>
              ${contents2}
            </details>`;
        }
        return contents2;
      })}
      </div>
    `
    );
    const result = Lit.Directives.repeat([contents], () => this.#parsedTrace, (template) => template);
    Lit.render(result, this.#shadow, { host: this });
  }
}
customElements.define("devtools-performance-sidebar-insights", SidebarInsightsTab);
//# sourceMappingURL=SidebarInsightsTab.js.map
