"use strict";
import {
  buildPath,
  createPathForQuad,
  drawPathWithLineStyle,
  emptyBounds,
  fillPathWithBoxStyle,
  hatchFillPath
} from "./highlight_common.js";
const ALIGNMENT_LINE_THICKNESS = 2;
const ALIGNMENT_ARROW_BODY_HEIGHT = 5;
const ALIGNMENT_ARROW_BODY_WIDTH = 5;
const ALIGNMENT_ARROW_TIP_HEIGHT = 6;
const ALIGNMENT_ARROW_TIP_WIDTH = 11;
const ALIGNMENT_ARROW_DISTANCE_FROM_LINE = 2;
const FLEXIBILITY_ARROW_THICKNESS = 1;
const FLEXIBILITY_ARROW_TIP_SIZE = 5;
export function drawLayoutFlexItemHighlight(highlight, itemPath, context, emulationScaleFactor) {
  const { baseSize, isHorizontalFlow } = highlight;
  const itemQuad = rectPathToQuad(itemPath);
  const baseSizeQuad = isHorizontalFlow ? {
    p1: itemQuad.p1,
    p2: getColinearPointAtDistance(itemQuad.p1, itemQuad.p2, baseSize),
    p3: getColinearPointAtDistance(itemQuad.p4, itemQuad.p3, baseSize),
    p4: itemQuad.p4
  } : {
    p1: itemQuad.p1,
    p2: itemQuad.p2,
    p3: getColinearPointAtDistance(itemQuad.p2, itemQuad.p3, baseSize),
    p4: getColinearPointAtDistance(itemQuad.p1, itemQuad.p4, baseSize)
  };
  drawItemBaseSize(highlight, itemQuad, baseSizeQuad, context, emulationScaleFactor);
  drawFlexibilityArrow(highlight, itemQuad, baseSizeQuad, context, emulationScaleFactor);
}
function drawItemBaseSize(highlight, itemQuad, baseSizeQuad, context, emulationScaleFactor) {
  const config = highlight.flexItemHighlightConfig;
  const bounds = emptyBounds();
  const path = buildPath(quadToPath(baseSizeQuad), bounds, emulationScaleFactor);
  const angle = Math.atan2(itemQuad.p4.y - itemQuad.p1.y, itemQuad.p4.x - itemQuad.p1.x) + Math.PI * 45 / 180;
  fillPathWithBoxStyle(context, path, bounds, angle, config.baseSizeBox);
  drawPathWithLineStyle(context, path, config.baseSizeBorder);
}
function drawFlexibilityArrow(highlight, itemQuad, baseSizeQuad, context, emulationScaleFactor) {
  const { isHorizontalFlow } = highlight;
  const config = highlight.flexItemHighlightConfig;
  if (!config.flexibilityArrow) {
    return;
  }
  const from = isHorizontalFlow ? {
    x: (baseSizeQuad.p2.x + baseSizeQuad.p3.x) / 2,
    y: (baseSizeQuad.p2.y + baseSizeQuad.p3.y) / 2
  } : {
    x: (baseSizeQuad.p4.x + baseSizeQuad.p3.x) / 2,
    y: (baseSizeQuad.p4.y + baseSizeQuad.p3.y) / 2
  };
  const to = isHorizontalFlow ? {
    x: (itemQuad.p2.x + itemQuad.p3.x) / 2,
    y: (itemQuad.p2.y + itemQuad.p3.y) / 2
  } : {
    x: (itemQuad.p4.x + itemQuad.p3.x) / 2,
    y: (itemQuad.p4.y + itemQuad.p3.y) / 2
  };
  if (to.x === from.x && to.y === from.y) {
    return;
  }
  const path = segmentToPath([from, to]);
  drawPathWithLineStyle(
    context,
    buildPath(path, emptyBounds(), emulationScaleFactor),
    config.flexibilityArrow,
    FLEXIBILITY_ARROW_THICKNESS
  );
  if (!config.flexibilityArrow.color) {
    return;
  }
  const tipPath = buildPath(
    [
      "M",
      to.x - FLEXIBILITY_ARROW_TIP_SIZE,
      to.y - FLEXIBILITY_ARROW_TIP_SIZE,
      "L",
      to.x,
      to.y,
      "L",
      to.x - FLEXIBILITY_ARROW_TIP_SIZE,
      to.y + FLEXIBILITY_ARROW_TIP_SIZE
    ],
    emptyBounds(),
    emulationScaleFactor
  );
  const angle = Math.atan2(to.y - from.y, to.x - from.x);
  context.save();
  context.translate(to.x + 0.5, to.y + 0.5);
  context.rotate(angle);
  context.translate(-to.x - 0.5, -to.y - 0.5);
  drawPathWithLineStyle(context, tipPath, config.flexibilityArrow, FLEXIBILITY_ARROW_THICKNESS);
  context.restore();
}
export function drawLayoutFlexContainerHighlight(highlight, context, emulationScaleFactor) {
  const config = highlight.flexContainerHighlightConfig;
  const bounds = emptyBounds();
  const borderPath = buildPath(highlight.containerBorder, bounds, emulationScaleFactor);
  const { isHorizontalFlow, isReverse, lines } = highlight;
  drawPathWithLineStyle(context, borderPath, config.containerBorder);
  if (!lines?.length) {
    return;
  }
  const lineQuads = getLinesAndItemsQuads(highlight.containerBorder, lines, isHorizontalFlow, isReverse);
  drawFlexLinesAndItems(highlight, context, emulationScaleFactor, lineQuads, isHorizontalFlow);
  drawFlexSpace(highlight, context, emulationScaleFactor, highlight.containerBorder, lineQuads);
  drawFlexAlignment(
    highlight,
    context,
    emulationScaleFactor,
    lineQuads,
    lines.map((line) => line.map((item) => item.baseline))
  );
}
function drawFlexLinesAndItems(highlight, context, emulationScaleFactor, lineQuads, isHorizontalFlow) {
  const config = highlight.flexContainerHighlightConfig;
  const paths = lineQuads.map((line, lineIndex) => {
    const nextLineQuad = lineQuads[lineIndex + 1]?.quad;
    return {
      path: isHorizontalFlow ? quadToHorizontalLinesPath(line.quad, nextLineQuad) : quadToVerticalLinesPath(line.quad, nextLineQuad),
      items: line.extendedItems.map((item, itemIndex) => {
        const nextItemQuad = line.extendedItems[itemIndex + 1] && line.extendedItems[itemIndex + 1];
        return isHorizontalFlow ? quadToVerticalLinesPath(item, nextItemQuad) : quadToHorizontalLinesPath(item, nextItemQuad);
      })
    };
  });
  const drawLines = paths.length > 1;
  for (const { path, items } of paths) {
    for (const itemPath of items) {
      drawPathWithLineStyle(context, buildPath(itemPath, emptyBounds(), emulationScaleFactor), config.itemSeparator);
    }
    if (drawLines) {
      drawPathWithLineStyle(context, buildPath(path, emptyBounds(), emulationScaleFactor), config.lineSeparator);
    }
  }
}
function drawFlexSpace(highlight, context, emulationScaleFactor, container, lineQuads) {
  const { isHorizontalFlow } = highlight;
  const { mainDistributedSpace, crossDistributedSpace, rowGapSpace, columnGapSpace } = highlight.flexContainerHighlightConfig;
  const mainGapSpace = isHorizontalFlow ? columnGapSpace : rowGapSpace;
  const crossGapSpace = isHorizontalFlow ? rowGapSpace : columnGapSpace;
  const drawMainSpace = mainDistributedSpace && Boolean(mainDistributedSpace.fillColor || mainDistributedSpace.hatchColor);
  const drawCrossSpace = lineQuads.length > 1 && crossDistributedSpace && Boolean(crossDistributedSpace.fillColor || crossDistributedSpace.hatchColor);
  const drawMainGapSpace = mainGapSpace && Boolean(mainGapSpace.fillColor || mainGapSpace.hatchColor);
  const drawCrossGapSpace = lineQuads.length > 1 && crossGapSpace && Boolean(crossGapSpace.fillColor || crossGapSpace.hatchColor);
  const isSameStyle = mainDistributedSpace && crossDistributedSpace && mainGapSpace && crossGapSpace && mainDistributedSpace.fillColor === crossDistributedSpace.fillColor && mainDistributedSpace.hatchColor === crossDistributedSpace.hatchColor && mainDistributedSpace.fillColor === mainGapSpace.fillColor && mainDistributedSpace.hatchColor === mainGapSpace.hatchColor && mainDistributedSpace.fillColor === crossGapSpace.fillColor && mainDistributedSpace.hatchColor === crossGapSpace.hatchColor;
  const containerQuad = rectPathToQuad(container);
  if (isSameStyle) {
    const allItemQuads = lineQuads.map((line) => line.extendedItems).flat().map((item) => item);
    drawFlexSpaceInQuad(containerQuad, allItemQuads, mainDistributedSpace, context, emulationScaleFactor);
    return;
  }
  const gapQuads = getGapQuads(highlight, lineQuads);
  if (drawCrossSpace) {
    const quadsToClip = [
      // But we clip holes where lines are.
      ...lineQuads.map((line) => line.quad),
      // And also clip holds where gaps are, if those are also drawn.
      ...drawCrossGapSpace ? gapQuads.crossGaps : []
    ];
    drawFlexSpaceInQuad(containerQuad, quadsToClip, crossDistributedSpace, context, emulationScaleFactor);
  }
  if (drawMainSpace) {
    for (const [index, line] of lineQuads.entries()) {
      const quadsToClip = [
        // But we clip holes were items on the lines are.
        ...line.extendedItems,
        // And where gaps are, if those are also drawn.
        ...drawMainGapSpace ? gapQuads.mainGaps[index] : []
      ];
      drawFlexSpaceInQuad(line.quad, quadsToClip, mainDistributedSpace, context, emulationScaleFactor);
    }
  }
  if (drawCrossGapSpace) {
    for (const quad of gapQuads.crossGaps) {
      drawFlexSpaceInQuad(quad, [], crossGapSpace, context, emulationScaleFactor);
    }
  }
  if (drawMainGapSpace) {
    for (const line of gapQuads.mainGaps) {
      for (const quad of line) {
        drawFlexSpaceInQuad(quad, [], mainGapSpace, context, emulationScaleFactor);
      }
    }
  }
}
function drawFlexAlignment(highlight, context, emulationScaleFactor, lineQuads, itemBaselines) {
  lineQuads.forEach(({ quad, items }, i) => {
    drawFlexAlignmentForLine(highlight, context, emulationScaleFactor, quad, items, itemBaselines[i]);
  });
}
function drawFlexAlignmentForLine(highlight, context, emulationScaleFactor, lineQuad, itemQuads, itemBaselines) {
  const { alignItemsStyle, isHorizontalFlow } = highlight;
  const { crossAlignment } = highlight.flexContainerHighlightConfig;
  if (!crossAlignment?.color) {
    return;
  }
  const linesToDraw = [];
  switch (alignItemsStyle) {
    case "flex-start":
      linesToDraw.push([
        isHorizontalFlow ? lineQuad.p1 : lineQuad.p4,
        isHorizontalFlow ? lineQuad.p2 : lineQuad.p1
      ]);
      break;
    case "flex-end":
      linesToDraw.push([
        isHorizontalFlow ? lineQuad.p3 : lineQuad.p2,
        isHorizontalFlow ? lineQuad.p4 : lineQuad.p3
      ]);
      break;
    case "center":
      if (isHorizontalFlow) {
        linesToDraw.push([
          {
            x: (lineQuad.p1.x + lineQuad.p4.x) / 2,
            y: (lineQuad.p1.y + lineQuad.p4.y) / 2
          },
          {
            x: (lineQuad.p2.x + lineQuad.p3.x) / 2,
            y: (lineQuad.p2.y + lineQuad.p3.y) / 2
          }
        ]);
        linesToDraw.push([
          {
            x: (lineQuad.p2.x + lineQuad.p3.x) / 2,
            y: (lineQuad.p2.y + lineQuad.p3.y) / 2
          },
          {
            x: (lineQuad.p1.x + lineQuad.p4.x) / 2,
            y: (lineQuad.p1.y + lineQuad.p4.y) / 2
          }
        ]);
      } else {
        linesToDraw.push([
          {
            x: (lineQuad.p1.x + lineQuad.p2.x) / 2,
            y: (lineQuad.p1.y + lineQuad.p2.y) / 2
          },
          {
            x: (lineQuad.p3.x + lineQuad.p4.x) / 2,
            y: (lineQuad.p3.y + lineQuad.p4.y) / 2
          }
        ]);
        linesToDraw.push([
          {
            x: (lineQuad.p3.x + lineQuad.p4.x) / 2,
            y: (lineQuad.p3.y + lineQuad.p4.y) / 2
          },
          {
            x: (lineQuad.p1.x + lineQuad.p2.x) / 2,
            y: (lineQuad.p1.y + lineQuad.p2.y) / 2
          }
        ]);
      }
      break;
    case "stretch":
    case "normal":
      linesToDraw.push([
        isHorizontalFlow ? lineQuad.p1 : lineQuad.p4,
        isHorizontalFlow ? lineQuad.p2 : lineQuad.p1
      ]);
      linesToDraw.push([
        isHorizontalFlow ? lineQuad.p3 : lineQuad.p2,
        isHorizontalFlow ? lineQuad.p4 : lineQuad.p3
      ]);
      break;
    case "baseline":
      if (isHorizontalFlow) {
        const itemQuad = itemQuads[0];
        const start = intersectSegments([itemQuad.p1, itemQuad.p2], [lineQuad.p2, lineQuad.p3]);
        const end = intersectSegments([itemQuad.p1, itemQuad.p2], [lineQuad.p1, lineQuad.p4]);
        const baseline = itemBaselines[0];
        const angle = Math.atan2(itemQuad.p4.y - itemQuad.p1.y, itemQuad.p4.x - itemQuad.p1.x);
        linesToDraw.push([
          {
            x: start.x + baseline * Math.cos(angle),
            y: start.y + baseline * Math.sin(angle)
          },
          {
            x: end.x + baseline * Math.cos(angle),
            y: end.y + baseline * Math.sin(angle)
          }
        ]);
      }
      break;
  }
  for (const points of linesToDraw) {
    const path = segmentToPath(points);
    drawPathWithLineStyle(
      context,
      buildPath(path, emptyBounds(), emulationScaleFactor),
      crossAlignment,
      ALIGNMENT_LINE_THICKNESS
    );
    drawAlignmentArrow(highlight, context, emulationScaleFactor, points[0], points[1]);
  }
}
function drawAlignmentArrow(highlight, context, emulationScaleFactor, startPoint, endPoint) {
  const { crossAlignment } = highlight.flexContainerHighlightConfig;
  if (!crossAlignment?.color) {
    return;
  }
  const angle = Math.atan2(endPoint.y - startPoint.y, endPoint.x - startPoint.x);
  const C = {
    x: -ALIGNMENT_ARROW_DISTANCE_FROM_LINE * Math.cos(angle - 0.5 * Math.PI) + (startPoint.x + endPoint.x) / 2,
    y: -ALIGNMENT_ARROW_DISTANCE_FROM_LINE * Math.sin(angle - 0.5 * Math.PI) + (startPoint.y + endPoint.y) / 2
  };
  const path = buildPath(
    [
      "M",
      C.x,
      C.y,
      "L",
      C.x + ALIGNMENT_ARROW_TIP_WIDTH / 2,
      C.y + ALIGNMENT_ARROW_TIP_HEIGHT,
      "L",
      C.x + ALIGNMENT_ARROW_BODY_WIDTH / 2,
      C.y + ALIGNMENT_ARROW_TIP_HEIGHT,
      "L",
      C.x + ALIGNMENT_ARROW_BODY_WIDTH / 2,
      C.y + ALIGNMENT_ARROW_TIP_HEIGHT + ALIGNMENT_ARROW_BODY_HEIGHT,
      "L",
      C.x - ALIGNMENT_ARROW_BODY_WIDTH / 2,
      C.y + ALIGNMENT_ARROW_TIP_HEIGHT + ALIGNMENT_ARROW_BODY_HEIGHT,
      "L",
      C.x - ALIGNMENT_ARROW_BODY_WIDTH / 2,
      C.y + ALIGNMENT_ARROW_TIP_HEIGHT,
      "L",
      C.x - ALIGNMENT_ARROW_TIP_WIDTH / 2,
      C.y + ALIGNMENT_ARROW_TIP_HEIGHT,
      "Z"
    ],
    emptyBounds(),
    emulationScaleFactor
  );
  context.save();
  context.translate(C.x, C.y);
  context.rotate(angle);
  context.translate(-C.x, -C.y);
  context.fillStyle = crossAlignment.color;
  context.fill(path);
  context.lineWidth = 1;
  context.strokeStyle = "white";
  context.stroke(path);
  context.restore();
}
function drawFlexSpaceInQuad(outerQuad, quadsToClip, boxStyle, context, emulationScaleFactor) {
  if (!boxStyle) {
    return;
  }
  if (boxStyle.fillColor) {
    const bounds = emptyBounds();
    const path = createPathForQuad(outerQuad, quadsToClip, bounds, emulationScaleFactor);
    context.fillStyle = boxStyle.fillColor;
    context.fill(path);
  }
  if (boxStyle.hatchColor) {
    const angle = Math.atan2(outerQuad.p2.y - outerQuad.p1.y, outerQuad.p2.x - outerQuad.p1.x) * 180 / Math.PI;
    const bounds = emptyBounds();
    const path = createPathForQuad(outerQuad, quadsToClip, bounds, emulationScaleFactor);
    hatchFillPath(context, path, bounds, 10, boxStyle.hatchColor, angle, false);
  }
}
export function getLinesAndItemsQuads(container, lines, isHorizontalFlow, isReverse) {
  const containerQuad = rectPathToQuad(container);
  const lineQuads = [];
  for (const line of lines) {
    if (!line.length) {
      continue;
    }
    let lineQuad = rectPathToQuad(line[0].itemBorder);
    const itemQuads = [];
    for (const { itemBorder } of line) {
      const itemQuad = rectPathToQuad(itemBorder);
      lineQuad = !lineQuad ? itemQuad : uniteQuads(lineQuad, itemQuad, isHorizontalFlow, isReverse);
      itemQuads.push(itemQuad);
    }
    const extendedLineQuad = lines.length === 1 ? containerQuad : growQuadToEdgesOf(lineQuad, containerQuad, isHorizontalFlow);
    const extendItemQuads = itemQuads.map((itemQuad) => growQuadToEdgesOf(itemQuad, extendedLineQuad, !isHorizontalFlow));
    lineQuads.push({
      quad: extendedLineQuad,
      items: itemQuads,
      extendedItems: extendItemQuads
    });
  }
  return lineQuads;
}
export function getGapQuads(highlight, lineQuads) {
  const { crossGap, mainGap, isHorizontalFlow, isReverse } = highlight;
  const mainGaps = [];
  const crossGaps = [];
  if (crossGap && lineQuads.length > 1) {
    for (let i = 0, j = i + 1; i < lineQuads.length - 1; i++, j = i + 1) {
      const line1 = lineQuads[i].quad;
      const line2 = lineQuads[j].quad;
      crossGaps.push(getGapQuadBetweenQuads(line1, line2, crossGap, isHorizontalFlow));
    }
  }
  for (const { extendedItems } of lineQuads) {
    const lineGapQuads = [];
    if (mainGap) {
      for (let i = 0, j = i + 1; i < extendedItems.length - 1; i++, j = i + 1) {
        const item1 = extendedItems[i];
        const item2 = extendedItems[j];
        lineGapQuads.push(getGapQuadBetweenQuads(item1, item2, mainGap, !isHorizontalFlow, isReverse));
      }
    }
    mainGaps.push(lineGapQuads);
  }
  return { mainGaps, crossGaps };
}
export function getGapQuadBetweenQuads(quad1, quad2, size, vertically, isReverse) {
  if (isReverse) {
    [quad1, quad2] = [quad2, quad1];
  }
  const angle = vertically ? Math.atan2(quad1.p4.y - quad1.p1.y, quad1.p4.x - quad1.p1.x) : Math.atan2(quad1.p2.y - quad1.p1.y, quad1.p2.x - quad1.p1.x);
  const d = vertically ? distance(quad1.p4, quad2.p1) : distance(quad1.p2, quad2.p1);
  const startOffset = d / 2 - size / 2;
  const endOffset = d / 2 + size / 2;
  return vertically ? {
    p1: {
      x: Math.round(quad1.p4.x + startOffset * Math.cos(angle)),
      y: Math.round(quad1.p4.y + startOffset * Math.sin(angle))
    },
    p2: {
      x: Math.round(quad1.p3.x + startOffset * Math.cos(angle)),
      y: Math.round(quad1.p3.y + startOffset * Math.sin(angle))
    },
    p3: {
      x: Math.round(quad1.p3.x + endOffset * Math.cos(angle)),
      y: Math.round(quad1.p3.y + endOffset * Math.sin(angle))
    },
    p4: {
      x: Math.round(quad1.p4.x + endOffset * Math.cos(angle)),
      y: Math.round(quad1.p4.y + endOffset * Math.sin(angle))
    }
  } : {
    p1: {
      x: Math.round(quad1.p2.x + startOffset * Math.cos(angle)),
      y: Math.round(quad1.p2.y + startOffset * Math.sin(angle))
    },
    p2: {
      x: Math.round(quad1.p2.x + endOffset * Math.cos(angle)),
      y: Math.round(quad1.p2.y + endOffset * Math.sin(angle))
    },
    p3: {
      x: Math.round(quad1.p3.x + endOffset * Math.cos(angle)),
      y: Math.round(quad1.p3.y + endOffset * Math.sin(angle))
    },
    p4: {
      x: Math.round(quad1.p3.x + startOffset * Math.cos(angle)),
      y: Math.round(quad1.p3.y + startOffset * Math.sin(angle))
    }
  };
}
function quadToHorizontalLinesPath(quad, nextQuad) {
  const skipEndLine = nextQuad && quad.p4.y === nextQuad.p1.y;
  const startLine = ["M", quad.p1.x, quad.p1.y, "L", quad.p2.x, quad.p2.y];
  return skipEndLine ? startLine : [...startLine, "M", quad.p3.x, quad.p3.y, "L", quad.p4.x, quad.p4.y];
}
function quadToVerticalLinesPath(quad, nextQuad) {
  const skipEndLine = nextQuad && quad.p2.x === nextQuad.p1.x;
  const startLine = ["M", quad.p1.x, quad.p1.y, "L", quad.p4.x, quad.p4.y];
  return skipEndLine ? startLine : [...startLine, "M", quad.p3.x, quad.p3.y, "L", quad.p2.x, quad.p2.y];
}
function quadToPath(quad) {
  return [
    "M",
    quad.p1.x,
    quad.p1.y,
    "L",
    quad.p2.x,
    quad.p2.y,
    "L",
    quad.p3.x,
    quad.p3.y,
    "L",
    quad.p4.x,
    quad.p4.y,
    "Z"
  ];
}
function segmentToPath(segment) {
  return ["M", segment[0].x, segment[0].y, "L", segment[1].x, segment[1].y];
}
function rectPathToQuad(commands) {
  return {
    p1: { x: commands[1], y: commands[2] },
    p2: { x: commands[4], y: commands[5] },
    p3: { x: commands[7], y: commands[8] },
    p4: { x: commands[10], y: commands[11] }
  };
}
export function uniteQuads(quad1, quad2, isHorizontalFlow, isReverse) {
  if (isReverse) {
    [quad1, quad2] = [quad2, quad1];
  }
  const mainStartSegment = isHorizontalFlow ? [quad1.p1, quad1.p4] : [quad1.p1, quad1.p2];
  const mainEndSegment = isHorizontalFlow ? [quad2.p2, quad2.p3] : [quad2.p4, quad2.p3];
  const crossStartSegment1 = isHorizontalFlow ? [quad1.p1, quad1.p2] : [quad1.p1, quad1.p4];
  const crossEndSegment1 = isHorizontalFlow ? [quad1.p4, quad1.p3] : [quad1.p2, quad1.p3];
  const crossStartSegment2 = isHorizontalFlow ? [quad2.p1, quad2.p2] : [quad2.p1, quad2.p4];
  const crossEndSegment2 = isHorizontalFlow ? [quad2.p4, quad2.p3] : [quad2.p2, quad2.p3];
  let p1, p2, p3, p4;
  if (isHorizontalFlow) {
    p1 = intersectSegments(mainStartSegment, crossStartSegment2);
    if (segmentContains(mainStartSegment, p1)) {
      p1 = quad1.p1;
    }
    p2 = intersectSegments(mainEndSegment, crossStartSegment1);
    if (segmentContains(mainEndSegment, p2)) {
      p2 = quad2.p2;
    }
    p3 = intersectSegments(mainEndSegment, crossEndSegment1);
    if (segmentContains(mainEndSegment, p3)) {
      p3 = quad2.p3;
    }
    p4 = intersectSegments(mainStartSegment, crossEndSegment2);
    if (segmentContains(mainStartSegment, p4)) {
      p4 = quad1.p4;
    }
  } else {
    p1 = intersectSegments(mainStartSegment, crossStartSegment2);
    if (segmentContains(mainStartSegment, p1)) {
      p1 = quad1.p1;
    }
    p2 = intersectSegments(mainStartSegment, crossEndSegment2);
    if (segmentContains(mainStartSegment, p2)) {
      p2 = quad1.p2;
    }
    p3 = intersectSegments(mainEndSegment, crossEndSegment1);
    if (segmentContains(mainEndSegment, p3)) {
      p3 = quad2.p3;
    }
    p4 = intersectSegments(mainEndSegment, crossStartSegment1);
    if (segmentContains(mainEndSegment, p4)) {
      p4 = quad2.p4;
    }
  }
  return { p1, p2, p3, p4 };
}
export function growQuadToEdgesOf(innerQuad, outerQuad, horizontally) {
  return {
    p1: horizontally ? intersectSegments([outerQuad.p1, outerQuad.p4], [innerQuad.p1, innerQuad.p2]) : intersectSegments([outerQuad.p1, outerQuad.p2], [innerQuad.p1, innerQuad.p4]),
    p2: horizontally ? intersectSegments([outerQuad.p2, outerQuad.p3], [innerQuad.p1, innerQuad.p2]) : intersectSegments([outerQuad.p1, outerQuad.p2], [innerQuad.p2, innerQuad.p3]),
    p3: horizontally ? intersectSegments([outerQuad.p2, outerQuad.p3], [innerQuad.p3, innerQuad.p4]) : intersectSegments([outerQuad.p3, outerQuad.p4], [innerQuad.p2, innerQuad.p3]),
    p4: horizontally ? intersectSegments([outerQuad.p1, outerQuad.p4], [innerQuad.p3, innerQuad.p4]) : intersectSegments([outerQuad.p3, outerQuad.p4], [innerQuad.p1, innerQuad.p4])
  };
}
export function intersectSegments([p1, p2], [p3, p4]) {
  const x = ((p1.x * p2.y - p1.y * p2.x) * (p3.x - p4.x) - (p1.x - p2.x) * (p3.x * p4.y - p3.y * p4.x)) / ((p1.x - p2.x) * (p3.y - p4.y) - (p1.y - p2.y) * (p3.x - p4.x));
  const y = ((p1.x * p2.y - p1.y * p2.x) * (p3.y - p4.y) - (p1.y - p2.y) * (p3.x * p4.y - p3.y * p4.x)) / ((p1.x - p2.x) * (p3.y - p4.y) - (p1.y - p2.y) * (p3.x - p4.x));
  return {
    x: Object.is(x, -0) ? 0 : x,
    y: Object.is(y, -0) ? 0 : y
  };
}
export function segmentContains([p1, p2], point) {
  if (p1.x < p2.x && (point.x < p1.x || point.x > p2.x)) {
    return false;
  }
  if (p1.x > p2.x && (point.x > p1.x || point.x < p2.x)) {
    return false;
  }
  if (p1.y < p2.y && (point.y < p1.y || point.y > p2.y)) {
    return false;
  }
  if (p1.y > p2.y && (point.y > p1.y || point.y < p2.y)) {
    return false;
  }
  return (point.y - p1.y) * (p2.x - p1.x) === (p2.y - p1.y) * (point.x - p1.x);
}
export function distance(p1, p2) {
  return Math.sqrt(Math.pow(p2.x - p1.x, 2) + Math.pow(p2.y - p1.y, 2));
}
export function getColinearPointAtDistance(p1, p2, distance2) {
  const slope = (p2.y - p1.y) / (p2.x - p1.x);
  const angle = Math.atan(slope);
  return {
    x: p1.x + distance2 * Math.cos(angle),
    y: p1.y + distance2 * Math.sin(angle)
  };
}
//# sourceMappingURL=highlight_flex_common.js.map
