// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import './page_favicon.js';
import '//resources/cr_elements/cr_action_menu/cr_action_menu.js';
import '//resources/cr_elements/cr_icon_button/cr_icon_button.js';
import { I18nMixinLit } from '//resources/cr_elements/i18n_mixin_lit.js';
import { assert } from '//resources/js/assert.js';
import { loadTimeData } from '//resources/js/load_time_data.js';
import { CrLitElement } from '//resources/lit/v3_0/lit.rollup.js';
import { BrowserProxyImpl } from './browser_proxy.js';
import { Annotation } from './history_cluster_types.mojom-webui.js';
import { getCss } from './url_visit.css.js';
import { getHtml } from './url_visit.html.js';
import { insertHighlightedTextWithMatchesIntoElement } from './utils.js';
/**
 * @fileoverview This file provides a custom element displaying a visit to a
 * page within a cluster. A visit features the page favicon, title, a timestamp,
 * as well as an action menu.
 */
/**
 * Maps supported annotations to localized string identifiers.
 */
const annotationToStringId = new Map([
    [Annotation.kBookmarked, 'bookmarked'],
]);
const ClusterMenuElementBase = I18nMixinLit(CrLitElement);
export class UrlVisitElement extends ClusterMenuElementBase {
    static get is() {
        return 'url-visit';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            /**
             * The current query for which related clusters are requested and shown.
             */
            query: { type: String },
            /**
             * The visit to display.
             */
            visit: { type: Object },
            /**
             * Whether this visit is within a persisted cluster.
             */
            fromPersistence: { type: Boolean },
            /**
             * Usually this is true, but this can be false if deleting history is
             * prohibited by Enterprise policy.
             */
            allowDeletingHistory_: { type: Boolean },
            /**
             * Whether the cluster is in the side panel.
             */
            inSidePanel_: {
                type: Boolean,
                reflect: true,
            },
            renderActionMenu_: { type: Boolean },
        };
    }
    #query_accessor_storage = '';
    //============================================================================
    // Properties
    //============================================================================
    get query() { return this.#query_accessor_storage; }
    set query(value) { this.#query_accessor_storage = value; }
    #visit_accessor_storage;
    get visit() { return this.#visit_accessor_storage; }
    set visit(value) { this.#visit_accessor_storage = value; }
    #fromPersistence_accessor_storage = false;
    get fromPersistence() { return this.#fromPersistence_accessor_storage; }
    set fromPersistence(value) { this.#fromPersistence_accessor_storage = value; }
    annotations_ = [];
    #allowDeletingHistory__accessor_storage = loadTimeData.getBoolean('allowDeletingHistory');
    get allowDeletingHistory_() { return this.#allowDeletingHistory__accessor_storage; }
    set allowDeletingHistory_(value) { this.#allowDeletingHistory__accessor_storage = value; }
    #inSidePanel__accessor_storage = loadTimeData.getBoolean('inSidePanel');
    get inSidePanel_() { return this.#inSidePanel__accessor_storage; }
    set inSidePanel_(value) { this.#inSidePanel__accessor_storage = value; }
    #renderActionMenu__accessor_storage = false;
    get renderActionMenu_() { return this.#renderActionMenu__accessor_storage; }
    set renderActionMenu_(value) { this.#renderActionMenu__accessor_storage = value; }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('visit')) {
            assert(this.visit);
            insertHighlightedTextWithMatchesIntoElement(this.$.title, this.visit.pageTitle, this.visit.titleMatchPositions);
            insertHighlightedTextWithMatchesIntoElement(this.$.url, this.visit.urlForDisplay, this.visit.urlForDisplayMatchPositions);
        }
    }
    //============================================================================
    // Event handlers
    //============================================================================
    onAuxClick_() {
        // Notify the parent <history-cluster> element of this event.
        this.fire('visit-clicked', this.visit);
    }
    onClick_(event) {
        // Ignore previously handled events.
        if (event.defaultPrevented) {
            return;
        }
        event.preventDefault(); // Prevent default browser action (navigation).
        // To record metrics.
        this.onAuxClick_();
        this.openUrl_(event);
    }
    onContextMenu_(event) {
        // Because WebUI has a Blink-provided context menu that's suitable, and
        // Side Panel always UIs always have a custom context menu.
        if (!loadTimeData.getBoolean('inSidePanel') || !this.visit) {
            return;
        }
        BrowserProxyImpl.getInstance().handler.showContextMenuForURL(this.visit.normalizedUrl, { x: event.clientX, y: event.clientY });
    }
    onKeydown_(e) {
        // To be consistent with <history-list>, only handle Enter, and not Space.
        if (e.key !== 'Enter') {
            return;
        }
        // To record metrics.
        this.onAuxClick_();
        this.openUrl_(e);
    }
    async onActionMenuButtonClick_(event) {
        event.preventDefault(); // Prevent default browser action (navigation).
        if (!this.renderActionMenu_) {
            this.renderActionMenu_ = true;
            await this.updateComplete;
        }
        const menu = this.shadowRoot.querySelector('cr-action-menu');
        assert(menu);
        menu.showAt(this.$.actionMenuButton);
    }
    onHideSelfButtonClick_(event) {
        this.emitMenuButtonClick_(event, 'hide-visit');
    }
    onRemoveSelfButtonClick_(event) {
        this.emitMenuButtonClick_(event, 'remove-visit');
    }
    emitMenuButtonClick_(event, emitEventName) {
        event.preventDefault(); // Prevent default browser action (navigation).
        this.fire(emitEventName, this.visit);
        // This can also be triggered from the hide visit icon, in which case the
        // menu may not be rendered.
        if (this.renderActionMenu_) {
            const menu = this.shadowRoot.querySelector('cr-action-menu');
            assert(menu);
            menu.close();
        }
    }
    //============================================================================
    // Helper methods
    //============================================================================
    computeAnnotations_() {
        // Disabling annotations until more appropriate design for annotations in
        // the side panel is complete.
        if (this.inSidePanel_ || !this.visit) {
            return [];
        }
        return this.visit.annotations
            .map((annotation) => annotationToStringId.get(annotation))
            .filter((id) => {
            return !!id;
        })
            .map((id) => loadTimeData.getString(id));
    }
    computeDebugInfo_() {
        if (!loadTimeData.getBoolean('isHistoryClustersDebug') || !this.visit) {
            return '';
        }
        return JSON.stringify(this.visit.debugInfo);
    }
    openUrl_(event) {
        assert(this.visit);
        BrowserProxyImpl.getInstance().handler.openHistoryUrl(this.visit.normalizedUrl, {
            middleButton: event.button === 1,
            altKey: event.altKey,
            ctrlKey: event.ctrlKey,
            metaKey: event.metaKey,
            shiftKey: event.shiftKey,
        });
    }
}
customElements.define(UrlVisitElement.is, UrlVisitElement);
