// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { BrowserProxyImpl } from './browser_proxy.js';
/**
 * Returns the boca app if it can be found in the DOM.
 */
function getApp() {
    const app = document.querySelector('boca-receiver-app');
    return app;
}
/**
 * Runs any initialization code on the boca app once it is in the dom.
 */
function initializeApp(app) {
    const proxy = BrowserProxyImpl.getInstance();
    proxy.callbackRouter.onInitReceiverInfo.addListener((receiverInfo) => app.onInitReceiverInfo(receiverInfo));
    proxy.callbackRouter.onInitReceiverError.addListener(() => app.onInitReceiverError());
    proxy.callbackRouter.onFrameReceived.addListener((frameData) => app.onFrameReceived(frameData));
    proxy.callbackRouter.onAudioPacket.addListener((audioPacket) => app.onAudioPacket(audioPacket));
    proxy.callbackRouter.onConnecting.addListener((initiator, presenter) => app.onConnecting(initiator, presenter));
    proxy.callbackRouter.onConnectionClosed.addListener((reason) => app.onConnectionClosed(reason));
}
/**
 * Called when a mutation occurs on document.body to check if the boca app is
 * available.
 */
function mutationCallback(_mutationsList, observer) {
    const app = getApp();
    if (!app) {
        return;
    }
    // The boca app now exists so we can initialize it.
    initializeApp(app);
    observer.disconnect();
}
window.addEventListener('DOMContentLoaded', () => {
    const app = getApp();
    if (app) {
        initializeApp(app);
        return;
    }
    // If translations need to be fetched, the app element may not be added yet.
    // In that case, observe <body> until it is.
    const observer = new MutationObserver(mutationCallback);
    observer.observe(document.body, { childList: true });
});
