// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { assert } from '../assert.js';
import { CoverPhoto } from '../cover_photo.js';
import * as dom from '../dom.js';
import { reportError } from '../error.js';
import { GalleryButton } from '../lit/components/gallery-button.js';
import { ChromeHelper } from '../mojo/chrome_helper.js';
import { ErrorLevel, ErrorType, MimeType, VideoType, } from '../type.js';
import { sleep } from '../util.js';
import { Filenamer } from './file_namer.js';
import * as filesystem from './file_system.js';
/**
 * Default handler for captured result photos and video.
 */
export class DefaultResultSaver {
    constructor() {
        /**
         * Cover photo from latest saved picture.
         */
        this.cover = null;
        /**
         * Directory holding saved pictures showing in gallery.
         */
        this.directory = null;
        this.galleryButton = dom.get('gallery-button', GalleryButton);
        this.retryingCheckCover = false;
    }
    /**
     * Initializes the result saver.
     *
     * @param dir Directory holding saved pictures showing in gallery.
     */
    async initialize(dir) {
        this.directory = dir;
        await this.checkCover();
        this.galleryButton.addEventListener('click', () => {
            if (this.cover !== null) {
                ChromeHelper.getInstance().openFileInGallery(this.cover.name);
            }
        });
    }
    /**
     * @param file File to be set as cover photo.
     */
    async updateCover(file) {
        const cover = file === null ? null : await CoverPhoto.create(file);
        if (this.cover === cover) {
            return;
        }
        if (this.cover !== null) {
            this.cover.release();
        }
        this.cover = cover;
        this.galleryButton.cover = cover;
        if (file !== null) {
            // The promise is only resolved after the file is deleted.
            void ChromeHelper.getInstance().monitorFileDeletion(file.name, async () => {
                try {
                    await this.checkCover();
                }
                catch (e) {
                    reportError(ErrorType.CHECK_COVER_FAILURE, ErrorLevel.ERROR, e);
                }
            });
        }
    }
    /**
     * Checks validity of cover photo from camera directory.
     */
    async checkCover() {
        if (this.directory === null) {
            return;
        }
        const dir = this.directory;
        // Checks existence of cached cover photo.
        if (this.cover !== null) {
            if (await dir.exists(this.cover.name)) {
                return;
            }
        }
        // Rescan file system. Only select files following CCA naming styles.
        const files = (await filesystem.getEntries())
            .filter((file) => Filenamer.isCcaFileFormat(file.name));
        if (files.length === 0) {
            await this.updateCover(null);
            return;
        }
        try {
            const filesWithTime = await Promise.all(files.map(async (file) => ({
                file,
                time: (await file.getLastModificationTime()),
            })));
            const lastFile = filesWithTime.reduce((last, cur) => last.time > cur.time ? last : cur)
                .file;
            await this.updateCover(lastFile);
        }
        catch (e) {
            // The file might be deleted at any time and cause the operation
            // interrupted. Since it might take a while when doing bulk deletion, only
            // try check cover again if the amount of files become stable.
            if (e instanceof DOMException && !this.retryingCheckCover) {
                this.retryingCheckCover = true;
                try {
                    await this.waitUntilCameraFolderStable();
                    await this.checkCover();
                }
                finally {
                    this.retryingCheckCover = false;
                }
            }
            else {
                throw e;
            }
        }
    }
    async waitUntilCameraFolderStable() {
        let prevFileCount = (await filesystem.getEntries()).length;
        while (true) {
            await sleep(500);
            const newFileCount = (await filesystem.getEntries()).length;
            if (prevFileCount === newFileCount) {
                return;
            }
            prevFileCount = newFileCount;
        }
    }
    async savePhoto(blob, name, metadata) {
        const file = await filesystem.saveBlob(blob, name);
        if (metadata !== null) {
            const metadataBlob = new Blob([JSON.stringify(metadata, null, 2)], { type: MimeType.JSON });
            await filesystem.saveBlob(metadataBlob, Filenamer.getMetadataName(name));
        }
        ChromeHelper.getInstance().sendNewCaptureBroadcast({ isVideo: false, name: file.name });
        await this.updateCover(file);
    }
    async saveGif(blob, name) {
        const file = await filesystem.saveBlob(blob, name);
        await this.updateCover(file);
    }
    async saveVideo(file) {
        const videoName = (new Filenamer()).newVideoName(VideoType.MP4);
        assert(this.directory !== null);
        await file.moveTo(this.directory, videoName);
        ChromeHelper.getInstance().sendNewCaptureBroadcast({ isVideo: true, name: file.name });
        await this.updateCover(file);
    }
}
