// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Polymer element for displaying network nameserver options.
 */
import '//resources/polymer/v3_0/iron-flex-layout/iron-flex-layout-classes.js';
import '//resources/ash/common/cr_elements/cr_input/cr_input.js';
import '//resources/ash/common/cr_elements/cr_radio_button/cr_radio_button.js';
import '//resources/ash/common/cr_elements/cr_radio_group/cr_radio_group.js';
import '//resources/ash/common/cr_elements/policy/cr_policy_indicator.js';
import '//resources/ash/common/cr_elements/md_select.css.js';
import './network_shared.css.js';
import { assert } from '//resources/js/assert.js';
import { IPConfigType } from '//resources/mojo/chromeos/services/network_config/public/mojom/network_types.mojom-webui.js';
import { mixinBehaviors, PolymerElement } from '//resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { I18nMixin } from 'chrome://resources/ash/common/cr_elements/i18n_mixin.js';
import { CrPolicyNetworkBehaviorMojo } from './cr_policy_network_behavior_mojo.js';
import { getTemplate } from './network_nameservers.html.js';
import { OncMojo } from './onc_mojo.js';
/**
 * UI configuration options for nameservers.
 */
var NameserversType;
(function (NameserversType) {
    NameserversType["AUTOMATIC"] = "automatic";
    NameserversType["CUSTOM"] = "custom";
    NameserversType["GOOGLE"] = "google";
})(NameserversType || (NameserversType = {}));
const GOOGLE_NAMESERVERS = [
    '8.8.4.4',
    '8.8.8.8',
];
const EMPTY_NAMESERVER = '0.0.0.0';
const MAX_NAMESERVERS = 4;
const NetworkNameserversElementBase = mixinBehaviors([CrPolicyNetworkBehaviorMojo], I18nMixin(PolymerElement));
export class NetworkNameserversElement extends NetworkNameserversElementBase {
    constructor() {
        super(...arguments);
        // Saved nameservers from the NameserversType.CUSTOM tab. If this is empty, it
        // that the user has not entered any custom nameservers yet.
        this.nameservers_ = [];
        this.nameserversType_ = NameserversType.AUTOMATIC;
        this.googleNameserversText_ = this.i18nAdvanced('networkNameserversGoogle', { substitutions: [], tags: ['a'] })
            .toString();
        this.savedCustomNameservers_ = [];
        // The last manually performed selection of the nameserver type. If this is
        // null, no explicit selection has been done for this network yet.
        this.savedNameserversType_ = null;
    }
    static get is() {
        return 'network-nameservers';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            disabled: {
                type: Boolean,
                value: false,
            },
            managedProperties: {
                type: Object,
                observer: 'managedPropertiesChanged_',
            },
            /**
             * Array of nameserver addresses stored as strings.
             */
            nameservers_: {
                type: Array,
            },
            /**
             * The selected nameserver type.
             */
            nameserversType_: {
                type: String,
            },
            /**
             * Enum values for |nameserversType_|.
             */
            nameserversTypeEnum_: {
                readOnly: true,
                type: Object,
                value: NameserversType,
            },
            googleNameserversText_: {
                type: String,
            },
            canChangeConfigType_: {
                type: Boolean,
                computed: 'computeCanChangeConfigType_(managedProperties)',
            },
        };
    }
    /*
     * Returns the nameserver type CrRadioGroupElement.
     */
    getNameserverRadioButtons() {
        return this.shadowRoot.querySelector('#nameserverType');
    }
    /**
     * Returns true if the nameservers in |nameservers1| match the nameservers in
     * |nameservers2|, ignoring order and empty / 0.0.0.0 entries.
     */
    nameserversMatch_(nameservers1, nameservers2) {
        const nonEmptySortedNameservers1 = this.clearEmptyNameServers_(nameservers1).sort();
        const nonEmptySortedNameservers2 = this.clearEmptyNameServers_(nameservers2).sort();
        if (nonEmptySortedNameservers1.length !==
            nonEmptySortedNameservers2.length) {
            return false;
        }
        for (let i = 0; i < nonEmptySortedNameservers1.length; i++) {
            if (nonEmptySortedNameservers1[i] !== nonEmptySortedNameservers2[i]) {
                return false;
            }
        }
        return true;
    }
    /**
     * Returns true if |nameservers| contains any all google nameserver entries
     * and only google nameserver entries or empty entries.
     */
    isGoogleNameservers_(nameservers) {
        return this.nameserversMatch_(nameservers, GOOGLE_NAMESERVERS);
    }
    /**
     * Returns the nameservers enforced by policy. If nameservers are not being
     * enforced, returns null.
     */
    getPolicyEnforcedNameservers_() {
        const staticIpConfig = this.managedProperties && this.managedProperties.staticIpConfig;
        if (!staticIpConfig || !staticIpConfig.nameServers) {
            return null;
        }
        return this.getEnforcedPolicyValue(staticIpConfig.nameServers);
    }
    /**
     * Returns the nameservers recommended by policy. If nameservers are not being
     * recommended, returns null. Note: also returns null if nameservers are being
     * enforced by policy.
     */
    getPolicyRecommendedNameservers_() {
        const staticIpConfig = this.managedProperties && this.managedProperties.staticIpConfig;
        if (!staticIpConfig || !staticIpConfig.nameServers) {
            return null;
        }
        return this.getRecommendedPolicyValue(staticIpConfig.nameServers);
    }
    managedPropertiesChanged_(newValue, oldValue) {
        if (!this.managedProperties) {
            return;
        }
        if (!oldValue || newValue.guid !== oldValue.guid) {
            this.savedCustomNameservers_ = [];
            this.savedNameserversType_ = null;
        }
        // Update the 'nameservers' property.
        let nameservers = [];
        const ipv4 = OncMojo.getIPConfigForType(this.managedProperties, IPConfigType.kIPv4);
        if (ipv4 && ipv4.nameServers) {
            nameservers = ipv4.nameServers.slice();
        }
        // Update the 'nameserversType' property.
        const configType = OncMojo.getActiveValue(this.managedProperties.nameServersConfigType);
        let nameserversType;
        if (configType === 'Static') {
            if (this.isGoogleNameservers_(nameservers) &&
                this.savedNameserversType_ !== NameserversType.CUSTOM) {
                nameserversType = NameserversType.GOOGLE;
                nameservers = GOOGLE_NAMESERVERS; // Use consistent order.
            }
            else {
                nameserversType = NameserversType.CUSTOM;
            }
        }
        else {
            nameserversType = NameserversType.AUTOMATIC;
            nameservers = this.clearEmptyNameServers_(nameservers);
        }
        // When a network is connected, we receive connection strength updates and
        // that prevents users from making any custom updates to network
        // nameservers. These below conditions allow connection strength updates to
        // be applied only if network is not connected or if nameservers type is set
        // to auto or if we are receiving the update for the first time.
        if (nameserversType !== NameserversType.CUSTOM || !oldValue ||
            newValue.guid !== (oldValue && oldValue.guid) ||
            !OncMojo.connectionStateIsConnected(this.managedProperties.connectionState)) {
            this.setNameservers_(nameserversType, nameservers, false /* send */);
        }
    }
    /**
     * @param sendNameservers If true, send the nameservers once they have been
     *     set in the UI.
     */
    setNameservers_(nameserversType, nameservers, sendNameservers) {
        if (nameserversType === NameserversType.CUSTOM) {
            // Add empty entries for unset custom nameservers.
            for (let i = nameservers.length; i < MAX_NAMESERVERS; ++i) {
                nameservers[i] = EMPTY_NAMESERVER;
            }
        }
        else {
            nameservers = this.clearEmptyNameServers_(nameservers);
        }
        this.nameservers_ = nameservers;
        this.nameserversType_ = nameserversType;
        if (sendNameservers) {
            this.sendNameServers_();
        }
    }
    /**
     * @return True if the nameservers config type type can be changed.
     */
    computeCanChangeConfigType_(managedProperties) {
        if (!managedProperties) {
            return false;
        }
        if (this.isNetworkPolicyEnforced(managedProperties.nameServersConfigType)) {
            return false;
        }
        return true;
    }
    /**
     * @return True if the nameservers are editable.
     */
    canEditCustomNameServers_(nameserversType, managedProperties) {
        if (!managedProperties) {
            return false;
        }
        if (nameserversType !== NameserversType.CUSTOM) {
            return false;
        }
        if (this.isNetworkPolicyEnforced(managedProperties.nameServersConfigType)) {
            return false;
        }
        if (managedProperties.staticIpConfig &&
            managedProperties.staticIpConfig.nameServers &&
            this.isNetworkPolicyEnforced(managedProperties.staticIpConfig.nameServers)) {
            return false;
        }
        return true;
    }
    showNameservers_(nameserversType, buttonNameserverstype, nameservers) {
        if (nameserversType !== buttonNameserverstype) {
            return false;
        }
        return buttonNameserverstype === NameserversType.CUSTOM ||
            nameservers.length > 0;
    }
    getNameserversString_(nameservers) {
        return nameservers.join(', ');
    }
    /**
     * Returns currently configured custom nameservers, to be used when toggling
     * to 'custom' from 'automatic' or 'google', prefer nameservers in the
     * following priority:
     *
     * 1) policy-enforced nameservers,
     * 2) previously manually entered nameservers (|savedCustomNameservers_|),
     * 3) policy-recommended nameservers,
     * 4) active nameservers (e.g. from DHCP).
     */
    getCustomNameServers_() {
        const policyEnforcedNameservers = this.getPolicyEnforcedNameservers_();
        if (policyEnforcedNameservers !== null) {
            return policyEnforcedNameservers.slice();
        }
        if (this.savedCustomNameservers_.length > 0) {
            return this.savedCustomNameservers_;
        }
        const policyRecommendedNameservers = this.getPolicyRecommendedNameservers_();
        if (policyRecommendedNameservers !== null) {
            return policyRecommendedNameservers.slice();
        }
        return this.nameservers_;
    }
    /**
     * Event triggered when the selected type changes. Updates nameservers and
     * sends the change value if necessary.
     */
    onTypeChange_() {
        const el = this.shadowRoot.querySelector('#nameserverType');
        assert(el);
        const nameserversType = el.selected;
        this.nameserversType_ = nameserversType;
        this.savedNameserversType_ = nameserversType;
        if (nameserversType === NameserversType.CUSTOM) {
            this.setNameservers_(nameserversType, this.getCustomNameServers_(), true /* send */);
            return;
        }
        this.sendNameServers_();
    }
    /**
     * Event triggered when a |nameservers_| value changes through the custom
     * namservers UI.
     * This gets called after data-binding updates a |nameservers_[i]| entry.
     * This saves the custom nameservers and reflects that change to the backend
     * (sending the custom nameservers).
     */
    onValueChange_() {
        this.savedCustomNameservers_ = this.nameservers_.slice();
        this.sendNameServers_();
    }
    /**
     * Sends the current nameservers type (for automatic) or value.
     */
    sendNameServers_() {
        let eventField;
        let eventValue;
        const nameserversType = this.nameserversType_;
        if (nameserversType === NameserversType.CUSTOM) {
            eventField = 'nameServers';
            eventValue = this.nameservers_;
        }
        else if (nameserversType === NameserversType.GOOGLE) {
            this.nameservers_ = GOOGLE_NAMESERVERS;
            eventField = 'nameServers';
            eventValue = GOOGLE_NAMESERVERS;
        }
        else { // nameserversType === NameserversType.AUTOMATIC
            // If not connected, properties will clear. Otherwise they may or may not
            // change so leave them as-is.
            assert(this.managedProperties);
            if (!OncMojo.connectionStateIsConnected(this.managedProperties.connectionState)) {
                this.nameservers_ = [];
            }
            else {
                this.nameservers_ = this.clearEmptyNameServers_(this.nameservers_);
            }
            eventField = 'nameServersConfigType';
            eventValue = 'DHCP';
        }
        const event = new CustomEvent('nameservers-change', {
            bubbles: true,
            composed: true,
            detail: {
                field: eventField,
                value: eventValue,
            },
        });
        this.dispatchEvent(event);
    }
    clearEmptyNameServers_(nameservers) {
        return nameservers.filter((nameserver) => (!!nameserver && nameserver !== EMPTY_NAMESERVER));
    }
    doNothing_(event) {
        event.stopPropagation();
    }
    /**
     * @return Accessibility label for nameserver input with given index.
     */
    getCustomNameServerInputA11yLabel_(index) {
        return this.i18n('networkNameserversCustomInputA11yLabel', index + 1);
    }
}
customElements.define(NetworkNameserversElement.is, NetworkNameserversElement);
