// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import * as MetaKeyTypes from './meta_key.mojom-webui.js';
/**
 * Refers to the state of an 'shortcut-input-key' item.
 */
export var KeyInputState;
(function (KeyInputState) {
    KeyInputState["NOT_SELECTED"] = "not-selected";
    KeyInputState["MODIFIER_SELECTED"] = "modifier-selected";
    KeyInputState["ALPHANUMERIC_SELECTED"] = "alpha-numeric-selected";
})(KeyInputState || (KeyInputState = {}));
export var Modifier;
(function (Modifier) {
    Modifier[Modifier["NONE"] = 0] = "NONE";
    Modifier[Modifier["SHIFT"] = 2] = "SHIFT";
    Modifier[Modifier["CONTROL"] = 4] = "CONTROL";
    Modifier[Modifier["ALT"] = 8] = "ALT";
    Modifier[Modifier["COMMAND"] = 16] = "COMMAND";
    Modifier[Modifier["FN_KEY"] = 32] = "FN_KEY";
})(Modifier || (Modifier = {}));
export const Modifiers = [
    Modifier.SHIFT,
    Modifier.CONTROL,
    Modifier.ALT,
    Modifier.COMMAND,
    Modifier.FN_KEY,
];
export var AllowedModifierKeyCodes;
(function (AllowedModifierKeyCodes) {
    AllowedModifierKeyCodes[AllowedModifierKeyCodes["SHIFT"] = 16] = "SHIFT";
    AllowedModifierKeyCodes[AllowedModifierKeyCodes["CTRL"] = 17] = "CTRL";
    AllowedModifierKeyCodes[AllowedModifierKeyCodes["ALT"] = 18] = "ALT";
    AllowedModifierKeyCodes[AllowedModifierKeyCodes["META_LEFT"] = 91] = "META_LEFT";
    AllowedModifierKeyCodes[AllowedModifierKeyCodes["META_RIGHT"] = 92] = "META_RIGHT";
    AllowedModifierKeyCodes[AllowedModifierKeyCodes["FN_KEY"] = 255] = "FN_KEY";
})(AllowedModifierKeyCodes || (AllowedModifierKeyCodes = {}));
export const ModifierKeyCodes = [
    AllowedModifierKeyCodes.SHIFT,
    AllowedModifierKeyCodes.ALT,
    AllowedModifierKeyCodes.CTRL,
    AllowedModifierKeyCodes.META_LEFT,
    AllowedModifierKeyCodes.META_RIGHT,
    AllowedModifierKeyCodes.FN_KEY,
];
export const MetaKey = MetaKeyTypes.MetaKey;
export const getSortedModifiers = (modifierStrings) => {
    const sortOrder = ['meta', 'ctrl', 'alt', 'shift', 'fn'];
    if (modifierStrings.length <= 1) {
        return modifierStrings;
    }
    return modifierStrings.sort((a, b) => sortOrder.indexOf(a) - sortOrder.indexOf(b));
};
// The keys in this map are pulled from the file:
// ui/events/keycodes/dom/dom_code_data.inc
export const KeyToIconNameMap = {
    'Accessibility': 'accessibility',
    'ArrowDown': 'arrow-down',
    'ArrowLeft': 'arrow-left',
    'ArrowRight': 'arrow-right',
    'ArrowUp': 'arrow-up',
    'AudioVolumeDown': 'volume-down',
    'AudioVolumeMute': 'volume-mute',
    'AudioVolumeUp': 'volume-up',
    'BrightnessDown': 'display-brightness-down',
    'BrightnessUp': 'brightness-up-refresh',
    'BrowserBack': 'back',
    'BrowserForward': 'forward',
    'BrowserHome': 'browser-home',
    'BrowserRefresh': 'refresh',
    'BrowserSearch': 'browser-search',
    'CameraAccessToggle': 'camera-access-toggle',
    'ContextMenu': 'menu',
    'DoNotDisturb': 'do-not-disturb',
    'EmojiPicker': 'emoji-picker',
    'EnableOrToggleDictation': 'dictation-toggle',
    'KeyboardBacklightToggle': 'keyboard-brightness-toggle',
    'KeyboardBrightnessUp': 'keyboard-brightness-up',
    'KeyboardBrightnessDown': 'keyboard-brightness-down',
    'LaunchApplication1': 'overview-refresh',
    'LaunchApplication2': 'calculator',
    'LaunchAssistant': 'assistant',
    'LaunchMail': 'launch-mail',
    'MediaFastForward': 'fast-forward',
    'MediaPause': 'pause',
    'MediaPlay': 'play',
    'MediaPlayPause': 'play-pause',
    'MediaTrackNext': 'next-track',
    'MediaTrackPrevious': 'last-track',
    'MicrophoneMuteToggle': 'microphone-mute',
    'ModeChange': 'globe',
    'ViewAllApps': 'view-all-apps',
    'Power': 'power',
    'PrintScreen': 'screenshot',
    'PrivacyScreenToggle': 'electronic-privacy-screen',
    'Settings': 'settings-icon',
    'Standby': 'lock',
    'ZoomToggle': 'fullscreen',
    'QuickInsert': 'quick-insert',
};
/**
 * Map the modifier keys to the bit value. Currently the modifiers only
 * contains the following four.
 */
export const modifierBitMaskToString = new Map([
    [Modifier.CONTROL, 'ctrl'],
    [Modifier.SHIFT, 'shift'],
    [Modifier.ALT, 'alt'],
    [Modifier.COMMAND, 'command'],
]);
export function createInputKeyParts(shortcutLabelProperties, useNarrowLayout = false) {
    const inputKeys = [];
    const pressedModifiers = [];
    for (const [bitValue, modifierName] of modifierBitMaskToString) {
        if ((shortcutLabelProperties.accelerator.modifiers & bitValue) !== 0) {
            const key = document.createElement('shortcut-input-key');
            key.keyState = KeyInputState.MODIFIER_SELECTED;
            // Current use cases outside keyboard page or shortcut page only consider
            // 'meta' instead of 'command'.
            key.key = modifierName === 'command' ? 'meta' : modifierName;
            key.metaKey = shortcutLabelProperties.metaKey;
            key.narrow = useNarrowLayout;
            inputKeys.push(key);
            pressedModifiers.push(modifierName);
        }
    }
    const keyDisplay = shortcutLabelProperties.keyDisplay;
    if (!pressedModifiers.includes(keyDisplay.toLowerCase())) {
        const key = document.createElement('shortcut-input-key');
        key.keyState = KeyInputState.ALPHANUMERIC_SELECTED;
        key.key = keyDisplay;
        key.narrow = useNarrowLayout;
        inputKeys.push(key);
    }
    return inputKeys;
}
// TODO(b/340609992): Encapsulate this as a new element too.
export function createShortcutAppendedKeyLabel(shortcutLabelProperties, useNarrowLayout = false) {
    const reminder = document.createElement('div');
    reminder.innerHTML = shortcutLabelProperties.shortcutLabelText;
    // TODO(b/340609992): Move this out of the helper function as a new element.
    const keyCodes = document.createElement('span');
    keyCodes.append(...createInputKeyParts(shortcutLabelProperties, useNarrowLayout));
    reminder.firstElementChild.replaceWith(keyCodes);
    reminder.classList.add('reminder-label');
    return reminder;
}
