// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
export default`#version 100\nprecision mediump float;\nprecision highp int;\n\nuniform vec4 u_top_left_background_color;\nuniform vec4 u_bottom_right_background_color;\nuniform vec4 u_sparkle_color;\nuniform vec2 u_resolution;\nuniform float u_time;\nuniform float u_pixel_density;\n\nuniform float u_grid_num;\nuniform float u_luma_matte_blend_factor;\nuniform float u_luma_matte_overall_brightness;\nuniform float u_inverse_luma;\nuniform float u_opacity;\nuniform vec3 u_noise_move;\n\nconst float PI = 3.1415926535897932384626;\nconst float PI_ROTATE_RIGHT = PI * 0.0078125;\nconst float PI_ROTATE_LEFT = PI * -0.0078125;\nconst float ONE_THIRD = 1. / 3.;\n\nvec4 get_uv_and_density_adjusted_uv(vec2 frag_coord,\n                                    vec2 resolution_pixels,\n                                    float pixel_density_divisor,\n                                    float aspect) {\n  vec2 uv = frag_coord / resolution_pixels;\n  vec2 density_adjusted_uv =\n      uv - mod(uv, pixel_density_divisor / resolution_pixels);\n\n  density_adjusted_uv.y = (1.0 - density_adjusted_uv.y) * aspect;\n  uv.y = (1.0 - uv.y) * aspect;\n\n  return vec4(uv, density_adjusted_uv);\n}\n\nfloat get_loudness(vec4 opposing_color) {\n  return 0.45 +\n         (2. * opposing_color.r + opposing_color.g + 0.5 * opposing_color.b) *\n             ONE_THIRD * 0.55;\n}\n\nhighp float triangle_noise(highp vec2 n) {\n  n = fract(n * vec2(5.3987, 5.4421));\n  n += dot(n.yx, n.xy + vec2(21.5351, 14.3137));\n  highp float xy = n.x * n.y;\n  // compute in [0..2[ and remap to [-1.0..1.0[\n  return fract(xy * 95.4307) + fract(xy * 75.04961) - 1.0;\n}\n\nhighp float sparkles(highp vec2 uv, highp float t) {\n  highp float n = triangle_noise(uv);\n  highp float s = 0.0;\n  for (highp float i = 0.0; i < 4.0; i += 1.0) {\n    highp float l = i * 0.01;\n    highp float h = l + 0.1;\n    highp float o = smoothstep(n - l, h, n);\n    o *= abs(sin(PI * o * (t + 0.55 * i)));\n    s += o;\n  }\n  return s;\n}\n\nvec2 circular_offset(float time, vec2 dist) {\n  float r = time * PI;\n  return dist * vec2(cos(r), sin(r));\n}\n\nmat2 rotate2d(float rad) {\n  return mat2(cos(rad), -sin(rad), sin(rad), cos(rad));\n}\n\nfloat soft_circle(vec2 uv, vec2 xy, float radius, float blur) {\n  float blur_half = blur / 2.0;\n  float d = distance(uv, xy);\n  return 1.0 - smoothstep(radius - blur_half, radius + blur_half, d);\n}\n\nfloat circle_grid(vec2 resolution,\n                  vec2 coord,\n                  float time,\n                  vec2 center,\n                  float rotation,\n                  float cell_diameter) {\n  coord =\n      rotate2d(rotation + cell_diameter * 10. * PI) * (center - coord) + center;\n  coord = mod(coord, cell_diameter) / resolution;\n\n  float normal_radius = cell_diameter / resolution.y * 0.5;\n  float radius = 0.65 * normal_radius;\n\n  return soft_circle(coord, vec2(normal_radius), radius, radius);\n}\n\nfloat turbulence(vec2 uv, float t) {\n  vec2 scale = vec2(1.5);\n\n  t = t * 10.0;\n\n  vec2 o1 = scale * 0.5 + circular_offset(t * 0.01, scale * vec2(0.55));\n  vec2 o2 = scale * 0.2 + circular_offset(t / -150.0, scale * vec2(0.45));\n  vec2 o3 = scale + circular_offset(t / 300.0, scale * vec2(0.35));\n\n  uv = uv * scale;\n\n  float g1 = circle_grid(scale, uv, t, o1, t * PI_ROTATE_RIGHT, 0.17);\n  float g2 = circle_grid(scale, uv, t, o2, t * PI_ROTATE_LEFT, 0.2);\n  float g3 = circle_grid(scale, uv, t, o3, t * PI_ROTATE_RIGHT, 0.275);\n\n  return smoothstep(0., 1., (g1 + g2 + g3) * 0.625);\n}\n\nvec2 distort(vec2 p,\n             float time,\n             float distort_amount_radial,\n             float distort_amount_xy) {\n  float angle = atan(p.y, p.x);\n  return p +\n         vec2(sin(angle * 8.0 + time * 0.003 + 1.641),\n              cos(angle * 5.0 + 2.14 + time * 0.00412)) *\n             distort_amount_radial +\n         vec2(sin(p.x * 0.01 + time * 0.00215 + 0.8123),\n              cos(p.y * 0.01 + time * 0.005931)) *\n             distort_amount_xy;\n}\n\n// Perceived luminosity, not absolute luminosity.\nfloat get_luminosity(vec3 c) {\n  return 0.3 * c.r + 0.59 * c.g + 0.11 * c.b;\n}\n\n// Creates a luminosity mask and clamp to the legal range.\nvec3 mask_luminosity(vec3 dest, float lum) {\n  dest.rgb *= vec3(lum);\n  // Clip back into the legal range\n  dest = clamp(dest, vec3(0.), vec3(1.0));\n  return dest;\n}\n\n// Return range [-1, 1].\nvec3 hash(vec3 p) {\n  p = fract(p * vec3(.3456, .1234, .9876));\n  p += dot(p, p.yxz + 43.21);\n  p = (p.xxy + p.yxx) * p.zyx;\n  return (fract(sin(p) * 4567.1234567) - .5) * 2.;\n}\n\n// Skew factors (non-uniform).\nconst float SKEW = 0.3333333;    // 1/3\nconst float UNSKEW = 0.1666667;  // 1/6\n\n// Return range roughly [-1,1].\n// It's because the hash function (that returns a random gradient vector)\n// returns different magnitude of vectors. Noise doesn't have to be in the\n// precise range thus skipped normalize.\nfloat simplex3d(vec3 p) {\n  // Skew the input coordinate, so that we get squashed cubical grid\n  vec3 s = floor(p + (p.x + p.y + p.z) * SKEW);\n\n  // Unskew back\n  vec3 u = s - (s.x + s.y + s.z) * UNSKEW;\n\n  // Unskewed coordinate that is relative to p, to compute the noise\n  // contribution based on the distance.\n  vec3 c0 = p - u;\n\n  // We have six simplices (in this case tetrahedron, since we are in 3D) that\n  // we could possibly in. Here, we are finding the correct tetrahedron (simplex\n  // shape), and traverse its four vertices (c0..3) when computing noise\n  // contribution. The way we find them is by comparing c0's x,y,z values. For\n  // example in 2D, we can find the triangle (simplex shape in 2D) that we are\n  // in by comparing x and y values. i.e. x>y lower, x<y, upper triangle. Same\n  // applies in 3D.\n  //\n  // Below indicates the offsets (or offset directions) when c0=(x0,y0,z0)\n  // x0>y0>z0: (1,0,0), (1,1,0), (1,1,1)\n  // x0>z0>y0: (1,0,0), (1,0,1), (1,1,1)\n  // z0>x0>y0: (0,0,1), (1,0,1), (1,1,1)\n  // z0>y0>x0: (0,0,1), (0,1,1), (1,1,1)\n  // y0>z0>x0: (0,1,0), (0,1,1), (1,1,1)\n  // y0>x0>z0: (0,1,0), (1,1,0), (1,1,1)\n  //\n  // The rule is:\n  // * For offset1, set 1 at the max component, otherwise 0.\n  // * For offset2, set 0 at the min component, otherwise 1.\n  // * For offset3, set 1 for all.\n  //\n  // Encode x0-y0, y0-z0, z0-x0 in a vec3\n  vec3 en = c0 - c0.yzx;\n  // Each represents whether x0>y0, y0>z0, z0>x0\n  en = step(vec3(0.), en);\n  // en.zxy encodes z0>x0, x0>y0, y0>x0\n  vec3 offset1 = en * (1. - en.zxy);       // find max\n  vec3 offset2 = 1. - en.zxy * (1. - en);  // 1-(find min)\n  vec3 offset3 = vec3(1.);\n\n  vec3 c1 = c0 - offset1 + UNSKEW;\n  vec3 c2 = c0 - offset2 + UNSKEW * 2.;\n  vec3 c3 = c0 - offset3 + UNSKEW * 3.;\n\n  // Kernel summation: dot(max(0, r^2-d^2))^4, noise contribution)\n  //\n  // First compute d^2, squared distance to the point.\n  vec4 w;  // w = max(0, r^2 - d^2))\n  w.x = dot(c0, c0);\n  w.y = dot(c1, c1);\n  w.z = dot(c2, c2);\n  w.w = dot(c3, c3);\n\n  // Noise contribution should decay to zero before they cross the simplex\n  // boundary. Usually r^2 is 0.5 or 0.6; 0.5 ensures continuity but 0.6\n  // increases the visual quality for the application where discontinuity isn't\n  // noticeable.\n  w = max(0.6 - w, 0.);\n\n  // Noise contribution from each point.\n  vec4 nc;\n  nc.x = dot(hash(s), c0);\n  nc.y = dot(hash(s + offset1), c1);\n  nc.z = dot(hash(s + offset2), c2);\n  nc.w = dot(hash(s + offset3), c3);\n\n  nc *= w * w * w * w;\n\n  // Add all the noise contributions.\n  // Should multiply by the possible max contribution to adjust the range in\n  // [-1,1].\n  return dot(vec4(32.), nc);\n}\n\n// Random rotations.\n// The way you create fractal noise is layering simplex noise with some\n// rotation. To make random cloud looking noise, the rotations should not align.\n// (Otherwise it creates patterned noise). Below rotations only rotate in one\n// axis.\nconst mat3 rot1 = mat3(1.0, 0., 0., 0., 0.15, -0.98, 0., 0.98, 0.15);\nconst mat3 rot2 = mat3(-0.95, 0., -0.3, 0., 1., 0., 0.3, 0., -0.95);\nconst mat3 rot3 = mat3(1.0, 0., 0., 0., -0.44, -0.89, 0., 0.89, -0.44);\n\n// Octave = 4\n// Divide each coefficient by 3 to produce more grainy noise.\nfloat simplex3d_fractal(vec3 p) {\n  return 0.675 * simplex3d(p * rot1) + 0.225 * simplex3d(2.0 * p * rot2) +\n         0.075 * simplex3d(4.0 * p * rot3) + 0.025 * simplex3d(8.0 * p);\n}\n\nfloat saturate(float value) {\n  return clamp(value, 0.0, 1.0);\n}\n\n// Screen blend\nvec3 screen(vec3 dest, vec3 src) {\n  return dest + src - dest * src;\n}\n\nvoid main() {\n  float aspect = u_resolution.y / u_resolution.x;\n  vec4 compound_uv = get_uv_and_density_adjusted_uv(\n      gl_FragCoord.xy, u_resolution, u_pixel_density, aspect);\n  vec2 uv = compound_uv.xy;\n  vec2 density_adjusted_uv = compound_uv.zw;\n\n  vec2 gradient_direction = gl_FragCoord.xy / u_resolution;\n  float gradient =\n      saturate((1.0 - gradient_direction.y) * aspect + gradient_direction.x);\n  vec4 background_color = mix(u_top_left_background_color,\n                              u_bottom_right_background_color, gradient);\n\n  // apply noise\n  vec3 noise_p =\n      vec3(density_adjusted_uv + u_noise_move.xy, u_noise_move.z) * u_grid_num;\n  vec3 color = background_color.rgb;\n\n  // Add dither with triangle distribution to avoid color banding. Dither in the\n  // shader here as we are in gamma space.\n  float dither = triangle_noise(gl_FragCoord.xy * u_pixel_density) / 255.;\n\n  // The result color should be pre-multiplied, i.e. [R*A, G*A, B*A, A], thus\n  // need to multiply rgb with a to get the correct result.\n  color = (color + dither) * u_opacity;\n  vec4 noise = vec4(color, u_opacity);\n\n  float sparkle_luma = 1.0 - get_luminosity(vec3(simplex3d(noise_p)));\n  sparkle_luma = max(/* intensity= */ 1.75 * sparkle_luma - /* dim= */ 1.3, 0.);\n  float sparkle_alpha =\n      sparkles(gl_FragCoord.xy - mod(gl_FragCoord.xy, u_pixel_density * 0.8),\n               u_time / 1000.0);\n  vec4 sparkle =\n      vec4(mask_luminosity(u_sparkle_color.rgb * sparkle_alpha, sparkle_luma) *\n               u_sparkle_color.a,\n           u_sparkle_color.a);\n\n  vec3 effect = noise.rgb + sparkle.rgb;\n  gl_FragColor =\n      mix(background_color, vec4(effect, 1.0), smoothstep(0., 0.75, 0.3));\n}`;