// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * The page/pillar name for the pages in the Highlights app.
 */
export var Page;
(function (Page) {
    Page["HOME"] = "Home";
    Page["EASY"] = "Easy";
    Page["FAST"] = "Fast";
    Page["POWERFUL"] = "Powerful";
    Page["SECURE"] = "Secure";
    Page["DISPLAY"] = "Display";
    Page["KEYBOARD"] = "Keyboard";
    Page["GRAPHICS"] = "Graphics";
    Page["GAME"] = "Game";
    Page["PERIPHERALS"] = "Peripherals";
    Page["PERFORMANCE"] = "Performance";
    Page["APPS"] = "Apps";
    Page["CHROMEOS"] = "ChromeOS";
    // New in 2024 Cycle 1 refresh
    // CBX:
    Page["GOOGLE_AI"] = "GoogleAI";
    Page["EASY_TO_USE"] = "EasyToUse";
    // CB (Shared with CBX: Performance, Apps):
    Page["GOOGLE_BUILT_IN"] = "GoogleBuiltIn";
})(Page || (Page = {}));
/**
 * The name for the details pages in the Highlights app.
 */
export var DetailsPage;
(function (DetailsPage) {
    // 2023 CBX first released:
    DetailsPage["ADOBE"] = "Adobe";
    DetailsPage["BATTERY"] = "Battery";
    DetailsPage["COMPARISON"] = "Comparison";
    DetailsPage["DISPLAY_ENTERTAINMENT"] = "DisplayEntertainment";
    DetailsPage["DISPLAY_PERFORMANCE"] = "DisplayPerformance";
    DetailsPage["ENTERTAINMENT_APPS"] = "EntertainmentApps";
    DetailsPage["GOOGLE_APPS"] = "GoogleApps";
    DetailsPage["LUMAFUSION"] = "LumaFusion";
    DetailsPage["MESSAGING"] = "Messaging";
    DetailsPage["MOBILE_GAMING"] = "MobileGaming";
    DetailsPage["MS_365_APPS"] = "MS365Apps";
    DetailsPage["MS_OFFICE"] = "MSOffice";
    DetailsPage["NEARBY_SHARE"] = "NearbyShare";
    DetailsPage["OFFLINE_MODE"] = "OfflineMode";
    DetailsPage["PC_CONSOLE_GAMING"] = "PCConsoleGaming";
    DetailsPage["PHOTOS"] = "Photos";
    DetailsPage["PROCESSOR"] = "Processor";
    DetailsPage["STORAGE"] = "Storage";
    DetailsPage["SWITCHING"] = "Switching";
    DetailsPage["VIDEO_CALL"] = "VideoCall";
    // New in 2024 Cycle 1 refresh
    // CBX:
    DetailsPage["BUILT_IN_SECURITY"] = "BuiltInSecurity";
    DetailsPage["WEBCAM"] = "Webcam";
    DetailsPage["GAME_DASH_BOARD"] = "GameDashboard";
    DetailsPage["GEMINI_FOR_ALL"] = "GeminiForAll";
    DetailsPage["HELP_ME_WRITE"] = "HelpMeWrite";
    DetailsPage["GEMINI_FOR_WORK_SPACE"] = "GeminiForWorkSpace";
    DetailsPage["AI_BACKGROUND"] = "AIBackground";
    DetailsPage["AI_PREMIUM_PLAN"] = "AIPremiumPlan";
    // CB, note that detail page for generic was not recorded before 2024 C1:
    DetailsPage["FAST_BOOT"] = "FastBoot";
    DetailsPage["AUTO_UPDATE"] = "AutoUpdate";
    DetailsPage["EASY_SETUP"] = "EasySetup";
    DetailsPage["LAUNCHER_SEARCH"] = "LauncherSearch";
    DetailsPage["GOOGLE_TOOLS_BUILT_IN"] = "GoogleToolsBuiltIn";
    DetailsPage["TITAN_C2"] = "TitanC2";
    DetailsPage["CREATIVITY"] = "Creativity";
    DetailsPage["ENTERTAINMENT"] = "Entertainment";
    DetailsPage["PRODUCTIVITY"] = "Productivity";
    DetailsPage["PLAY_STORE"] = "PlayStore";
    // Enum shared between CB & CBX are: BATTERY, GOOGLE_APPS, NEARBY_SHARE,
    // MESSAGING, BUILT_IN_SECURITY,MS_365_APPS, SWITCHING, COMPARISON
    // New in 2024 Cycle 2 refresh
    // CBX:
    DetailsPage["HELP_ME_READ"] = "HelpMeRead";
    DetailsPage["LIVE_TRANSLATE"] = "LiveTranslate";
    // New in 2025 Refresh
    // CB Generic:
    DetailsPage["QUICK_INSERT"] = "QuickInsert";
    DetailsPage["GEMINI_PWA"] = "GeminiPWA";
    DetailsPage["SELECT_TO_SEARCH"] = "SelectToSearch";
    DetailsPage["GEMINI_ADVANCE"] = "GeminiAdvance";
    DetailsPage["DRIVE_INTEGRATION"] = "DriveIntegration";
    DetailsPage["BETTER_TOGETHER"] = "BetterTogether";
    DetailsPage["PHONE_HUB"] = "PhoneHub";
    DetailsPage["WELCOM_RECAP"] = "WelcomeRecap";
    // CBX:
    DetailsPage["FILE_SYNC"] = "FileSync";
    DetailsPage["MAGIC_EDITOR"] = "MagicEditor";
    // Enum shared between CB & CBX are: GEMINI_PWA, SELECT_TO_SEARCH,
    // QUICK_INSERT, GEMINI_ADVANCE, BETTER_TOGETHER, and PHONE_HUB
})(DetailsPage || (DetailsPage = {}));
/**
 * The buttons in each pillar page of the Highlights app.
 */
export var PillarButton;
(function (PillarButton) {
    PillarButton["NEXT"] = "Next";
    PillarButton["PREVIOUS"] = "Previous";
})(PillarButton || (PillarButton = {}));
/**
 * Errors in the Highlights app.
 *
 * This is used by histogram: DemoMode.Highlights.Error
 *
 * These values are persisted to logs, so entries should not be renumbered and
 * numeric values should never be reused.
 */
var DemoModeHighlightsError;
(function (DemoModeHighlightsError) {
    DemoModeHighlightsError[DemoModeHighlightsError["ATTRACTION_LOOP_TIMESTAMP_INVALID"] = 0] = "ATTRACTION_LOOP_TIMESTAMP_INVALID";
    DemoModeHighlightsError[DemoModeHighlightsError["PAGE_VIEW_DURATION_INVALID"] = 1] = "PAGE_VIEW_DURATION_INVALID";
    DemoModeHighlightsError[DemoModeHighlightsError["DETAILS_PAGE_VIEW_DURATION_INVALID"] = 2] = "DETAILS_PAGE_VIEW_DURATION_INVALID";
})(DemoModeHighlightsError || (DemoModeHighlightsError = {}));
/**
 * A map between the Page in this js file and DemoModeHighlightsAction enum in
 * the UMA enums.xml.
 */
const FirstInteractionActionMap = new Map([
    [Page.HOME, 0],
    [Page.EASY, 1],
    [Page.FAST, 2],
    [Page.POWERFUL, 3],
    [Page.SECURE, 4],
    [Page.DISPLAY, 5],
    [Page.KEYBOARD, 6],
    [Page.GRAPHICS, 7],
    [Page.GAME, 8],
    [Page.PERIPHERALS, 9],
    [Page.PERFORMANCE, 10],
    [Page.APPS, 11],
    [Page.CHROMEOS, 12],
    [Page.GOOGLE_AI, 13],
    [Page.EASY_TO_USE, 14],
    [Page.GOOGLE_BUILT_IN, 15],
    ['MAX_VALUE', 15],
]);
/**
 * Provides interfaces for emitting metrics from demo mode apps to UMA.
 *
 * Note: DemoMode.Highlights.* metrics and actions are recorded via
 * runtime-downloaded content that is not checked into Chromium. Please do not
 * delete this code, even if it looks like there's no production references in
 * Chromium, without first consulting the Demo Mode team.
 */
class DemoMetricsService {
    firstInteractionRecorded = false;
    // Record the action that the user breaks the current Attract Loop.
    recordAttractLoopBreak() {
        chrome.metricsPrivateIndividualApis.recordUserAction('DemoMode_AttractLoop_Break');
    }
    /**
     * Record the timestamp (i.e. milliseconds from the beginning of the Attract
     * Loop video) at which the user broke the Attract Loop.
     */
    recordAttractLoopBreakTimestamp(timestampInMilliseconds) {
        if (isNaN(timestampInMilliseconds)) {
            this.recordError_(DemoModeHighlightsError.ATTRACTION_LOOP_TIMESTAMP_INVALID);
            return;
        }
        chrome.metricsPrivateIndividualApis.recordMediumTime('DemoMode.AttractLoop.Timestamp', timestampInMilliseconds);
    }
    /**
     * Record the first action of current user.
     */
    recordFirstInteraction(action) {
        if (!this.firstInteractionRecorded) {
            chrome.metricsPrivateIndividualApis.recordEnumerationValue('DemoMode.Highlights.FirstInteraction', FirstInteractionActionMap.get(action), FirstInteractionActionMap.get('MAX_VALUE'));
            this.firstInteractionRecorded = true;
        }
    }
    /**
     * Record the button clicks in home page of the current user.
     */
    recordHomePageButtonClick(page) {
        chrome.metricsPrivateIndividualApis.recordUserAction('DemoMode_Highlights_HomePage_Click_' + page + 'Button');
        this.recordFirstInteraction(page);
    }
    /**
     * Record the button clicks in home page of the current user.
     */
    recordNavbarButtonClick(page) {
        chrome.metricsPrivateIndividualApis.recordUserAction('DemoMode_Highlights_Navbar_Click_' + page + 'Button');
        this.recordFirstInteraction(page);
    }
    /**
     * Record the button click in pillar pages of the current user.
     */
    recordPillarPageButtonClick(pillarButton) {
        chrome.metricsPrivateIndividualApis.recordUserAction('DemoMode_Highlights_PillarPage_Click_' + pillarButton + 'Button');
    }
    /**
     * Record the duration of the user staying on the page.
     */
    recordPageViewDuration(page, durationInMilliseconds) {
        if (!durationInMilliseconds) {
            this.recordError_(DemoModeHighlightsError.PAGE_VIEW_DURATION_INVALID);
            return;
        }
        chrome.metricsPrivateIndividualApis.recordMediumTime('DemoMode.Highlights.PageStayDuration.' + page + 'Page', durationInMilliseconds);
    }
    /**
     * Record the details page clicked by the current user
     */
    recordDetailsPageClicked(detailsPage) {
        chrome.metricsPrivateIndividualApis.recordUserAction('DemoMode_Highlights_DetailsPage_Clicked_' + detailsPage + 'Button');
    }
    /**
     * Record the duration of the user staying on a details page
     */
    recordDetailsPageViewDuration(detailsPage, durationInMilliseconds) {
        if (!durationInMilliseconds) {
            this.recordError_(DemoModeHighlightsError.DETAILS_PAGE_VIEW_DURATION_INVALID);
            return;
        }
        chrome.metricsPrivateIndividualApis.recordMediumTime('DemoMode.Highlights.DetailsPageStayDuration.' + detailsPage + 'Page', durationInMilliseconds);
    }
    /**
     * Record error in highlight app.
     */
    recordError_(error) {
        const maxValue = Object.keys(DemoModeHighlightsError).length;
        chrome.metricsPrivateIndividualApis.recordEnumerationValue('DemoMode.Highlights.Error', error, maxValue);
    }
}
export const metricsService = new DemoMetricsService();
