// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import './diagnostics_shared.css.js';
import '/strings.m.js';
import 'chrome://resources/d3/d3.min.js';
import { I18nMixin } from 'chrome://resources/ash/common/cr_elements/i18n_mixin.js';
import { loadTimeData } from 'chrome://resources/ash/common/load_time_data.m.js';
import { assert } from 'chrome://resources/js/assert.js';
import { PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { getTemplate } from './realtime_cpu_chart.html.js';
const RealtimeCpuChartElementBase = I18nMixin(PolymerElement);
/**
 * @fileoverview
 * 'realtime-cpu-chart' is a moving stacked area graph component used to display
 * a realtime cpu usage information.
 */
export class RealtimeCpuChartElement extends RealtimeCpuChartElementBase {
    static get is() {
        return 'realtime-cpu-chart';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            user: {
                type: Number,
                value: 0,
            },
            system: {
                type: Number,
                value: 0,
            },
            numDataPoints: {
                type: Number,
                value: 50,
            },
            dataRefreshPerSecond: {
                type: Number,
                value: 2,
            },
            /**
             * Chart rendering frames per second.
             * Strongly preferred to be multiples of dataRefreshPerSecond. If not,
             * there will be a small (hard to notice) jittering at every data refresh.
             */
            framesPerSecond: {
                type: Number,
                value: 30,
            },
            /**
             * Duration of each frame in milliseconds.
             */
            frameDuration: {
                readOnly: true,
                type: Number,
                computed: 'getFrameDuration(dataRefreshPerSecond, framesPerSecond)',
            },
            width: {
                type: Number,
                value: 560,
            },
            height: {
                type: Number,
                value: 114,
            },
            padding: {
                type: Object,
                value: { top: 10, right: 5, bottom: 8, left: 50, tick: 10 },
            },
            graphWidth: {
                readOnly: true,
                type: Number,
                computed: 'getGraphDimension(width, padding.left, padding.right)',
            },
            graphHeight: {
                readOnly: true,
                type: Number,
                computed: 'getGraphDimension(height, padding.top, padding.bottom)',
            },
        };
    }
    constructor() {
        super();
        // Helper function to map range of x coordinates to graph width.
        this.xAxisScaleFn = d3.scaleLinear();
        // Helper function to map range of y coordinates to graph height.
        this.yAxisScaleFn = d3.scaleLinear();
        this.data = [];
        // DOMHighResTimeStamp of last graph render.
        this.lastRender = 0;
        // Current render frame out of this.framesPerSecond.
        this.currentFrame = 0;
        // Y-Values where we should mark ticks for the y-axis on the left.
        this.yAxisTicks = [0, 25, 50, 75, 100];
        // Initialize the data array with data outside the chart boundary.
        // Note that with side nav DOM manipulation, created() isn't guaranteed to
        // be called only once.
        this.data = [];
        for (let i = 0; i < this.numDataPoints; ++i) {
            this.data.push({ user: -1, system: -1 });
        }
    }
    static get observers() {
        return ['setScaling(graphWidth)'];
    }
    ready() {
        super.ready();
        this.setScaling();
        this.initializeChart();
        window.addEventListener('resize', () => this.updateChartWidth());
        // Set the initial chart width.
        this.updateChartWidth();
    }
    updateChartWidth() {
        // parseFloat() is used to convert the string returned by
        // getComputedStyleValue() into a number ("642px" --> 642).
        const width = parseFloat(window.getComputedStyle(this).getPropertyValue('--chart-width-nav'));
        if (!isNaN(width)) {
            this.width = width;
        }
    }
    /**
     * Calculate the duration of each frame in milliseconds.
     */
    getFrameDuration() {
        assert(this.dataRefreshPerSecond > 0);
        assert(this.framesPerSecond > 0);
        assert(this.framesPerSecond % this.dataRefreshPerSecond === 0);
        return 1000 / (this.framesPerSecond / this.dataRefreshPerSecond);
    }
    /**
     * Get actual graph dimensions after accounting for margins.
     */
    getGraphDimension(base, ...margins) {
        return margins.reduce(((acc, margin) => acc - margin), base);
    }
    /**
     * Sets scaling functions that convert data -> svg coordinates.
     */
    setScaling() {
        // Map y-values [0, 100] to [graphHeight, 0] inverse linearly.
        // Data value of 0 will map to graphHeight, 100 maps to 0.
        this.yAxisScaleFn =
            d3.scaleLinear().domain([0, 100]).range([this.graphHeight, 0]);
        // Map x-values [0, numDataPoints - 3] to [0, graphWidth] linearly.
        // Data value of 0 maps to 0, and (numDataPoints - 2) maps to graphWidth.
        // numDataPoints is subtracted since 1) data array is zero based, and
        // 2) to smooth out the curve function.
        this.xAxisScaleFn =
            d3.scaleLinear().domain([0, this.numDataPoints - 2]).range([
                0,
                this.graphWidth,
            ]);
        // Draw the y-axis legend and also draw the horizontal gridlines by
        // reversing the ticks back into the chart body.
        const chartGroup = d3.select(this.shadowRoot.querySelector('#chartGroup'));
        assert(chartGroup);
        chartGroup.select('#gridLines')
            .call(d3.axisLeft(this.yAxisScaleFn)
            .tickValues(this.yAxisTicks)
            .tickFormat((y) => this.getPercentageLabel(y))
            .tickPadding(this.padding.tick)
            .tickSize(-this.graphWidth));
    }
    initializeChart() {
        const chartGroup = d3.select(this.shadowRoot.querySelector('#chartGroup'));
        assert(chartGroup);
        // Position chartGroup inside the margin.
        chartGroup.attr('transform', `translate(${this.padding.left},${this.padding.top})`);
        const plotGroup = d3.select(this.shadowRoot.querySelector('#plotGroup'));
        assert(plotGroup);
        // Feed data array to the plot group.
        plotGroup.datum(this.data);
        // Select each area and configure the transition for animation.
        // d3.transition API @ https://github.com/d3/d3-transition#d3-transition.
        // d3.easing API @ https://github.com/d3/d3-ease#api-reference.
        plotGroup.select('.user-area')
            .transition()
            .duration(this.frameDuration)
            .ease((t) => +t); // Linear transition
        plotGroup.select('.system-area')
            .transition()
            .duration(this.frameDuration)
            .ease((t) => +t); // Linear transition
        // Draw initial data and kick off the rendering process.
        this.getDataSnapshotAndRedraw();
        this.render(/*timeStamp=*/ 0);
    }
    getAreaDefinition(areaClass) {
        return d3
            .area()
            // Take the index of each data as x values.
            .x((_, i) => this.xAxisScaleFn(i))
            // Bottom coordinates of each area. System area extends down to -1
            // instead of 0 to avoid the area border from showing up.
            .y0((data) => this?.yAxisScaleFn(areaClass === 'system-area' ? -1 : data?.system))
            // Top coordinates of each area.
            .y1((data) => this?.yAxisScaleFn(areaClass === 'system-area' ? data?.system :
            data?.system + data?.user));
    }
    /**
     * Takes a snapshot of current CPU readings and appends to the data array for
     * redrawing. This method is called after each transition cycle.
     */
    getDataSnapshotAndRedraw() {
        this.data.push({ user: this.user, system: this.system });
        this.data.shift();
        const userArea = this.shadowRoot.querySelector(`path.user-area`);
        assert(userArea);
        const systemArea = this.shadowRoot.querySelector(`path.system-area`);
        assert(systemArea);
        d3.select(userArea).attr('d', this.getAreaDefinition('user-area'));
        d3.select(systemArea).attr('d', this.getAreaDefinition('system-area'));
    }
    render(timeStamp) {
        // Re-render only when this.frameDuration has passed since last render.
        // If we acquire the animation frame before this, do nothing.
        if (timeStamp - this.lastRender > this.frameDuration) {
            this.lastRender = performance.now();
            // Get new data and redraw on each cycle.
            const framesPerCycle = this.framesPerSecond / this.dataRefreshPerSecond;
            if (this.currentFrame === framesPerCycle) {
                this.currentFrame = 0;
                this.getDataSnapshotAndRedraw();
            }
            const userArea = this.shadowRoot.querySelector(`path.user-area`);
            assert(userArea);
            const systemArea = this.shadowRoot.querySelector(`path.system-area`);
            assert(systemArea);
            // Calculate the new position. Use this.currentFrame + 1 since on frame
            // 0, it is already at position 0.
            const pos = -1 * ((this.currentFrame + 1) / framesPerCycle);
            // Slide it to the left
            d3.select(userArea).attr('transform', 'translate(' + this.xAxisScaleFn(pos) + ',0)');
            d3.select(systemArea)
                .attr('transform', 'translate(' + this.xAxisScaleFn(pos) + ',0)');
            this.currentFrame++;
        }
        // Request another frame.
        requestAnimationFrame((timeStamp) => this.render(timeStamp));
    }
    getPercentageLabel(value) {
        return loadTimeData.getStringF('percentageLabel', value);
    }
    getPaddingForTesting() {
        return this.padding;
    }
    getFrameDurationForTesting() {
        return this.frameDuration;
    }
}
customElements.define(RealtimeCpuChartElement.is, RealtimeCpuChartElement);
