import { mojo } from 'chrome://resources/mojo/mojo/public/js/bindings.js';
import { UrlSpec } from 'chrome://resources/mojo/url/mojom/url.mojom-webui.js';
import { String16Spec } from 'chrome://resources/mojo/mojo/public/mojom/base/string16.mojom-webui.js';

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * The Object placed in MessageData.message (and thrown by the Promise returned
 * by sendMessage) if an exception is caught on the receiving end.
 * Note this must be a class (not an interface) whilst there are .js files
 * importing `GenericErrorResponse`, otherwise the export is invisible.
 */
/**
 * To handle generic errors such as `DOMException` not being an `Error`
 * defensively assign '' if the attribute is undefined. Without explicitly
 * extracting fields, `Errors` are sent as `{}` across the pipe.
 */
function serializeError(error) {
    return {
        message: error.message || '',
        name: error.name || '',
        stack: error.stack || '',
    };
}
/**
 * Creates a new JavaScript native Promise and captures its resolve and reject
 * callbacks. The promise, resolve, and reject are available as properties.
 * Inspired by goog.promise.NativeResolver.
 */
class NativeResolver {
    resolve;
    reject;
    promise;
    constructor() {
        this.promise = new Promise((resolve, reject) => {
            this.resolve = resolve;
            this.reject = reject;
        });
    }
}
/**
 * A simplified "assert" that casts away null types. Assumes preconditions that
 * satisfy the assert have already been checked.
 * TODO(b/150650426): consolidate this better.
 */
function assertCast(condition) {
    if (!condition) {
        throw new Error('Failed assertion');
    }
    return condition;
}
/**
 * Enum for reserved message types used in generated messages.
 */
var ReservedMessageTypes;
(function (ReservedMessageTypes) {
    /**
     * Indicates a autogenerated response message for a previously received
     * message.
     */
    ReservedMessageTypes["RESPONSE_TYPE"] = "___response";
    /**
     * Indicates a autogenerated error message for a previously received
     * message.
     */
    ReservedMessageTypes["ERROR_TYPE"] = "___error";
})(ReservedMessageTypes || (ReservedMessageTypes = {}));
/**
 * Checks if a provided message type indicates a generated message.
 */
function isGeneratedMessage(messageType) {
    // Any message type with three underscores before it should only be used
    // in generated messages.
    return messageType.substr(0, 3) === '___';
}
/**
 * Checks a message type is not reserved by generated messages, if it is, throws
 * a error indicating this to the user.
 */
function throwIfReserved(messageType) {
    if (isGeneratedMessage(messageType)) {
        throw new Error(`Unexpected reserved message type: '${messageType}'`);
    }
}
/**
 * The message pipe allows two windows to communicate in 1 direction without
 * having to handle the internals. The caller can send messages to the other
 * window and receive async responses.
 */
class MessagePipe {
    target_;
    targetOrigin_;
    /**
     * If true any errors thrown in a handler during message handling will be
     * thrown again in addition to being sent over the pipe to the message
     * sender. true by default.
     */
    rethrowErrors;
    /**
     * Client error logger. Mockable for tests that check for errors. This is
     * only used to log errors generated from handlers. Logging occurs on both
     * sides of the message pipe if rethrowErrors is set, otherwise only on
     * the side that sent the message.
     */
    logClientError = (object) => console.error(JSON.stringify(object));
    /**
     * Maps a message type to a message handler, a function which takes in
     * the message and returns a response message or a promise which resolves
     * with a response message.
     */
    messageHandlers_ = new Map();
    /**
     * Maps a message id to a resolver.
     */
    pendingMessages_ = new Map();
    /**
     * The id the next message the object sends will have.
     */
    nextMessageId_ = 0;
    /**
     * The message listener we attach to the window. We need a reference to the
     * function for later removal.
     */
    messageListener_ = (m) => this.receiveMessage_(m);
    /**
     * Constructs a new message pipe to the `target` window which has the
     * `targetOrigin` origin.
     *
     * @param target If not specified, the document tree will be
     *     queried for a iframe with src `targetOrigin` to target.
     */
    constructor(targetOrigin, target, rethrowErrors = true) {
        if (!target) {
            const frame = document.querySelector(`iframe[src^='${targetOrigin}']`);
            if (!frame || !frame.contentWindow) {
                throw new Error('Unable to locate target content window.');
            }
            target = assertCast(frame.contentWindow);
        }
        this.target_ = target;
        this.targetOrigin_ = targetOrigin;
        this.rethrowErrors = rethrowErrors;
        // Make sure we aren't trying to send messages to ourselves.
        console.assert(this.target_ !== window, 'target !== window');
        window.addEventListener('message', this.messageListener_);
    }
    /**
     * Registers a handler to be called when a message of type `messageType` is
     * received. The return value of this handler will automatically be sent to
     * the message source as a response message. If the handler should throw an
     * error while handling a message, the error message will be caught and sent
     * to the message source automatically.
     * NOTE: The message type can not be prefixed with 3 underscores as that is
     * reserved for generated messages. i.e `___hello` is disallowed.
     *
     */
    registerHandler(messageType, handler) {
        throwIfReserved(messageType);
        if (this.messageHandlers_.has(messageType)) {
            throw new Error(`A handler already exists for ${messageType}`);
        }
        this.messageHandlers_.set(messageType, handler);
    }
    /**
     * Wraps `sendMessageImpl()` catching errors from the target context to throw
     * more useful errors with the current context stacktrace attached.
     */
    async sendMessage(messageType, message = {}) {
        try {
            return await this.sendMessageImpl(messageType, message);
        }
        catch (errorResponse) {
            // Create an error with the name of the IPC function invoked, append the
            // stacktrace from the target context (origin of the error) with the
            // stacktrace of the current context.
            const error = new Error(`${messageType}: ${errorResponse.message}`);
            error.name = errorResponse.name || 'Unknown Error';
            error.stack +=
                `\nError from ${this.targetOrigin_}\n${errorResponse.stack}`;
            // TODO(b/156205603): use internal `chrome.crashReportPrivate.reportError`
            // to log this error.
            throw error;
        }
    }
    /**
     * Sends a message to the target window and return a Promise that will resolve
     * on response. If the target handler does not send a response the promise
     * will resolve with a empty object.
     */
    async sendMessageImpl(messageType, message = {}) {
        throwIfReserved(messageType);
        const messageId = this.nextMessageId_++;
        const resolver = new NativeResolver();
        this.pendingMessages_.set(messageId, resolver);
        this.postToTarget_(messageType, message, messageId);
        return resolver.promise;
    }
    /**
     * Removes all listeners this object attaches to window in preparation for
     * destruction.
     */
    detach() {
        window.removeEventListener('message', this.messageListener_);
    }
    /**
     * Handles a message which represents the targets response to a previously
     * sent message.
     */
    handleMessageResponse_(messageType, message, messageId) {
        const { RESPONSE_TYPE, ERROR_TYPE } = ReservedMessageTypes;
        const resolver = assertCast(this.pendingMessages_.get(messageId));
        if (messageType === RESPONSE_TYPE) {
            resolver.resolve(message);
        }
        else if (messageType === ERROR_TYPE) {
            this.logClientError(message);
            resolver.reject(message);
        }
        else {
            console.error(`Response for message ${messageId} received with invalid message type ${messageType}`);
        }
        this.pendingMessages_.delete(messageId);
    }
    /**
     * Calls the relevant handler for a received message and generates the right
     * response message to send back to the source.
     */
    async callHandlerForMessageType_(messageType, message, messageId) {
        const { RESPONSE_TYPE, ERROR_TYPE } = ReservedMessageTypes;
        let response;
        let error = null;
        let sawError = false;
        try {
            const handler = assertCast(this.messageHandlers_.get(messageType));
            response = await handler(message);
        }
        catch (err) {
            // If an error happened capture the error and send it back.
            sawError = true;
            error = err;
            response = serializeError(err);
        }
        this.postToTarget_(sawError ? ERROR_TYPE : RESPONSE_TYPE, response, messageId);
        if (sawError && this.rethrowErrors) {
            // Rethrow the error so the current frame has visibility on its handler
            // failures.
            this.logClientError(error);
            throw error;
        }
    }
    receiveMessage_(e) {
        // Ignore message events missing a type.
        if (typeof e.data !== 'object' || !e.data ||
            typeof e.data.type !== 'string') {
            return;
        }
        const { messageId, type, message } = e.data;
        const { ERROR_TYPE } = ReservedMessageTypes;
        // Ignore any messages that are not from the target origin unless we are
        // explicitly accepting messages from any origin.
        if (e.origin !== this.targetOrigin_ && this.targetOrigin_ !== '*') {
            return;
        }
        // The case that the message is a response to a previously sent message.
        if (isGeneratedMessage(type) && this.pendingMessages_.has(messageId)) {
            this.handleMessageResponse_(type, message, messageId);
            return;
        }
        if (isGeneratedMessage(type)) {
            // Currently all generated messages are only sent in a response, so should
            // have been handled above.
            console.error(`Response with type ${type} for unknown message received.`);
            return;
        }
        if (!this.messageHandlers_.has(type)) {
            // If there is no listener for this event send a error message to source.
            const error = new Error(`No handler registered for message type '${type}'`);
            const errorResponse = serializeError(error);
            this.postToTarget_(ERROR_TYPE, errorResponse, messageId);
            return;
        }
        this.callHandlerForMessageType_(type, message, messageId);
    }
    postToTarget_(messageType, message, messageId) {
        const messageWrapper = {
            messageId,
            type: messageType,
            message: message || {},
        };
        // The next line should probably be passing a transfer argument, but that
        // causes Chrome to send a "null" message. The transfer seems to work
        // without the third argument (but inefficiently, perhaps).
        this.target_.postMessage(messageWrapper, this.targetOrigin_);
    }
}

// ash/webui/help_app_ui/help_app_ui.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const ActionTypeIdSpec = { $: mojo.internal.Enum() };
var ActionTypeId;
(function (ActionTypeId) {
    ActionTypeId[ActionTypeId["MIN_VALUE"] = 0] = "MIN_VALUE";
    ActionTypeId[ActionTypeId["MAX_VALUE"] = 10] = "MAX_VALUE";
    ActionTypeId[ActionTypeId["INVALID"] = 0] = "INVALID";
    ActionTypeId[ActionTypeId["OPEN_CHROME"] = 1] = "OPEN_CHROME";
    ActionTypeId[ActionTypeId["OPEN_LAUNCHER"] = 2] = "OPEN_LAUNCHER";
    ActionTypeId[ActionTypeId["OPEN_PERSONALIZATION_APP"] = 3] = "OPEN_PERSONALIZATION_APP";
    ActionTypeId[ActionTypeId["OPEN_PLAY_STORE"] = 4] = "OPEN_PLAY_STORE";
    ActionTypeId[ActionTypeId["OPEN_GOOGLE_DOCS"] = 5] = "OPEN_GOOGLE_DOCS";
    ActionTypeId[ActionTypeId["OPEN_GOOGLE_PHOTOS"] = 6] = "OPEN_GOOGLE_PHOTOS";
    ActionTypeId[ActionTypeId["OPEN_SETTINGS_PRINTER"] = 7] = "OPEN_SETTINGS_PRINTER";
    ActionTypeId[ActionTypeId["OPEN_PHONE_HUB"] = 8] = "OPEN_PHONE_HUB";
    ActionTypeId[ActionTypeId["OPEN_YOUTUBE"] = 9] = "OPEN_YOUTUBE";
    ActionTypeId[ActionTypeId["OPEN_FILE_MANAGER"] = 10] = "OPEN_FILE_MANAGER";
})(ActionTypeId || (ActionTypeId = {}));
const SettingsComponentSpec = { $: mojo.internal.Enum() };
var SettingsComponent;
(function (SettingsComponent) {
    SettingsComponent[SettingsComponent["MIN_VALUE"] = 0] = "MIN_VALUE";
    SettingsComponent[SettingsComponent["MAX_VALUE"] = 13] = "MAX_VALUE";
    SettingsComponent[SettingsComponent["HOME"] = 0] = "HOME";
    SettingsComponent[SettingsComponent["ACCESSIBILITY"] = 1] = "ACCESSIBILITY";
    SettingsComponent[SettingsComponent["BLUETOOTH"] = 2] = "BLUETOOTH";
    SettingsComponent[SettingsComponent["DISPLAY"] = 3] = "DISPLAY";
    SettingsComponent[SettingsComponent["INPUT"] = 4] = "INPUT";
    SettingsComponent[SettingsComponent["MULTI_DEVICE"] = 5] = "MULTI_DEVICE";
    SettingsComponent[SettingsComponent["PEOPLE"] = 6] = "PEOPLE";
    SettingsComponent[SettingsComponent["PER_DEVICE_KEYBOARD"] = 7] = "PER_DEVICE_KEYBOARD";
    SettingsComponent[SettingsComponent["PER_DEVICE_TOUCHPAD"] = 8] = "PER_DEVICE_TOUCHPAD";
    SettingsComponent[SettingsComponent["PERSONALIZATION"] = 9] = "PERSONALIZATION";
    SettingsComponent[SettingsComponent["PRINTING"] = 10] = "PRINTING";
    SettingsComponent[SettingsComponent["SECURITY_AND_SIGN_IN"] = 11] = "SECURITY_AND_SIGN_IN";
    SettingsComponent[SettingsComponent["TOUCHPAD_REVERSE_SCROLLING"] = 12] = "TOUCHPAD_REVERSE_SCROLLING";
    SettingsComponent[SettingsComponent["TOUCHPAD_SIMULATE_RIGHT_CLICK"] = 13] = "TOUCHPAD_SIMULATE_RIGHT_CLICK";
})(SettingsComponent || (SettingsComponent = {}));
class PageHandlerFactoryPendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'ash.help_app.mojom.PageHandlerFactory', scope);
    }
}
class PageHandlerFactoryRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(PageHandlerFactoryPendingReceiver, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    createPageHandler(handler) {
        this.proxy.sendMessage(0, PageHandlerFactory_CreatePageHandler_ParamsSpec.$, null, [
            handler
        ], false);
    }
}
class PageHandlerFactory {
    static get $interfaceName() {
        return "ash.help_app.mojom.PageHandlerFactory";
    }
    /**
     * Returns a remote for this interface which sends messages to the browser.
     * The browser must have an interface request binder registered for this
     * interface and accessible to the calling document's frame.
     */
    static getRemote() {
        let remote = new PageHandlerFactoryRemote;
        remote.$.bindNewPipeAndPassReceiver().bindInBrowser();
        return remote;
    }
}
class PageHandlerPendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'ash.help_app.mojom.PageHandler', scope);
    }
}
class PageHandlerRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(PageHandlerPendingReceiver, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    openFeedbackDialog() {
        return this.proxy.sendMessage(0, PageHandler_OpenFeedbackDialog_ParamsSpec.$, PageHandler_OpenFeedbackDialog_ResponseParamsSpec.$, [], false);
    }
    showOnDeviceAppControls() {
        this.proxy.sendMessage(1, PageHandler_ShowOnDeviceAppControls_ParamsSpec.$, null, [], false);
    }
    showParentalControls() {
        this.proxy.sendMessage(2, PageHandler_ShowParentalControls_ParamsSpec.$, null, [], false);
    }
    triggerWelcomeTipCallToAction(actionTypeId) {
        this.proxy.sendMessage(3, PageHandler_TriggerWelcomeTipCallToAction_ParamsSpec.$, null, [
            actionTypeId
        ], false);
    }
    isLauncherSearchEnabled() {
        return this.proxy.sendMessage(4, PageHandler_IsLauncherSearchEnabled_ParamsSpec.$, PageHandler_IsLauncherSearchEnabled_ResponseParamsSpec.$, [], false);
    }
    launchMicrosoft365Setup() {
        this.proxy.sendMessage(5, PageHandler_LaunchMicrosoft365Setup_ParamsSpec.$, null, [], false);
    }
    maybeShowReleaseNotesNotification() {
        this.proxy.sendMessage(6, PageHandler_MaybeShowReleaseNotesNotification_ParamsSpec.$, null, [], false);
    }
    getDeviceInfo() {
        return this.proxy.sendMessage(7, PageHandler_GetDeviceInfo_ParamsSpec.$, PageHandler_GetDeviceInfo_ResponseParamsSpec.$, [], false);
    }
    openUrlInBrowserAndTriggerInstallDialog(url) {
        this.proxy.sendMessage(8, PageHandler_OpenUrlInBrowserAndTriggerInstallDialog_ParamsSpec.$, null, [
            url
        ], false);
    }
    openSettings(component) {
        this.proxy.sendMessage(9, PageHandler_OpenSettings_ParamsSpec.$, null, [
            component
        ], false);
    }
    setHasCompletedNewDeviceChecklist() {
        this.proxy.sendMessage(10, PageHandler_SetHasCompletedNewDeviceChecklist_ParamsSpec.$, null, [], false);
    }
    setHasVisitedHowToPage() {
        this.proxy.sendMessage(11, PageHandler_SetHasVisitedHowToPage_ParamsSpec.$, null, [], false);
    }
}
const DeviceInfoSpec = { $: {} };
const PageHandlerFactory_CreatePageHandler_ParamsSpec = { $: {} };
const PageHandler_OpenFeedbackDialog_ParamsSpec = { $: {} };
const PageHandler_OpenFeedbackDialog_ResponseParamsSpec = { $: {} };
const PageHandler_ShowOnDeviceAppControls_ParamsSpec = { $: {} };
const PageHandler_ShowParentalControls_ParamsSpec = { $: {} };
const PageHandler_TriggerWelcomeTipCallToAction_ParamsSpec = { $: {} };
const PageHandler_IsLauncherSearchEnabled_ParamsSpec = { $: {} };
const PageHandler_IsLauncherSearchEnabled_ResponseParamsSpec = { $: {} };
const PageHandler_LaunchMicrosoft365Setup_ParamsSpec = { $: {} };
const PageHandler_MaybeShowReleaseNotesNotification_ParamsSpec = { $: {} };
const PageHandler_GetDeviceInfo_ParamsSpec = { $: {} };
const PageHandler_GetDeviceInfo_ResponseParamsSpec = { $: {} };
const PageHandler_OpenUrlInBrowserAndTriggerInstallDialog_ParamsSpec = { $: {} };
const PageHandler_OpenSettings_ParamsSpec = { $: {} };
const PageHandler_SetHasCompletedNewDeviceChecklist_ParamsSpec = { $: {} };
const PageHandler_SetHasVisitedHowToPage_ParamsSpec = { $: {} };
mojo.internal.Struct(DeviceInfoSpec.$, 'DeviceInfo', [
    mojo.internal.StructField('board', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('model', 8, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('userType', 16, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('isSteamAllowed', 24, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 40],]);
mojo.internal.Struct(PageHandlerFactory_CreatePageHandler_ParamsSpec.$, 'PageHandlerFactory_CreatePageHandler_Params', [
    mojo.internal.StructField('handler', 0, 0, mojo.internal.InterfaceRequest(PageHandlerPendingReceiver), null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_OpenFeedbackDialog_ParamsSpec.$, 'PageHandler_OpenFeedbackDialog_Params', [], [[0, 8],]);
mojo.internal.Struct(PageHandler_OpenFeedbackDialog_ResponseParamsSpec.$, 'PageHandler_OpenFeedbackDialog_ResponseParams', [
    mojo.internal.StructField('errorMessage', 0, 0, mojo.internal.String, null, true /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_ShowOnDeviceAppControls_ParamsSpec.$, 'PageHandler_ShowOnDeviceAppControls_Params', [], [[0, 8],]);
mojo.internal.Struct(PageHandler_ShowParentalControls_ParamsSpec.$, 'PageHandler_ShowParentalControls_Params', [], [[0, 8],]);
mojo.internal.Struct(PageHandler_TriggerWelcomeTipCallToAction_ParamsSpec.$, 'PageHandler_TriggerWelcomeTipCallToAction_Params', [
    mojo.internal.StructField('actionTypeId', 0, 0, ActionTypeIdSpec.$, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_IsLauncherSearchEnabled_ParamsSpec.$, 'PageHandler_IsLauncherSearchEnabled_Params', [], [[0, 8],]);
mojo.internal.Struct(PageHandler_IsLauncherSearchEnabled_ResponseParamsSpec.$, 'PageHandler_IsLauncherSearchEnabled_ResponseParams', [
    mojo.internal.StructField('enabled', 0, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_LaunchMicrosoft365Setup_ParamsSpec.$, 'PageHandler_LaunchMicrosoft365Setup_Params', [], [[0, 8],]);
mojo.internal.Struct(PageHandler_MaybeShowReleaseNotesNotification_ParamsSpec.$, 'PageHandler_MaybeShowReleaseNotesNotification_Params', [], [[0, 8],]);
mojo.internal.Struct(PageHandler_GetDeviceInfo_ParamsSpec.$, 'PageHandler_GetDeviceInfo_Params', [], [[0, 8],]);
mojo.internal.Struct(PageHandler_GetDeviceInfo_ResponseParamsSpec.$, 'PageHandler_GetDeviceInfo_ResponseParams', [
    mojo.internal.StructField('deviceInfo', 0, 0, DeviceInfoSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_OpenUrlInBrowserAndTriggerInstallDialog_ParamsSpec.$, 'PageHandler_OpenUrlInBrowserAndTriggerInstallDialog_Params', [
    mojo.internal.StructField('url', 0, 0, UrlSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_OpenSettings_ParamsSpec.$, 'PageHandler_OpenSettings_Params', [
    mojo.internal.StructField('component', 0, 0, SettingsComponentSpec.$, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_SetHasCompletedNewDeviceChecklist_ParamsSpec.$, 'PageHandler_SetHasCompletedNewDeviceChecklist_Params', [], [[0, 8],]);
mojo.internal.Struct(PageHandler_SetHasVisitedHowToPage_ParamsSpec.$, 'PageHandler_SetHasVisitedHowToPage_Params', [], [[0, 8],]);

// chromeos/ash/components/local_search_service/public/mojom/types.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const ResponseStatusSpec = { $: mojo.internal.Enum() };
var ResponseStatus;
(function (ResponseStatus) {
    ResponseStatus[ResponseStatus["MIN_VALUE"] = 0] = "MIN_VALUE";
    ResponseStatus[ResponseStatus["MAX_VALUE"] = 3] = "MAX_VALUE";
    ResponseStatus[ResponseStatus["kUnknownError"] = 0] = "kUnknownError";
    ResponseStatus[ResponseStatus["kSuccess"] = 1] = "kSuccess";
    ResponseStatus[ResponseStatus["kEmptyQuery"] = 2] = "kEmptyQuery";
    ResponseStatus[ResponseStatus["kEmptyIndex"] = 3] = "kEmptyIndex";
})(ResponseStatus || (ResponseStatus = {}));
const ContentSpec = { $: {} };
const DataSpec = { $: {} };
const SearchParamsSpec = { $: {} };
const PositionSpec = { $: {} };
const ResultSpec = { $: {} };
mojo.internal.Struct(ContentSpec.$, 'Content', [
    mojo.internal.StructField('id', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('content', 8, 0, String16Spec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('weight', 16, 0, mojo.internal.Double, 1.0, false /* nullable */, 0, undefined, undefined),
], [[0, 32],]);
mojo.internal.Struct(DataSpec.$, 'Data', [
    mojo.internal.StructField('id', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('contents', 8, 0, mojo.internal.Array(ContentSpec.$, false), null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('locale', 16, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 32],]);
mojo.internal.Struct(SearchParamsSpec.$, 'SearchParams', [
    mojo.internal.StructField('relevanceThreshold', 0, 0, mojo.internal.Double, 0.32, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('prefixThreshold', 8, 0, mojo.internal.Double, 0.6, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('fuzzyThreshold', 16, 0, mojo.internal.Double, 0.6, false /* nullable */, 0, undefined, undefined),
], [[0, 32],]);
mojo.internal.Struct(PositionSpec.$, 'Position', [
    mojo.internal.StructField('contentId', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('start', 8, 0, mojo.internal.Uint32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('length', 12, 0, mojo.internal.Uint32, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(ResultSpec.$, 'Result', [
    mojo.internal.StructField('id', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('score', 8, 0, mojo.internal.Double, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('positions', 16, 0, mojo.internal.Array(PositionSpec.$, false), null, false /* nullable */, 0, undefined, undefined),
], [[0, 32],]);

// chromeos/ash/components/local_search_service/public/mojom/index.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class IndexPendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'ash.local_search_service.mojom.Index', scope);
    }
}
class IndexRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(IndexPendingReceiver, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    getSize() {
        return this.proxy.sendMessage(0, Index_GetSize_ParamsSpec.$, Index_GetSize_ResponseParamsSpec.$, [], false);
    }
    addOrUpdate(data) {
        return this.proxy.sendMessage(1, Index_AddOrUpdate_ParamsSpec.$, Index_AddOrUpdate_ResponseParamsSpec.$, [
            data
        ], false);
    }
    delete(ids) {
        return this.proxy.sendMessage(2, Index_Delete_ParamsSpec.$, Index_Delete_ResponseParamsSpec.$, [
            ids
        ], false);
    }
    updateDocuments(data) {
        return this.proxy.sendMessage(3, Index_UpdateDocuments_ParamsSpec.$, Index_UpdateDocuments_ResponseParamsSpec.$, [
            data
        ], false);
    }
    find(query, maxResults) {
        return this.proxy.sendMessage(4, Index_Find_ParamsSpec.$, Index_Find_ResponseParamsSpec.$, [
            query,
            maxResults
        ], false);
    }
    clearIndex() {
        return this.proxy.sendMessage(5, Index_ClearIndex_ParamsSpec.$, Index_ClearIndex_ResponseParamsSpec.$, [], false);
    }
    setSearchParams(searchParams) {
        return this.proxy.sendMessage(6, Index_SetSearchParams_ParamsSpec.$, Index_SetSearchParams_ResponseParamsSpec.$, [
            searchParams
        ], false);
    }
}
class Index {
    static get $interfaceName() {
        return "ash.local_search_service.mojom.Index";
    }
    /**
     * Returns a remote for this interface which sends messages to the browser.
     * The browser must have an interface request binder registered for this
     * interface and accessible to the calling document's frame.
     */
    static getRemote() {
        let remote = new IndexRemote;
        remote.$.bindNewPipeAndPassReceiver().bindInBrowser();
        return remote;
    }
}
const Index_GetSize_ParamsSpec = { $: {} };
const Index_GetSize_ResponseParamsSpec = { $: {} };
const Index_AddOrUpdate_ParamsSpec = { $: {} };
const Index_AddOrUpdate_ResponseParamsSpec = { $: {} };
const Index_Delete_ParamsSpec = { $: {} };
const Index_Delete_ResponseParamsSpec = { $: {} };
const Index_UpdateDocuments_ParamsSpec = { $: {} };
const Index_UpdateDocuments_ResponseParamsSpec = { $: {} };
const Index_Find_ParamsSpec = { $: {} };
const Index_Find_ResponseParamsSpec = { $: {} };
const Index_ClearIndex_ParamsSpec = { $: {} };
const Index_ClearIndex_ResponseParamsSpec = { $: {} };
const Index_SetSearchParams_ParamsSpec = { $: {} };
const Index_SetSearchParams_ResponseParamsSpec = { $: {} };
mojo.internal.Struct(Index_GetSize_ParamsSpec.$, 'Index_GetSize_Params', [], [[0, 8],]);
mojo.internal.Struct(Index_GetSize_ResponseParamsSpec.$, 'Index_GetSize_ResponseParams', [
    mojo.internal.StructField('numItems', 0, 0, mojo.internal.Uint64, BigInt(0), false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(Index_AddOrUpdate_ParamsSpec.$, 'Index_AddOrUpdate_Params', [
    mojo.internal.StructField('data', 0, 0, mojo.internal.Array(DataSpec.$, false), null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(Index_AddOrUpdate_ResponseParamsSpec.$, 'Index_AddOrUpdate_ResponseParams', [], [[0, 8],]);
mojo.internal.Struct(Index_Delete_ParamsSpec.$, 'Index_Delete_Params', [
    mojo.internal.StructField('ids', 0, 0, mojo.internal.Array(mojo.internal.String, false), null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(Index_Delete_ResponseParamsSpec.$, 'Index_Delete_ResponseParams', [
    mojo.internal.StructField('numDeleted', 0, 0, mojo.internal.Uint32, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(Index_UpdateDocuments_ParamsSpec.$, 'Index_UpdateDocuments_Params', [
    mojo.internal.StructField('data', 0, 0, mojo.internal.Array(DataSpec.$, false), null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(Index_UpdateDocuments_ResponseParamsSpec.$, 'Index_UpdateDocuments_ResponseParams', [
    mojo.internal.StructField('numDeleted', 0, 0, mojo.internal.Uint32, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(Index_Find_ParamsSpec.$, 'Index_Find_Params', [
    mojo.internal.StructField('query', 0, 0, String16Spec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('maxResults', 8, 0, mojo.internal.Uint32, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(Index_Find_ResponseParamsSpec.$, 'Index_Find_ResponseParams', [
    mojo.internal.StructField('status', 0, 0, ResponseStatusSpec.$, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('results', 8, 0, mojo.internal.Array(ResultSpec.$, false), null, true /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(Index_ClearIndex_ParamsSpec.$, 'Index_ClearIndex_Params', [], [[0, 8],]);
mojo.internal.Struct(Index_ClearIndex_ResponseParamsSpec.$, 'Index_ClearIndex_ResponseParams', [], [[0, 8],]);
mojo.internal.Struct(Index_SetSearchParams_ParamsSpec.$, 'Index_SetSearchParams_Params', [
    mojo.internal.StructField('searchParams', 0, 0, SearchParamsSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(Index_SetSearchParams_ResponseParamsSpec.$, 'Index_SetSearchParams_ResponseParams', [], [[0, 8],]);

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview
 * Message definitions passed over the HelpApp privileged/unprivileged pipe.
 */
/** Enum for message types. */
var Message;
(function (Message) {
    Message["OPEN_FEEDBACK_DIALOG"] = "open-feedback-dialog";
    Message["SHOW_ON_DEVICE_APP_CONTROLS"] = "show-on-device-app-controls";
    Message["SHOW_PARENTAL_CONTROLS"] = "show-parental-controls";
    Message["TRIGGER_WELCOME_TIP_CALL_TO_ACTION"] = "trigger-welcome-tip-call-to-action";
    Message["ADD_OR_UPDATE_SEARCH_INDEX"] = "add-or-update-search-index";
    Message["CLEAR_SEARCH_INDEX"] = "clear-search-index";
    Message["FIND_IN_SEARCH_INDEX"] = "find-in-search-index";
    Message["CLOSE_BACKGROUND_PAGE"] = "close-background-page";
    Message["UPDATE_LAUNCHER_SEARCH_INDEX"] = "update-launcher-search-index";
    Message["LAUNCH_MICROSOFT_365_SETUP"] = "launch-microsoft-365-setup";
    Message["MAYBE_SHOW_RELEASE_NOTES_NOTIFICATION"] = "maybe-show-release-notes-notification";
    Message["GET_DEVICE_INFO"] = "get-device-info";
    Message["OPEN_SETTINGS"] = "open-settings";
    Message["OPEN_URL_IN_BROWSER_AND_TRIGGER_INSTALL_DIALOG"] = "open-url-in-browser-and-trigger-install-dialog";
    Message["SET_HAS_COMPLETED_NEW_DEVICE_CHECKLIST"] = "set-has-completed-new-device-checklist";
    Message["SET_HAS_VISITED_HOW_TO_PAGE"] = "set-has-visited-how-to-page";
    Message["OPEN_APP_MALL_PATH"] = "open-app-mall-path";
})(Message || (Message = {}));

// ash/webui/help_app_ui/search/search.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class SearchResultsObserverPendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'ash.help_app.mojom.SearchResultsObserver', scope);
    }
}
class SearchResultsObserverRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(SearchResultsObserverPendingReceiver, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    onSearchResultAvailabilityChanged() {
        this.proxy.sendMessage(0, SearchResultsObserver_OnSearchResultAvailabilityChanged_ParamsSpec.$, null, [], false);
    }
}
class SearchHandlerPendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'ash.help_app.mojom.SearchHandler', scope);
    }
}
class SearchHandlerRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(SearchHandlerPendingReceiver, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    search(query, maxNumResults) {
        return this.proxy.sendMessage(0, SearchHandler_Search_ParamsSpec.$, SearchHandler_Search_ResponseParamsSpec.$, [
            query,
            maxNumResults
        ], false);
    }
    update(concepts) {
        return this.proxy.sendMessage(1, SearchHandler_Update_ParamsSpec.$, SearchHandler_Update_ResponseParamsSpec.$, [
            concepts
        ], false);
    }
    observe(observer) {
        this.proxy.sendMessage(2, SearchHandler_Observe_ParamsSpec.$, null, [
            observer
        ], false);
    }
}
class SearchHandler {
    static get $interfaceName() {
        return "ash.help_app.mojom.SearchHandler";
    }
    /**
     * Returns a remote for this interface which sends messages to the browser.
     * The browser must have an interface request binder registered for this
     * interface and accessible to the calling document's frame.
     */
    static getRemote() {
        let remote = new SearchHandlerRemote;
        remote.$.bindNewPipeAndPassReceiver().bindInBrowser();
        return remote;
    }
}
const SearchResultSpec = { $: {} };
const SearchConceptSpec = { $: {} };
const SearchResultsObserver_OnSearchResultAvailabilityChanged_ParamsSpec = { $: {} };
const SearchHandler_Search_ParamsSpec = { $: {} };
const SearchHandler_Search_ResponseParamsSpec = { $: {} };
const SearchHandler_Update_ParamsSpec = { $: {} };
const SearchHandler_Update_ResponseParamsSpec = { $: {} };
const SearchHandler_Observe_ParamsSpec = { $: {} };
mojo.internal.Struct(SearchResultSpec.$, 'SearchResult', [
    mojo.internal.StructField('id', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('title', 8, 0, String16Spec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('mainCategory', 16, 0, String16Spec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('urlPathWithParameters', 24, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('locale', 32, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('relevanceScore', 40, 0, mojo.internal.Double, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 56],]);
mojo.internal.Struct(SearchConceptSpec.$, 'SearchConcept', [
    mojo.internal.StructField('id', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('title', 8, 0, String16Spec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('mainCategory', 16, 0, String16Spec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('tags', 24, 0, mojo.internal.Array(String16Spec.$, false), null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('tagLocale', 32, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('urlPathWithParameters', 40, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('locale', 48, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 64],]);
mojo.internal.Struct(SearchResultsObserver_OnSearchResultAvailabilityChanged_ParamsSpec.$, 'SearchResultsObserver_OnSearchResultAvailabilityChanged_Params', [], [[0, 8],]);
mojo.internal.Struct(SearchHandler_Search_ParamsSpec.$, 'SearchHandler_Search_Params', [
    mojo.internal.StructField('query', 0, 0, String16Spec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('maxNumResults', 8, 0, mojo.internal.Uint32, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(SearchHandler_Search_ResponseParamsSpec.$, 'SearchHandler_Search_ResponseParams', [
    mojo.internal.StructField('results', 0, 0, mojo.internal.Array(SearchResultSpec.$, false), null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(SearchHandler_Update_ParamsSpec.$, 'SearchHandler_Update_Params', [
    mojo.internal.StructField('concepts', 0, 0, mojo.internal.Array(SearchConceptSpec.$, false), null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(SearchHandler_Update_ResponseParamsSpec.$, 'SearchHandler_Update_ResponseParams', [], [[0, 8],]);
mojo.internal.Struct(SearchHandler_Observe_ParamsSpec.$, 'SearchHandler_Observe_Params', [
    mojo.internal.StructField('observer', 0, 0, mojo.internal.InterfaceProxy(SearchResultsObserverRemote), null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const helpApp = {
    handler: new PageHandlerRemote(),
};
// Set up a page handler to talk to the browser process.
PageHandlerFactory.getRemote().createPageHandler(helpApp.handler.$.bindNewPipeAndPassReceiver());
// Set up an index remote to talk to Local Search Service.
const indexRemote = Index.getRemote();
// Expose `indexRemote` on `window`, because it is accessed by a CrOS Tast test.
Object.assign(window, { indexRemote });
/**
 * Talks to the search handler. Use for updating the content for launcher
 * search.
 */
const searchHandlerRemote = SearchHandler.getRemote();
const GUEST_ORIGIN = 'chrome-untrusted://help-app';
const MAX_STRING_LEN = 9999;
const guestFrame = document.createElement('iframe');
guestFrame.src = `${GUEST_ORIGIN}${location.pathname}${location.search}`;
document.body.appendChild(guestFrame);
// Cached result of whether Launcher Search is enabled.
const isLauncherSearchEnabled = helpApp.handler.isLauncherSearchEnabled().then(result => result.enabled);
/** Converts a string or object to url. */
function toUrl(url) {
    // TODO(b/279132899): Figure out why `url` is an empty object when it should
    // have been an empty string.
    if (url === '' || typeof (url) !== 'string') {
        return { url: '' };
    }
    return { url };
}
const TITLE_ID = 'title';
const BODY_ID = 'body';
const CATEGORY_ID = 'main-category';
const SUBCATEGORY_ID = 'subcategory';
const SUBHEADING_ID = 'subheading';
/**
 * A pipe through which we can send messages to the guest frame.
 * Use an undefined `target` to find the <iframe> automatically.
 * Do not rethrow errors, since handlers installed here are expected to
 * throw exceptions that are handled on the other side of the pipe (in the guest
 * frame), not on this side.
 */
const guestMessagePipe = new MessagePipe('chrome-untrusted://help-app', /*target=*/ undefined, 
/*rethrowErrors=*/ false);
guestMessagePipe.registerHandler(Message.OPEN_FEEDBACK_DIALOG, () => {
    return helpApp.handler.openFeedbackDialog();
});
guestMessagePipe.registerHandler(Message.SHOW_ON_DEVICE_APP_CONTROLS, () => void helpApp.handler.showOnDeviceAppControls());
guestMessagePipe.registerHandler(Message.SHOW_PARENTAL_CONTROLS, () => void helpApp.handler.showParentalControls());
guestMessagePipe.registerHandler(Message.TRIGGER_WELCOME_TIP_CALL_TO_ACTION, (actionTypeId) => void helpApp.handler.triggerWelcomeTipCallToAction(actionTypeId));
guestMessagePipe.registerHandler(Message.ADD_OR_UPDATE_SEARCH_INDEX, async (data) => {
    const dataToSend = data.map(searchableItem => {
        const contents = [
            {
                id: TITLE_ID,
                content: truncate(searchableItem.title),
                weight: 1.0,
            },
            {
                id: CATEGORY_ID,
                content: truncate(searchableItem.mainCategoryName),
                weight: 0.1,
            },
        ];
        if (searchableItem.subcategoryNames) {
            for (let i = 0; i < searchableItem.subcategoryNames.length; ++i) {
                const subcategoryName = searchableItem.subcategoryNames[i];
                contents.push({
                    id: SUBCATEGORY_ID + i,
                    content: truncate(subcategoryName),
                    weight: 0.1,
                });
            }
        }
        // If there are subheadings, use those instead of the body.
        const subheadings = searchableItem.subheadings;
        if (subheadings) {
            for (let i = 0; i < subheadings.length; ++i) {
                const subheading = subheadings[i];
                if (!subheading)
                    continue;
                contents.push({
                    id: SUBHEADING_ID + i,
                    content: truncate(subheading),
                    weight: 0.4,
                });
            }
        }
        else if (searchableItem.body) {
            contents.push({
                id: BODY_ID,
                content: truncate(searchableItem.body),
                weight: 0.2,
            });
        }
        return {
            id: searchableItem.id,
            contents,
            locale: searchableItem.locale,
        };
    });
    return indexRemote.addOrUpdate(dataToSend);
});
guestMessagePipe.registerHandler(Message.CLEAR_SEARCH_INDEX, async () => {
    return indexRemote.clearIndex();
});
guestMessagePipe.registerHandler(Message.FIND_IN_SEARCH_INDEX, async (dataFromApp) => {
    const response = await indexRemote.find(truncate(dataFromApp.query), dataFromApp.maxResults || 50);
    if (response.status !== ResponseStatus.kSuccess || !response.results) {
        return { results: null };
    }
    const searchResults = (response.results);
    // Sort results by decreasing score.
    searchResults.sort((a, b) => b.score - a.score);
    const results = searchResults.map(result => {
        const titlePositions = [];
        const bodyPositions = [];
        // Id of the best subheading that appears in positions. We consider
        // the subheading containing the most match positions to be the best.
        // "" means no subheading positions found.
        let bestSubheadingId = '';
        /** Counts how many positions there are for each subheading id. */
        const subheadingPosCounts = {};
        // Note: result.positions is not sorted.
        for (const position of result.positions) {
            if (position.contentId === TITLE_ID) {
                titlePositions.push({ length: position.length, start: position.start });
            }
            else if (position.contentId === BODY_ID) {
                bodyPositions.push({ length: position.length, start: position.start });
            }
            else if (position.contentId.startsWith(SUBHEADING_ID)) {
                // Update the subheadings's position count and check if it's the new
                // best subheading.
                const newCount = (subheadingPosCounts[position.contentId] || 0) + 1;
                subheadingPosCounts[position.contentId] = newCount;
                if (!bestSubheadingId ||
                    newCount > (subheadingPosCounts[bestSubheadingId] ?? 0)) {
                    bestSubheadingId = position.contentId;
                }
            }
        }
        // Use only the positions of the best subheading.
        const subheadingPositions = [];
        if (bestSubheadingId) {
            for (const position of result.positions) {
                if (position.contentId === bestSubheadingId) {
                    subheadingPositions.push({
                        start: position.start,
                        length: position.length,
                    });
                }
            }
            subheadingPositions.sort(compareByStart);
        }
        // Sort positions by start index.
        titlePositions.sort(compareByStart);
        bodyPositions.sort(compareByStart);
        return {
            id: result.id,
            titlePositions,
            bodyPositions,
            subheadingIndex: bestSubheadingId ?
                Number(bestSubheadingId.substring(SUBHEADING_ID.length)) : null,
            subheadingPositions: bestSubheadingId ? subheadingPositions : null,
        };
    });
    return { results };
});
guestMessagePipe.registerHandler(Message.CLOSE_BACKGROUND_PAGE, async () => {
    // TODO(b/186180962): Add background page and test that it closes when done.
    if (window.location.pathname !== '/background') {
        return;
    }
    window.close();
    return;
});
guestMessagePipe.registerHandler(Message.UPDATE_LAUNCHER_SEARCH_INDEX, async (message) => {
    if (!(await isLauncherSearchEnabled)) {
        return;
    }
    const dataToSend = message.map(searchableItem => ({
        id: truncate(searchableItem.id),
        title: truncate(searchableItem.title),
        mainCategory: truncate(searchableItem.mainCategoryName),
        tags: searchableItem.tags.map(tag => truncate(tag))
            .filter(tag => tag.length > 0),
        tagLocale: searchableItem.tagLocale || '',
        urlPathWithParameters: truncate(searchableItem.urlPathWithParameters),
        locale: truncate(searchableItem.locale),
    }));
    // Filter out invalid items. No field can be empty except locale.
    const dataFiltered = dataToSend.filter(item => {
        const valid = item.id && item.title && item.mainCategory &&
            item.tags.length > 0 && item.urlPathWithParameters;
        // This is a google-internal histogram. If changing this, also change
        // the corresponding histograms file.
        if (!valid) {
            window.chrome.metricsPrivate
                .recordSparseValueWithPersistentHash('Discover.LauncherSearch.InvalidConceptInUpdate', item.id);
        }
        return valid;
    });
    return searchHandlerRemote.update(dataFiltered);
});
guestMessagePipe.registerHandler(Message.LAUNCH_MICROSOFT_365_SETUP, () => void helpApp.handler.launchMicrosoft365Setup());
guestMessagePipe.registerHandler(Message.MAYBE_SHOW_RELEASE_NOTES_NOTIFICATION, () => void helpApp.handler.maybeShowReleaseNotesNotification());
guestMessagePipe.registerHandler(Message.GET_DEVICE_INFO, async () => {
    return (await helpApp.handler.getDeviceInfo()).deviceInfo;
});
guestMessagePipe.registerHandler(Message.OPEN_SETTINGS, (path) => void helpApp.handler.openSettings(path));
guestMessagePipe.registerHandler(Message.OPEN_URL_IN_BROWSER_AND_TRIGGER_INSTALL_DIALOG, (url) => {
    helpApp.handler.openUrlInBrowserAndTriggerInstallDialog(toUrl(url));
});
guestMessagePipe.registerHandler(Message.SET_HAS_COMPLETED_NEW_DEVICE_CHECKLIST, () => void helpApp.handler.setHasCompletedNewDeviceChecklist());
guestMessagePipe.registerHandler(Message.SET_HAS_VISITED_HOW_TO_PAGE, () => void helpApp.handler.setHasVisitedHowToPage());
guestMessagePipe.registerHandler(Message.OPEN_APP_MALL_PATH, ({ path }) => {
    window.open(`chrome://mall/${path}`);
});
/** Compare two positions by their start index. Use for sorting. */
function compareByStart(a, b) {
    return a.start - b.start;
}
/**
 * Limits the maximum length of the input string. Converts non-strings into
 * empty string.
 *
 * @param s Probably a string, but might not be.
 */
function truncate(s) {
    if (typeof s !== 'string') {
        return '';
    }
    if (s.length <= MAX_STRING_LEN) {
        return s;
    }
    return s.substring(0, MAX_STRING_LEN);
}
const TEST_ONLY = { guestMessagePipe };

export { TEST_ONLY };
//# sourceMappingURL=browser_proxy.rollup.js.map
