import { mojo } from '//resources/mojo/mojo/public/js/bindings.js';

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

/**
 * @fileoverview Minimal version of load_time_data.js for chrome-untrusted://
 * origins. They are sandboxed, so cannot use chrome://resources ("unable to
 * load local resource") which load_time_data.js relies on through strings.js.
 * Since we don't want to maintain a "mirror" of all the module dependencies on
 * each chrome-untrusted:// origin. For simplicity, this version lacks all the
 * validation done by load_time_data.js, and just aims to provide a compatible
 * API.
 */

const impl = {
  /**
   * Needs to be separate from data since some tast tests expect a data_ field.
   * TODO(b/192977700): Remove this.
   * @type {!Object}
   */
  data_: {},
  /**
   * Used by strings.js to populate loadTimeData.
   * Note we don't provide a getter since the original load_time_data object
   * interface explicitly disallows reading the data object directly.
   * @param {!Object} value
   */
  set data(value) {
    impl.data_ = value;
  },
  getValue: (id) => impl.data_[id],
  getString: (id) => /** @type{string} */ (impl.data_[id]),
  getBoolean: (id) => /** @type{boolean} */ (impl.data_[id]),
  getInteger: (id) => /** @type{number} */ (impl.data_[id]),
  valueExists: (id) => impl.data_[id] !== undefined,
};
window['loadTimeData'] = impl;

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Verify |value| is truthy.
 * @param value A value to check for truthiness. Note that this
 *     may be used to test whether |value| is defined or not, and we don't want
 *     to force a cast to boolean.
 */
function assert(value, message) {
    if (value) {
        return;
    }
    throw new Error('Assertion failed' + (''));
}

// ui/webui/resources/cr_components/color_change_listener/color_change_listener.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class PageHandlerPendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'color_change_listener.mojom.PageHandler', scope);
    }
}
class PageHandlerRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(PageHandlerPendingReceiver, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    setPage(page) {
        this.proxy.sendMessage(0, PageHandler_SetPage_ParamsSpec.$, null, [
            page
        ], false);
    }
}
class PageHandler {
    static get $interfaceName() {
        return "color_change_listener.mojom.PageHandler";
    }
    /**
     * Returns a remote for this interface which sends messages to the browser.
     * The browser must have an interface request binder registered for this
     * interface and accessible to the calling document's frame.
     */
    static getRemote() {
        let remote = new PageHandlerRemote;
        remote.$.bindNewPipeAndPassReceiver().bindInBrowser();
        return remote;
    }
}
class PagePendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'color_change_listener.mojom.Page', scope);
    }
}
class PageRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(PagePendingReceiver, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    onColorProviderChanged() {
        this.proxy.sendMessage(0, Page_OnColorProviderChanged_ParamsSpec.$, null, [], false);
    }
}
/**
 * An object which receives request messages for the Page
 * mojom interface and dispatches them as callbacks. One callback receiver exists
 * on this object for each message defined in the mojom interface, and each
 * receiver can have any number of listeners added to it.
 */
class PageCallbackRouter {
    helper_internal_;
    $;
    router_;
    onColorProviderChanged;
    onConnectionError;
    constructor() {
        this.helper_internal_ = new mojo.internal.interfaceSupport.InterfaceReceiverHelperInternal(PageRemote);
        this.$ = new mojo.internal.interfaceSupport.InterfaceReceiverHelper(this.helper_internal_);
        this.router_ = new mojo.internal.interfaceSupport.CallbackRouter;
        this.onColorProviderChanged =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(0, Page_OnColorProviderChanged_ParamsSpec.$, null, this.onColorProviderChanged.createReceiverHandler(false /* expectsResponse */), false);
        this.onConnectionError = this.helper_internal_.getConnectionErrorEventRouter();
    }
    /**
     * @param id An ID returned by a prior call to addListener.
     * @return True iff the identified listener was found and removed.
     */
    removeListener(id) {
        return this.router_.removeListener(id);
    }
}
const PageHandler_SetPage_ParamsSpec = { $: {} };
const Page_OnColorProviderChanged_ParamsSpec = { $: {} };
mojo.internal.Struct(PageHandler_SetPage_ParamsSpec.$, 'PageHandler_SetPage_Params', [
    mojo.internal.StructField('page', 0, 0, mojo.internal.InterfaceProxy(PageRemote), null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(Page_OnColorProviderChanged_ParamsSpec.$, 'Page_OnColorProviderChanged_Params', [], [[0, 8],]);

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview This file provides a singleton class that exposes the Mojo
 * handler interface used for one way communication between the JS and the
 * browser.
 * TODO(tluk): Convert this into typescript once all dependencies have been
 * fully migrated.
 */
let instance = null;
class BrowserProxy {
    callbackRouter;
    constructor() {
        this.callbackRouter = new PageCallbackRouter();
        const pageHandlerRemote = PageHandler.getRemote();
        pageHandlerRemote.setPage(this.callbackRouter.$.bindNewPipeAndPassRemote());
    }
    static getInstance() {
        return instance || (instance = new BrowserProxy());
    }
    static setInstance(newInstance) {
        instance = newInstance;
    }
}

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview This file holds the functions that allow WebUI to update its
 * colors CSS stylesheet when a ColorProvider change in the browser is detected.
 */
/**
 * The CSS selector used to get the <link> node with the colors.css stylesheet.
 * The wildcard is needed since the URL ends with a timestamp.
 */
const COLORS_CSS_SELECTOR = 'link[href*=\'//theme/colors.css\']';
let documentInstance = null;
// 
// Event fired after updated colors have been fetched and applied.
const COLOR_PROVIDER_CHANGED = 'color-provider-changed';
// 
class ColorChangeUpdater {
    listenerId_ = null;
    root_;
    // 
    eventTarget = new EventTarget();
    // 
    constructor(root) {
        assert(documentInstance === null || root !== document);
        this.root_ = root;
    }
    /**
     * Starts listening for ColorProvider changes from the browser and updates the
     * `root_` whenever changes occur.
     */
    start() {
        if (this.listenerId_ !== null) {
            return;
        }
        this.listenerId_ = BrowserProxy.getInstance()
            .callbackRouter.onColorProviderChanged.addListener(this.onColorProviderChanged.bind(this));
    }
    // TODO(dpapad): Figure out how to properly trigger
    // `callbackRouter.onColorProviderChanged` listeners from tests and make this
    // method private.
    async onColorProviderChanged() {
        await this.refreshColorsCss();
        // 
        this.eventTarget.dispatchEvent(new CustomEvent(COLOR_PROVIDER_CHANGED));
        // 
    }
    /**
     * Forces `root_` to refresh its colors.css stylesheet. This is used to
     * fetch an updated stylesheet when the ColorProvider associated with the
     * WebUI has changed.
     * @return A promise which resolves to true once the new colors are loaded and
     *     installed into the DOM. In the case of an error returns false. When a
     *     new colors.css is loaded, this will always freshly query the existing
     *     colors.css, allowing multiple calls to successfully remove existing,
     *     outdated CSS.
     */
    async refreshColorsCss() {
        const colorCssNode = this.root_.querySelector(COLORS_CSS_SELECTOR);
        if (!colorCssNode) {
            return false;
        }
        const href = colorCssNode.getAttribute('href');
        if (!href) {
            return false;
        }
        const hrefURL = new URL(href, location.href);
        const params = new URLSearchParams(hrefURL.search);
        params.set('version', new Date().getTime().toString());
        const newHref = `${hrefURL.origin}${hrefURL.pathname}?${params.toString()}`;
        // A flickering effect may take place when setting the href property of
        // the existing color css node with a new value. In order to avoid
        // flickering, we create a new link element and once it is loaded we
        // remove the old one. See crbug.com/1365320 for additional details.
        const newColorsCssLink = document.createElement('link');
        newColorsCssLink.setAttribute('href', newHref);
        newColorsCssLink.rel = 'stylesheet';
        newColorsCssLink.type = 'text/css';
        const newColorsLoaded = new Promise(resolve => {
            newColorsCssLink.onload = resolve;
        });
        if (this.root_ === document) {
            document.getElementsByTagName('body')[0].appendChild(newColorsCssLink);
        }
        else {
            this.root_.appendChild(newColorsCssLink);
        }
        await newColorsLoaded;
        const oldColorCssNode = document.querySelector(COLORS_CSS_SELECTOR);
        if (oldColorCssNode) {
            oldColorCssNode.remove();
        }
        return true;
    }
    static forDocument() {
        return documentInstance ||
            (documentInstance = new ColorChangeUpdater(document));
    }
}

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * The Object placed in MessageData.message (and thrown by the Promise returned
 * by sendMessage) if an exception is caught on the receiving end.
 * Note this must be a class (not an interface) whilst there are .js files
 * importing `GenericErrorResponse`, otherwise the export is invisible.
 */
/**
 * To handle generic errors such as `DOMException` not being an `Error`
 * defensively assign '' if the attribute is undefined. Without explicitly
 * extracting fields, `Errors` are sent as `{}` across the pipe.
 */
function serializeError(error) {
    return {
        message: error.message || '',
        name: error.name || '',
        stack: error.stack || '',
    };
}
/**
 * Creates a new JavaScript native Promise and captures its resolve and reject
 * callbacks. The promise, resolve, and reject are available as properties.
 * Inspired by goog.promise.NativeResolver.
 */
class NativeResolver {
    resolve;
    reject;
    promise;
    constructor() {
        this.promise = new Promise((resolve, reject) => {
            this.resolve = resolve;
            this.reject = reject;
        });
    }
}
/**
 * A simplified "assert" that casts away null types. Assumes preconditions that
 * satisfy the assert have already been checked.
 * TODO(b/150650426): consolidate this better.
 */
function assertCast(condition) {
    if (!condition) {
        throw new Error('Failed assertion');
    }
    return condition;
}
/**
 * Enum for reserved message types used in generated messages.
 */
var ReservedMessageTypes;
(function (ReservedMessageTypes) {
    /**
     * Indicates a autogenerated response message for a previously received
     * message.
     */
    ReservedMessageTypes["RESPONSE_TYPE"] = "___response";
    /**
     * Indicates a autogenerated error message for a previously received
     * message.
     */
    ReservedMessageTypes["ERROR_TYPE"] = "___error";
})(ReservedMessageTypes || (ReservedMessageTypes = {}));
/**
 * Checks if a provided message type indicates a generated message.
 */
function isGeneratedMessage(messageType) {
    // Any message type with three underscores before it should only be used
    // in generated messages.
    return messageType.substr(0, 3) === '___';
}
/**
 * Checks a message type is not reserved by generated messages, if it is, throws
 * a error indicating this to the user.
 */
function throwIfReserved(messageType) {
    if (isGeneratedMessage(messageType)) {
        throw new Error(`Unexpected reserved message type: '${messageType}'`);
    }
}
/**
 * The message pipe allows two windows to communicate in 1 direction without
 * having to handle the internals. The caller can send messages to the other
 * window and receive async responses.
 */
class MessagePipe {
    target_;
    targetOrigin_;
    /**
     * If true any errors thrown in a handler during message handling will be
     * thrown again in addition to being sent over the pipe to the message
     * sender. true by default.
     */
    rethrowErrors;
    /**
     * Client error logger. Mockable for tests that check for errors. This is
     * only used to log errors generated from handlers. Logging occurs on both
     * sides of the message pipe if rethrowErrors is set, otherwise only on
     * the side that sent the message.
     */
    logClientError = (object) => console.error(JSON.stringify(object));
    /**
     * Maps a message type to a message handler, a function which takes in
     * the message and returns a response message or a promise which resolves
     * with a response message.
     */
    messageHandlers_ = new Map();
    /**
     * Maps a message id to a resolver.
     */
    pendingMessages_ = new Map();
    /**
     * The id the next message the object sends will have.
     */
    nextMessageId_ = 0;
    /**
     * The message listener we attach to the window. We need a reference to the
     * function for later removal.
     */
    messageListener_ = (m) => this.receiveMessage_(m);
    /**
     * Constructs a new message pipe to the `target` window which has the
     * `targetOrigin` origin.
     *
     * @param target If not specified, the document tree will be
     *     queried for a iframe with src `targetOrigin` to target.
     */
    constructor(targetOrigin, target, rethrowErrors = true) {
        if (!target) {
            const frame = document.querySelector(`iframe[src^='${targetOrigin}']`);
            if (!frame || !frame.contentWindow) {
                throw new Error('Unable to locate target content window.');
            }
            target = assertCast(frame.contentWindow);
        }
        this.target_ = target;
        this.targetOrigin_ = targetOrigin;
        this.rethrowErrors = rethrowErrors;
        // Make sure we aren't trying to send messages to ourselves.
        console.assert(this.target_ !== window, 'target !== window');
        window.addEventListener('message', this.messageListener_);
    }
    /**
     * Registers a handler to be called when a message of type `messageType` is
     * received. The return value of this handler will automatically be sent to
     * the message source as a response message. If the handler should throw an
     * error while handling a message, the error message will be caught and sent
     * to the message source automatically.
     * NOTE: The message type can not be prefixed with 3 underscores as that is
     * reserved for generated messages. i.e `___hello` is disallowed.
     *
     */
    registerHandler(messageType, handler) {
        throwIfReserved(messageType);
        if (this.messageHandlers_.has(messageType)) {
            throw new Error(`A handler already exists for ${messageType}`);
        }
        this.messageHandlers_.set(messageType, handler);
    }
    /**
     * Wraps `sendMessageImpl()` catching errors from the target context to throw
     * more useful errors with the current context stacktrace attached.
     */
    async sendMessage(messageType, message = {}) {
        try {
            return await this.sendMessageImpl(messageType, message);
        }
        catch (errorResponse) {
            // Create an error with the name of the IPC function invoked, append the
            // stacktrace from the target context (origin of the error) with the
            // stacktrace of the current context.
            const error = new Error(`${messageType}: ${errorResponse.message}`);
            error.name = errorResponse.name || 'Unknown Error';
            error.stack +=
                `\nError from ${this.targetOrigin_}\n${errorResponse.stack}`;
            // TODO(b/156205603): use internal `chrome.crashReportPrivate.reportError`
            // to log this error.
            throw error;
        }
    }
    /**
     * Sends a message to the target window and return a Promise that will resolve
     * on response. If the target handler does not send a response the promise
     * will resolve with a empty object.
     */
    async sendMessageImpl(messageType, message = {}) {
        throwIfReserved(messageType);
        const messageId = this.nextMessageId_++;
        const resolver = new NativeResolver();
        this.pendingMessages_.set(messageId, resolver);
        this.postToTarget_(messageType, message, messageId);
        return resolver.promise;
    }
    /**
     * Removes all listeners this object attaches to window in preparation for
     * destruction.
     */
    detach() {
        window.removeEventListener('message', this.messageListener_);
    }
    /**
     * Handles a message which represents the targets response to a previously
     * sent message.
     */
    handleMessageResponse_(messageType, message, messageId) {
        const { RESPONSE_TYPE, ERROR_TYPE } = ReservedMessageTypes;
        const resolver = assertCast(this.pendingMessages_.get(messageId));
        if (messageType === RESPONSE_TYPE) {
            resolver.resolve(message);
        }
        else if (messageType === ERROR_TYPE) {
            this.logClientError(message);
            resolver.reject(message);
        }
        else {
            console.error(`Response for message ${messageId} received with invalid message type ${messageType}`);
        }
        this.pendingMessages_.delete(messageId);
    }
    /**
     * Calls the relevant handler for a received message and generates the right
     * response message to send back to the source.
     */
    async callHandlerForMessageType_(messageType, message, messageId) {
        const { RESPONSE_TYPE, ERROR_TYPE } = ReservedMessageTypes;
        let response;
        let error = null;
        let sawError = false;
        try {
            const handler = assertCast(this.messageHandlers_.get(messageType));
            response = await handler(message);
        }
        catch (err) {
            // If an error happened capture the error and send it back.
            sawError = true;
            error = err;
            response = serializeError(err);
        }
        this.postToTarget_(sawError ? ERROR_TYPE : RESPONSE_TYPE, response, messageId);
        if (sawError && this.rethrowErrors) {
            // Rethrow the error so the current frame has visibility on its handler
            // failures.
            this.logClientError(error);
            throw error;
        }
    }
    receiveMessage_(e) {
        // Ignore message events missing a type.
        if (typeof e.data !== 'object' || !e.data ||
            typeof e.data.type !== 'string') {
            return;
        }
        const { messageId, type, message } = e.data;
        const { ERROR_TYPE } = ReservedMessageTypes;
        // Ignore any messages that are not from the target origin unless we are
        // explicitly accepting messages from any origin.
        if (e.origin !== this.targetOrigin_ && this.targetOrigin_ !== '*') {
            return;
        }
        // The case that the message is a response to a previously sent message.
        if (isGeneratedMessage(type) && this.pendingMessages_.has(messageId)) {
            this.handleMessageResponse_(type, message, messageId);
            return;
        }
        if (isGeneratedMessage(type)) {
            // Currently all generated messages are only sent in a response, so should
            // have been handled above.
            console.error(`Response with type ${type} for unknown message received.`);
            return;
        }
        if (!this.messageHandlers_.has(type)) {
            // If there is no listener for this event send a error message to source.
            const error = new Error(`No handler registered for message type '${type}'`);
            const errorResponse = serializeError(error);
            this.postToTarget_(ERROR_TYPE, errorResponse, messageId);
            return;
        }
        this.callHandlerForMessageType_(type, message, messageId);
    }
    postToTarget_(messageType, message, messageId) {
        const messageWrapper = {
            messageId,
            type: messageType,
            message: message || {},
        };
        // The next line should probably be passing a transfer argument, but that
        // causes Chrome to send a "null" message. The transfer seems to work
        // without the third argument (but inefficiently, perhaps).
        this.target_.postMessage(messageWrapper, this.targetOrigin_);
    }
}

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview
 * Message definitions passed over the HelpApp privileged/unprivileged pipe.
 */
/** Enum for message types. */
var Message;
(function (Message) {
    Message["OPEN_FEEDBACK_DIALOG"] = "open-feedback-dialog";
    Message["SHOW_ON_DEVICE_APP_CONTROLS"] = "show-on-device-app-controls";
    Message["SHOW_PARENTAL_CONTROLS"] = "show-parental-controls";
    Message["TRIGGER_WELCOME_TIP_CALL_TO_ACTION"] = "trigger-welcome-tip-call-to-action";
    Message["ADD_OR_UPDATE_SEARCH_INDEX"] = "add-or-update-search-index";
    Message["CLEAR_SEARCH_INDEX"] = "clear-search-index";
    Message["FIND_IN_SEARCH_INDEX"] = "find-in-search-index";
    Message["CLOSE_BACKGROUND_PAGE"] = "close-background-page";
    Message["UPDATE_LAUNCHER_SEARCH_INDEX"] = "update-launcher-search-index";
    Message["LAUNCH_MICROSOFT_365_SETUP"] = "launch-microsoft-365-setup";
    Message["MAYBE_SHOW_RELEASE_NOTES_NOTIFICATION"] = "maybe-show-release-notes-notification";
    Message["GET_DEVICE_INFO"] = "get-device-info";
    Message["OPEN_SETTINGS"] = "open-settings";
    Message["OPEN_URL_IN_BROWSER_AND_TRIGGER_INSTALL_DIALOG"] = "open-url-in-browser-and-trigger-install-dialog";
    Message["SET_HAS_COMPLETED_NEW_DEVICE_CHECKLIST"] = "set-has-completed-new-device-checklist";
    Message["SET_HAS_VISITED_HOW_TO_PAGE"] = "set-has-visited-how-to-page";
    Message["OPEN_APP_MALL_PATH"] = "open-app-mall-path";
})(Message || (Message = {}));

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview
 * A script for the app inside the iframe. Implements a delegate.
 */
/** A pipe through which we can send messages to the parent frame. */
const parentMessagePipe = new MessagePipe('chrome://help-app', window.parent);
/**
 * A delegate which exposes privileged WebUI functionality to the help
 * app.
 */
const DELEGATE = {
    async openFeedbackDialog() {
        const response = await parentMessagePipe.sendMessage(Message.OPEN_FEEDBACK_DIALOG);
        return response['errorMessage'];
    },
    showOnDeviceAppControls() {
        return parentMessagePipe.sendMessage(Message.SHOW_ON_DEVICE_APP_CONTROLS);
    },
    showParentalControls() {
        return parentMessagePipe.sendMessage(Message.SHOW_PARENTAL_CONTROLS);
    },
    triggerWelcomeTipCallToAction(actionTypeId) {
        return parentMessagePipe.sendMessage(Message.TRIGGER_WELCOME_TIP_CALL_TO_ACTION, actionTypeId);
    },
    addOrUpdateSearchIndex(data) {
        return parentMessagePipe.sendMessage(Message.ADD_OR_UPDATE_SEARCH_INDEX, data);
    },
    clearSearchIndex() {
        return parentMessagePipe.sendMessage(Message.CLEAR_SEARCH_INDEX);
    },
    findInSearchIndex(query, maxResults = 50) {
        return parentMessagePipe.sendMessage(Message.FIND_IN_SEARCH_INDEX, { query, maxResults });
    },
    closeBackgroundPage() {
        parentMessagePipe.sendMessage(Message.CLOSE_BACKGROUND_PAGE);
    },
    updateLauncherSearchIndex(data) {
        return parentMessagePipe.sendMessage(Message.UPDATE_LAUNCHER_SEARCH_INDEX, data);
    },
    launchMicrosoft365Setup() {
        return parentMessagePipe.sendMessage(Message.LAUNCH_MICROSOFT_365_SETUP);
    },
    maybeShowReleaseNotesNotification() {
        return parentMessagePipe.sendMessage(Message.MAYBE_SHOW_RELEASE_NOTES_NOTIFICATION);
    },
    getDeviceInfo() {
        return parentMessagePipe.sendMessage(Message.GET_DEVICE_INFO);
    },
    openSettings(path) {
        parentMessagePipe.sendMessage(Message.OPEN_SETTINGS, path);
    },
    openUrlInBrowserAndTriggerInstallDialog(url) {
        return parentMessagePipe.sendMessage(Message.OPEN_URL_IN_BROWSER_AND_TRIGGER_INSTALL_DIALOG, url);
    },
    setHasCompletedNewDeviceChecklist() {
        parentMessagePipe.sendMessage(Message.SET_HAS_COMPLETED_NEW_DEVICE_CHECKLIST);
    },
    setHasVisitedHowToPage() {
        parentMessagePipe.sendMessage(Message.SET_HAS_VISITED_HOW_TO_PAGE);
    },
    openAppMallPath(path) {
        parentMessagePipe.sendMessage(Message.OPEN_APP_MALL_PATH, { path });
    },
};
window.customLaunchData = {
    delegate: DELEGATE,
};
window.addEventListener('DOMContentLoaded', function () {
    // Start listening to color change events. These events get picked up by
    // logic in ts_helpers.ts on the google3 side.
    ColorChangeUpdater.forDocument().start();
});
// Expose functions to bind to color change events to window so they can be
// automatically picked up by installColors(). See ts_helpers.ts in google3.
window.addColorChangeListener = function (listener) {
    ColorChangeUpdater.forDocument().eventTarget.addEventListener(COLOR_PROVIDER_CHANGED, listener);
};
window.removeColorChangeListener = function (listener) {
    ColorChangeUpdater.forDocument().eventTarget.removeEventListener(COLOR_PROVIDER_CHANGED, listener);
};
const TEST_ONLY = { parentMessagePipe };

export { TEST_ONLY };
//# sourceMappingURL=receiver.rollup.js.map
