// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview WallpaperImages displays a list of wallpaper images from a
 * wallpaper collection. It requires a parameter collection-id to fetch
 * and display the images. It also caches the list of wallpaper images by
 * wallpaper collection id to avoid refetching data unnecessarily.
 */
import 'chrome://resources/ash/common/personalization/common.css.js';
import 'chrome://resources/ash/common/personalization/wallpaper.css.js';
import 'chrome://resources/polymer/v3_0/iron-icon/iron-icon.js';
import 'chrome://resources/polymer/v3_0/iron-list/iron-list.js';
import '../../common/icons.html.js';
import { isImageDataUrl, isNonEmptyFilePath, isUrl } from 'chrome://resources/ash/common/sea_pen/sea_pen_utils.js';
import { assert } from 'chrome://resources/js/assert.js';
import { afterNextRender } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { WallpaperType } from '../../personalization_app.mojom-webui.js';
import { WithPersonalizationStore } from '../personalization_store.js';
import { kDefaultImageSymbol } from './constants.js';
import { getTemplate } from './local_images_element.html.js';
import { getPathOrSymbol, isDefaultImage } from './utils.js';
import { selectWallpaper } from './wallpaper_controller.js';
import { getWallpaperProvider } from './wallpaper_interface_provider.js';
export class LocalImagesElement extends WithPersonalizationStore {
    static get is() {
        return 'local-images';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            images_: {
                type: Array,
                observer: 'onImagesChanged_',
            },
            /** Mapping of local image path to data url. */
            imageData_: Object,
            /** Mapping of local image path to boolean. */
            imageDataLoading_: Object,
            currentSelected_: Object,
            /** The pending selected image. */
            pendingSelected_: Object,
            imagesToDisplay_: {
                type: Array,
                value: [],
            },
        };
    }
    static get observers() {
        return ['onImageLoaded_(imageData_, imageDataLoading_)'];
    }
    constructor() {
        super();
        this.wallpaperProvider_ = getWallpaperProvider();
    }
    ready() {
        super.ready();
        afterNextRender(this, () => {
            this.shadowRoot.getElementById('main').focus();
        });
    }
    connectedCallback() {
        super.connectedCallback();
        this.watch('images_', state => state.wallpaper.local.images);
        this.watch('imageData_', state => state.wallpaper.local.data);
        this.watch('imageDataLoading_', state => state.wallpaper.loading.local.data);
        this.watch('currentSelected_', state => state.wallpaper.currentSelected);
        this.watch('pendingSelected_', state => state.wallpaper.pendingSelected);
        this.updateFromStore();
    }
    /** Sets |imagesToDisplay| when a new set of local images loads. */
    onImagesChanged_(images) {
        this.imagesToDisplay_ = (images || []).filter(image => {
            const key = getPathOrSymbol(image);
            if (this.imageDataLoading_[key] === false) {
                const data = this.imageData_[key];
                return isUrl(data) && isImageDataUrl(data);
            }
            return true;
        });
    }
    /**
     * Called each time a new image thumbnail is loaded. Removes images
     * from the list of displayed images if it has failed to load.
     */
    onImageLoaded_(imageData, imageDataLoading) {
        if (!imageData || !imageDataLoading) {
            return;
        }
        // Iterate backwards in case we need to splice to remove from
        // |imagesToDisplay| while iterating.
        for (let i = this.imagesToDisplay_.length - 1; i >= 0; i--) {
            const image = this.imagesToDisplay_[i];
            const key = getPathOrSymbol(image);
            const doneLoading = imageDataLoading[key] === false;
            const failed = doneLoading &&
                (!isUrl(imageData[key]) || !isImageDataUrl(imageData[key]));
            if (failed) {
                this.splice('imagesToDisplay_', i, 1);
            }
        }
    }
    isImageSelected_(image, currentSelected, pendingSelected) {
        if (!image || (!currentSelected && !pendingSelected)) {
            return false;
        }
        if (isDefaultImage(image)) {
            return ((isDefaultImage(pendingSelected)) ||
                (!pendingSelected && !!currentSelected &&
                    currentSelected.type === WallpaperType.kDefault));
        }
        return (isNonEmptyFilePath(pendingSelected) &&
            image.path === pendingSelected.path ||
            !!currentSelected && image.path === currentSelected.key &&
                !pendingSelected);
    }
    getAriaLabel_(image, imageDataLoading) {
        if (this.isImageLoading_(image, imageDataLoading)) {
            return this.i18n('ariaLabelLoading');
        }
        if (isDefaultImage(image)) {
            return this.i18n('defaultWallpaper');
        }
        if (!isNonEmptyFilePath(image)) {
            return '';
        }
        const path = image.path;
        return path.substring(path.lastIndexOf('/') + 1);
    }
    isImageLoading_(image, imageDataLoading) {
        if (!image || !imageDataLoading) {
            return true;
        }
        const key = getPathOrSymbol(image);
        // If key is not present, then loading has not yet started. Still show a
        // loading tile in this case.
        return !imageDataLoading.hasOwnProperty(key) ||
            imageDataLoading[key] === true;
    }
    getImageData_(image, imageData, imageDataLoading) {
        if (!image || this.isImageLoading_(image, imageDataLoading)) {
            return null;
        }
        const data = imageData[getPathOrSymbol(image)];
        // Return a "fail" url that will not load.
        if (!isUrl(data) || !isImageDataUrl(data)) {
            return { url: '' };
        }
        return data;
    }
    getImageDataId_(image) {
        if (!image) {
            return '';
        }
        return isNonEmptyFilePath(image) ? image.path : image.toString();
    }
    onImageSelected_(event) {
        assert(event.model.item === kDefaultImageSymbol ||
            isNonEmptyFilePath(event.model.item), 'local image is a file path or default image');
        selectWallpaper(event.model.item, this.wallpaperProvider_, this.getStore());
    }
    getAriaIndex_(i) {
        return i + 1;
    }
}
customElements.define(LocalImagesElement.is, LocalImagesElement);
