// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { assert } from 'chrome://resources/js/assert.js';
import { EventTracker } from 'chrome://resources/js/event_tracker.js';
import { PrinterType } from '../../print.mojom-webui.js';
import { createCustomEvent } from '../utils/event_utils.js';
import { getDestinationProvider } from '../utils/mojo_data_providers.js';
import { DESTINATION_MANAGER_ACTIVE_DESTINATION_CHANGED, DestinationManager } from './destination_manager.js';
/**
 * @fileoverview
 * 'capabilties_manager' responsible for requesting and storing the printing
 * capabilities for the active destination.
 */
export const CAPABILITIES_MANAGER_SESSION_INITIALIZED = 'capabilities-manager.session-initialized';
export const CAPABILITIES_MANAGER_ACTIVE_DESTINATION_CAPS_LOADING = 'capabilities-manager.active-destination-caps-loading';
export const CAPABILITIES_MANAGER_ACTIVE_DESTINATION_CAPS_READY = 'capabilities-manager.active-destination-caps-ready';
export class CapabilitiesManager extends EventTarget {
    static { this.instance = null; }
    static getInstance() {
        if (CapabilitiesManager.instance === null) {
            CapabilitiesManager.instance = new CapabilitiesManager();
        }
        return CapabilitiesManager.instance;
    }
    static resetInstanceForTesting() {
        CapabilitiesManager.instance = null;
    }
    // Prevent additional initialization.
    constructor() {
        super();
        this.eventTracker = new EventTracker();
        this.destinationManager = DestinationManager.getInstance();
        this.capabilitiesCache = new Map();
        this.activeDestinationCapabilitiesLoaded = false;
        // Setup mojo data providers.
        this.destinationProvider = getDestinationProvider();
        this.eventTracker.add(this.destinationManager, DESTINATION_MANAGER_ACTIVE_DESTINATION_CHANGED, () => this.fetchCapabilitesForActiveDestination());
    }
    // `initializeSession` is only intended to be called once from the
    // `PrintPreviewCrosAppController`.
    initializeSession(sessionContext) {
        assert(!this.sessionContext, 'SessionContext should only be configured once');
        this.sessionContext = sessionContext;
        this.dispatchEvent(createCustomEvent(CAPABILITIES_MANAGER_SESSION_INITIALIZED));
    }
    fetchCapabilitesForActiveDestination() {
        const destination = this.destinationManager.getActiveDestination();
        if (destination === null) {
            return;
        }
        this.setActiveCapabilitiesLoading();
        const cachedCapabilities = this.capabilitiesCache.get(destination.id);
        if (cachedCapabilities) {
            this.setActiveCapabilitiesReady();
            return;
        }
        // TODO(b/323421684): Use printer type from destination once the
        // `Destination` object is mojo typed.
        this.destinationProvider
            .fetchCapabilities(destination.id, PrinterType.kPdf)
            .then(
        // TODO(b/323421684): Create a CapabilitiesResponse.
        (response) => this.onCapabilitiesFetched(response.capabilities));
    }
    onCapabilitiesFetched(caps) {
        // TODO(b/323421684): Handle failed capabilities call.
        if (!caps) {
            return;
        }
        this.capabilitiesCache.set(caps.destinationId, caps);
        // Since multiple capabilities requests can be in-flight simultaneously,
        // verify this capabilities response belongs to the active destination
        // before sending the ready event.
        const activeDestination = this.destinationManager.getActiveDestination();
        assert(activeDestination);
        if (caps.destinationId === activeDestination.id) {
            this.setActiveCapabilitiesReady();
        }
    }
    // Returns the capabilities from the active destination if available.
    getActiveDestinationCapabilities() {
        const activeDestination = this.destinationManager.getActiveDestination();
        if (activeDestination === null) {
            return undefined;
        }
        return this.capabilitiesCache.get(activeDestination.id);
    }
    setActiveCapabilitiesLoading() {
        this.activeDestinationCapabilitiesLoaded = false;
        this.dispatchEvent(createCustomEvent(CAPABILITIES_MANAGER_ACTIVE_DESTINATION_CAPS_LOADING));
    }
    setActiveCapabilitiesReady() {
        this.activeDestinationCapabilitiesLoaded = true;
        this.dispatchEvent(createCustomEvent(CAPABILITIES_MANAGER_ACTIVE_DESTINATION_CAPS_READY));
    }
    areActiveDestinationCapabilitiesLoaded() {
        return this.activeDestinationCapabilitiesLoaded;
    }
    // Returns true only after the `initializeSession` function has been called
    // with a valid `SessionContext`.
    isSessionInitialized() {
        return !!this.sessionContext;
    }
}
