// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { FakeMethodResolver } from 'chrome://resources/ash/common/fake_method_resolver.js';
import { FakeObservables } from 'chrome://resources/ash/common/fake_observables.js';
import { assert } from 'chrome://resources/js/assert.js';
import { ColorType, DuplexType, PageOrientation } from '../../printer_capabilities.mojom-webui.js';
/**
 * @fileoverview
 * 'fake_destination_provider' is a mock implementation of the
 * `DestinationProvider` mojo interface.
 */
export const GET_LOCAL_DESTINATIONS_METHOD = 'getLocalDestinations';
export const FAKE_GET_LOCAL_DESTINATIONS_SUCCESSFUL_EMPTY = [];
export const OBSERVE_DESTINATION_CHANGES_METHOD = 'observeDestinationChanges';
export const FETCH_CAPABILITIES_METHOD = 'fetchCapabilities';
const OBSERVABLE_ON_DESTINATIONS_CHANGED_METHOD = 'onDestinationsChanged';
// Fake implementation of the DestinationProvider for tests and UI.
export class FakeDestinationProvider {
    constructor() {
        this.methods = new FakeMethodResolver();
        this.callCount = new Map();
        this.testDelayMs = 0;
        this.observables = new FakeObservables();
        this.registerMethods();
        this.registerObservables();
    }
    registerMethods() {
        this.methods.register(GET_LOCAL_DESTINATIONS_METHOD);
        this.methods.setResult(GET_LOCAL_DESTINATIONS_METHOD, { destinations: FAKE_GET_LOCAL_DESTINATIONS_SUCCESSFUL_EMPTY });
        this.callCount.set(GET_LOCAL_DESTINATIONS_METHOD, 0);
        this.methods.register(FETCH_CAPABILITIES_METHOD);
        this.callCount.set(FETCH_CAPABILITIES_METHOD, 0);
        this.methods.setResult(FETCH_CAPABILITIES_METHOD, getFakeCapabilitiesResponse());
    }
    registerObservables() {
        this.observables.register(OBSERVABLE_ON_DESTINATIONS_CHANGED_METHOD);
        const defaultOnDestinationChanged = [];
        this.setDestinationsChangesData(defaultOnDestinationChanged);
    }
    incrementCallCount(methodName) {
        const prevCallCount = this.callCount.get(methodName) ?? 0;
        this.callCount.set(methodName, prevCallCount + 1);
    }
    // Handles restoring state of fake to initial state.
    reset() {
        this.callCount.clear();
        this.methods = new FakeMethodResolver();
        this.registerMethods();
        this.testDelayMs = 0;
        this.observables = new FakeObservables();
        this.registerObservables();
    }
    getCallCount(method) {
        return this.callCount.get(method) ?? 0;
    }
    setTestDelay(delay) {
        assert(delay >= 0);
        this.testDelayMs = delay;
    }
    getMethodsForTesting() {
        return this.methods;
    }
    getLocalDestinations() {
        this.incrementCallCount(GET_LOCAL_DESTINATIONS_METHOD);
        return this.methods.resolveMethodWithDelay(GET_LOCAL_DESTINATIONS_METHOD, this.testDelayMs);
    }
    // Setup observable for `OBSERVABLE_ON_DESTINATIONS_CHANGED_METHOD` to enable
    // testing observer updates when `triggerOnDestinationChanged` called.
    observeDestinationChanges(observer) {
        this.observables.observe(OBSERVABLE_ON_DESTINATIONS_CHANGED_METHOD, (destinations) => {
            observer.onDestinationsChanged(destinations);
        });
        this.incrementCallCount(OBSERVE_DESTINATION_CHANGES_METHOD);
    }
    // Set destination list to be returned when observer is called.
    setDestinationsChangesData(destinations) {
        // Parameters from observer functions are returned in an array.
        this.observables.setObservableData(OBSERVABLE_ON_DESTINATIONS_CHANGED_METHOD, [[destinations]]);
    }
    // Trigger destination observer function onDestinationsChanged.
    // `observeDestinationChanges` must be called at least once or this function
    // will throw an error.
    triggerOnDestinationsChanged() {
        this.observables.trigger(OBSERVABLE_ON_DESTINATIONS_CHANGED_METHOD);
    }
    setLocalDestinationResult(destinations) {
        this.methods.setResult(GET_LOCAL_DESTINATIONS_METHOD, { destinations: destinations });
    }
    fetchCapabilities(_destinationId, _printerType) {
        this.incrementCallCount(FETCH_CAPABILITIES_METHOD);
        return this.methods.resolveMethodWithDelay(FETCH_CAPABILITIES_METHOD, this.testDelayMs);
    }
    setCapabilities(capabilities) {
        this.methods.setResult(FETCH_CAPABILITIES_METHOD, { capabilities });
    }
}
// TODO(b/323421684): Move this function to "fake_data.ts" once all the
//    DestinationProvider methods are migrated to mojo.
export function getFakeCapabilitiesResponse(destinationId = 'Printer1') {
    const collate = {
        valueDefault: true,
    };
    const color = {
        options: [
            {
                type: ColorType.kStandardColor,
                vendorId: '1',
                isDefault: true,
            },
            {
                type: ColorType.kStandardMonochrome,
                vendorId: '2',
            },
        ],
        resetToDefault: false,
    };
    const copies = {
        valueDefault: 1,
        max: 9999,
    };
    const duplex = {
        options: [
            {
                type: DuplexType.kNoDuplex,
                isDefault: true,
            },
            {
                type: DuplexType.kLongEdge,
            },
            {
                type: DuplexType.kShortEdge,
            },
        ],
    };
    const pageOrientation = {
        options: [
            {
                option: PageOrientation.kPortrait,
                isDefault: true,
            },
            {
                option: PageOrientation.kLandscape,
            },
            {
                option: PageOrientation.kAuto,
            },
        ],
        resetToDefault: false,
    };
    const mediaSize = {
        options: [
            {
                vendorId: 'na_govt-letter_8x10in',
                heightMicrons: 254000,
                widthMicrons: 203200,
                imageableAreaLeftMicrons: 3000,
                imageableAreaBottomMicrons: 3000,
                imageableAreaRightMicrons: 200200,
                imageableAreaTopMicrons: 251000,
                hasBorderlessVariant: true,
                customDisplayName: '8 x 10 in',
                name: 'NA_GOVT_LETTER',
            },
            {
                vendorId: 'na_legal_8.5x14in',
                heightMicrons: 297000,
                widthMicrons: 210000,
                imageableAreaLeftMicrons: 3000,
                imageableAreaBottomMicrons: 3000,
                imageableAreaRightMicrons: 207000,
                imageableAreaTopMicrons: 294000,
                hasBorderlessVariant: true,
                customDisplayName: 'A4',
                name: 'ISO_A4',
            },
            {
                vendorId: 'na_legal_8.5x14in',
                heightMicrons: 355600,
                widthMicrons: 215900,
                imageableAreaLeftMicrons: 3000,
                imageableAreaBottomMicrons: 3000,
                imageableAreaRightMicrons: 212900,
                imageableAreaTopMicrons: 352600,
                customDisplayName: 'Legal',
                name: 'NA_LEGAL',
            },
            {
                vendorId: 'na_letter_8.5x11in',
                heightMicrons: 279400,
                widthMicrons: 215900,
                imageableAreaLeftMicrons: 3000,
                imageableAreaBottomMicrons: 3000,
                imageableAreaRightMicrons: 212900,
                imageableAreaTopMicrons: 276400,
                hasBorderlessVariant: true,
                customDisplayName: 'Letter',
                name: 'NA_LETTER',
            },
        ],
        resetToDefault: false,
    };
    const mediaType = {
        options: [
            {
                vendorId: 'stationery',
                customDisplayName: 'Paper (Plain)',
                isDefault: true,
            },
            {
                vendorId: 'photographic',
                customDisplayName: 'Photo',
            },
            {
                vendorId: 'envelope',
                customDisplayName: 'Envelope',
            },
        ],
        resetToDefault: false,
    };
    const dpi = {
        options: [
            {
                horizontalDpi: 300,
                verticalDpi: 300,
                isDefault: true,
            },
            {
                horizontalDpi: 600,
                verticalDpi: 600,
            },
            {
                horizontalDpi: 800,
                verticalDpi: 1000,
            },
        ],
        resetToDefault: false,
    };
    const pin = {
        supported: false,
    };
    const capabilities = {
        destinationId: destinationId,
        collate: collate,
        color: color,
        copies: copies,
        duplex: duplex,
        pageOrientation: pageOrientation,
        mediaSize: mediaSize,
        mediaType: mediaType,
        dpi: dpi,
        pin: pin,
    };
    return { capabilities: capabilities };
}
