// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import{classMap,css,html,nothing,repeat,svg}from"chrome://resources/mwc/lit/index.js";import{POWER_BARS_PER_SECOND,POWER_SCALE_FACTOR}from"../core/audio_constants.js";import{i18n}from"../core/i18n.js";import{ReactiveLitElement}from"../core/reactive/lit.js";import{computed}from"../core/reactive/signal.js";import{assert,assertExists,assertInstanceof}from"../core/utils/assert.js";import{InteriorMutableArray}from"../core/utils/interior_mutable_array.js";import{getNumSpeakerClass,getSpeakerLabelClass,SPEAKER_LABEL_COLORS}from"./styles/speaker_label.js";const BAR_WIDTH=4;const BAR_GAP=5;const BAR_MIN_HEIGHT=4.5;const BAR_MAX_HEIGHT=100;const SPEAKER_LABEL_LINE_HEIGHT=128;function toViewBoxString(viewBox){if(viewBox===null){return nothing}const{x:x,y:y,width:width,height:height}=viewBox;return`${x} ${y} ${width} ${height}`}function timestampToBarIndex(seconds,barsPerSecond){return Math.floor(seconds*barsPerSecond)}function getBarX(barIdx){return barIdx*(BAR_WIDTH+BAR_GAP)}function xCoordinateToRoughIdx(x){return Math.floor(x/(BAR_WIDTH+BAR_GAP))}export class AudioWaveform extends ReactiveLitElement{constructor(){super(...arguments);this.values=new InteriorMutableArray([]);this.currentTime=null;this.barsPerSecond=POWER_BARS_PER_SECOND;this.currentTimeSignal=this.propSignal("currentTime");this.currentTimeBarIdx=computed((()=>{if(this.currentTimeSignal.value===null){return null}return timestampToBarIndex(this.currentTimeSignal.value,this.barsPerSecond)}));this.size=null;this.transcription=null;this.transcriptionSignal=this.propSignal("transcription");this.speakerLabelInfo=computed((()=>{const transcription=this.transcriptionSignal.value;if(transcription===null){return{speakerLabels:[],ranges:[]}}const paragraphs=transcription.getParagraphs();const speakerLabels=transcription.getSpeakerLabels();const ranges=[];for(const paragraph of paragraphs){const firstPart=assertExists(paragraph[0]);const lastPart=assertExists(paragraph.at(-1));const speakerLabel=firstPart.speakerLabel;if(speakerLabel===null){continue}const speakerLabelIndex=speakerLabels.indexOf(speakerLabel);assert(speakerLabelIndex!==-1);const startMs=firstPart.timeRange?.startMs??null;const endMs=lastPart.timeRange?.endMs??null;if(startMs===null||endMs===null){continue}assert(startMs<=endMs);const startBarIdx=timestampToBarIndex(startMs/1e3,this.barsPerSecond);const endBarIdx=timestampToBarIndex(endMs/1e3,this.barsPerSecond);assert(ranges.length===0||assertExists(ranges.at(-1)).endBarIdx<=startBarIdx);if(startBarIdx!==endBarIdx){ranges.push({speakerLabelIndex:speakerLabelIndex,startBarIdx:startBarIdx,endBarIdx:endBarIdx})}}return{speakerLabels:speakerLabels,ranges:ranges}}));this.resizeObserver=new ResizeObserver((()=>{this.size=this.getBoundingClientRect()}))}static{this.styles=[SPEAKER_LABEL_COLORS,css`
      :host {
        display: block;
        position: relative;
      }

      #chart {
        inset: 0;
        position: absolute;
      }

      .speaker-single {
        & .range {
          display: none;
        }
      }

      .speaker-duo,
      .speaker-multiple {
        & .no-speaker {
          --speaker-label-shapes-color: var(--cros-sys-primary_container);
        }
      }

      .speaker-range-start {
        /* The dash and space looks equal length with rounded linecap. */
        stroke-dasharray: 2, 6;
        stroke-linecap: round;
        stroke-width: 2;
        stroke: var(--speaker-label-shapes-color);

        .speaker-single & {
          display: none;
        }

        &.future {
          opacity: var(--cros-disabled-opacity);
        }

        .range:hover & {
          stroke-dasharray: none;
        }
      }

      .bar {
        /* Don't block hover on the background. */
        pointer-events: none;

        .speaker-single & {
          fill: var(--cros-sys-primary);

          &.future {
            fill: var(--cros-sys-primary_container);
          }
        }

        :is(.speaker-duo, .speaker-multiple) & {
          fill: var(--speaker-label-shapes-color);

          &.future {
            opacity: var(--cros-disabled-opacity);
          }
        }
      }

      .background {
        /* fill: none prevents :hover state, so we set opacity: 0 instead. */
        opacity: 0;
        fill: var(--speaker-label-container-color);

        .range:hover & {
          opacity: 1;

          &.future {
            opacity: var(--cros-disabled-opacity);
          }
        }
      }

      .speaker-label {
        align-items: center;
        background: var(--speaker-label-shapes-color);
        border-radius: 10px 10px 10px 0;
        bottom: 0;
        box-sizing: border-box;
        color: var(--speaker-label-label-color);
        display: flex;
        font: var(--cros-label-1-font);
        height: 20px;
        justify-content: center;
        left: 0;
        min-width: 20px;
        padding: 4px;
        position: absolute;
        width: fit-content;

        &.outside {
          display: none;
        }

        & > .full {
          display: none;
        }

        .range:hover & {
          display: block;

          /* TODO: b/336963138 - Animation on hover? */
          height: 26px;
          padding: 8px;

          & > .full {
            display: inline;
          }

          & > .short {
            display: none;
          }
        }
      }

      .playhead {
        fill: var(--cros-sys-on_surface_variant);

        /* Don't block hover on the background. */
        pointer-events: none;
      }
    `]}static{this.properties={values:{attribute:false},size:{state:true},currentTime:{type:Number},barsPerSecond:{attribute:false},transcription:{attribute:false}}}get chart(){return assertInstanceof(assertExists(this.shadowRoot).querySelector("#chart"),SVGElement)}connectedCallback(){super.connectedCallback();this.resizeObserver.observe(this)}disconnectedCallback(){super.disconnectedCallback();this.resizeObserver.disconnect()}getBarLocation(idx,val,minHeight,maxHeight){const width=BAR_WIDTH;const height=minHeight+(maxHeight-minHeight)*(val/(POWER_SCALE_FACTOR-1));const x=getBarX(idx)-width/2;const y=-height/2;return{x:x,y:y,width:width,height:height}}isAfterCurrentTime(idx){return this.currentTimeBarIdx.value!==null&&idx>=this.currentTimeBarIdx.value}renderSpeakerRangeStart({startBarIdx:startBarIdx,speakerLabelIndex:speakerLabelIndex}){const startX=getBarX(startBarIdx-.5);const classes={[getSpeakerLabelClass(speakerLabelIndex)]:true,future:this.isAfterCurrentTime(startBarIdx)};const height=SPEAKER_LABEL_LINE_HEIGHT;return svg`<line
      x1=${startX}
      x2=${startX}
      y1=${-height/2}
      y2=${height/2}
      class="speaker-range-start ${classMap(classes)} "
    />`}renderSpeakerRangeLabel(speakerLabels,{startBarIdx:startBarIdx,speakerLabelIndex:speakerLabelIndex},viewBox){const startX=getBarX(startBarIdx-.5)-1;const classes={[getSpeakerLabelClass(speakerLabelIndex)]:true,outside:startX<viewBox.x};const maxHeight=26;const x=Math.max(startX,viewBox.x);const y=-SPEAKER_LABEL_LINE_HEIGHT/2-maxHeight;const shortLabel=assertExists(speakerLabels[speakerLabelIndex]);const fullLabel=i18n.transcriptionSpeakerLabelLabel(shortLabel);return svg`<foreignObject
      x=${x}
      y=${y}
      width="100"
      height=${maxHeight}
    >
      <div class="speaker-label ${classMap(classes)}" aria-label=${fullLabel}>
        <span class="short" aria-hidden="true">${shortLabel}</span>
        <span class="full" aria-hidden="true">${fullLabel}</span>
      </div>
    </foreignObject>`}getBackgroundPath(startX,endX){const height=SPEAKER_LABEL_LINE_HEIGHT;const radius=12;return`\n      M ${startX} ${-height/2}\n      v ${height}\n      H ${endX-radius}\n      a ${radius} ${radius} 0 0 0 ${radius} ${-radius}\n      V ${-height/2+radius}\n      a ${radius} ${radius} 0 0 0 ${-radius} ${-radius}\n      H ${startX}\n    `}renderSpeakerRangeBackground({startBarIdx:startBarIdx,endBarIdx:endBarIdx,speakerLabelIndex:speakerLabelIndex}){const startX=getBarX(startBarIdx-.5);const endX=getBarX(endBarIdx-.5);const classes={[getSpeakerLabelClass(speakerLabelIndex)]:true};const currentTimeIdx=this.currentTimeBarIdx.value;if(currentTimeIdx!==null&&startBarIdx<=currentTimeIdx&&currentTimeIdx<endBarIdx){const centerX=getBarX(currentTimeIdx)-BAR_WIDTH/2;const height=SPEAKER_LABEL_LINE_HEIGHT;const y=-height/2;return[svg`<rect
          x=${startX}
          y=${y}
          width=${centerX-startX}
          height=${height}
          class="background ${classMap(classes)}"
        />`,svg`<path
          d=${this.getBackgroundPath(centerX,endX)}
          class="background future ${classMap(classes)}"
        />`]}else{classes["future"]=this.isAfterCurrentTime(startBarIdx);return svg`<path
        d=${this.getBackgroundPath(startX,endX)}
        class="background ${classMap(classes)}"
      />`}}renderSpeakerRange(speakerLabels,range,viewBox){return svg`<g class="range">
      ${this.renderSpeakerRangeBackground(range)}
      ${this.renderSpeakerRangeStart(range)}
      ${this.renderSpeakerRangeLabel(speakerLabels,range,viewBox)}
    </g>`}renderCurrentTimeBar(viewBox){if(this.currentTimeBarIdx.value===null){return nothing}const width=2;const x=getBarX(this.currentTimeBarIdx.value)-BAR_WIDTH/2-width;const y=viewBox.y;return svg`<rect
      x=${x}
      y=${y}
      width=${width}
      height=${viewBox.height}
      rx="1"
      class="playhead"
    />`}renderAudioBars(viewBox){if(this.values.length===0){return nothing}const speakerLabelRanges=this.speakerLabelInfo.value.ranges;let currentSpeakerLabelRangeIdx=0;let currentSpeakerLabelRangeRendered=false;function getSpeakerLabelRange(barIdx){while(currentSpeakerLabelRangeIdx<speakerLabelRanges.length){const range=assertExists(speakerLabelRanges[currentSpeakerLabelRangeIdx]);if(barIdx<range.startBarIdx){return null}if(barIdx<range.endBarIdx){return range}currentSpeakerLabelRangeIdx+=1;currentSpeakerLabelRangeRendered=false}return null}const startIdx=Math.max(xCoordinateToRoughIdx(viewBox.x)-5,0);const endIdx=Math.min(xCoordinateToRoughIdx(viewBox.x+viewBox.width)+5,this.values.length-1);if(endIdx<startIdx){return nothing}const idxRange=Array.from({length:endIdx-startIdx+1},((_,i)=>i+startIdx));const toRenderBars=[];const toRenderSpeakerLabelRanges=[];for(const i of idxRange){const val=assertExists(this.values.array[i]);const rect=this.getBarLocation(i,val,BAR_MIN_HEIGHT,Math.min(viewBox.height,BAR_MAX_HEIGHT));if(rect.x+rect.width<viewBox.x||rect.x>viewBox.x+viewBox.width){continue}const classes={future:this.isAfterCurrentTime(i)};const range=getSpeakerLabelRange(i);if(range!==null){if(!currentSpeakerLabelRangeRendered){toRenderSpeakerLabelRanges.push(range);currentSpeakerLabelRangeRendered=true}classes[getSpeakerLabelClass(range.speakerLabelIndex)]=true}else{classes["no-speaker"]=true}toRenderBars.push({idx:i,rect:rect,classes:classes})}return[repeat(toRenderSpeakerLabelRanges,(({startBarIdx:startBarIdx})=>startBarIdx),(range=>this.renderSpeakerRange(this.speakerLabelInfo.value.speakerLabels,range,viewBox))),repeat(toRenderBars,(({idx:idx})=>idx),(({rect:rect,classes:classes})=>svg`<rect
          x=${rect.x}
          y=${rect.y}
          width=${rect.width}
          height=${rect.height}
          rx=${rect.width/2}
          class="bar ${classMap(classes)}"
        />`))]}renderSvgContent(viewBox){if(viewBox===null){return nothing}return[this.renderAudioBars(viewBox),this.renderCurrentTimeBar(viewBox)]}getViewBox(){if(this.size===null){return null}const{width:width,height:height}=this.size;const x=(()=>{if(this.currentTimeBarIdx.value!==null){const x=getBarX(this.currentTimeBarIdx.value);return x-width/2}else{return this.values.length*(BAR_WIDTH+BAR_GAP)-width}})();const y=-height/2;return{x:x,y:y,width:width,height:height}}render(){if(this.size===null){return nothing}const numSpeakerClass=getNumSpeakerClass(this.speakerLabelInfo.value.speakerLabels.length);const viewBox=this.getViewBox();return html`<svg
      id="chart"
      viewBox=${toViewBoxString(viewBox)}
      class=${numSpeakerClass}
    >
      ${this.renderSvgContent(viewBox)}
    </svg>`}}window.customElements.define("audio-waveform",AudioWaveform);