// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import"chrome://resources/mwc/@material/web/focus/md-focus-ring.js";import"chrome://resources/cros_components/button/button.js";import{classMap,createRef,css,html,ifDefined,nothing,ref,repeat}from"chrome://resources/mwc/lit/index.js";import{i18n}from"../core/i18n.js";import{ReactiveLitElement}from"../core/reactive/lit.js";import{signal}from"../core/reactive/signal.js";import{assert,assertExists,assertInstanceof}from"../core/utils/assert.js";import{formatDuration}from"../core/utils/datetime.js";import{clamp,parseNumber,sliceWhen}from"../core/utils/utils.js";import{getNumSpeakerClass,getSpeakerLabelClass,SPEAKER_LABEL_COLORS}from"./styles/speaker_label.js";const SCROLL_MARGIN=3;function inBetween(x,[low,high]){return x>=Math.min(low,high)-SCROLL_MARGIN&&x<=Math.max(low,high)+SCROLL_MARGIN}export class TranscriptionView extends ReactiveLitElement{constructor(){super(...arguments);this.transcription=null;this.currentTime=null;this.seekable=false;this.autoscrollEnabled=signal(true);this.lastAutoScrollRange=null;this.lastAutoScrollTime=null;this.containerRef=createRef();this.needAutoscrollAnchor=true}static{this.styles=[SPEAKER_LABEL_COLORS,css`
      :host {
        display: block;
        position: relative;
      }

      #container {
        box-sizing: border-box;
        display: flex;
        flex-flow: column;
        gap: 12px;
        max-height: 100%;
        overflow-y: auto;
        padding: 12px 0 64px;
        width: 100%;
      }

      #transcript {
        display: grid;
        grid-template-columns:
          minmax(calc(12px + 40px + 10px), max-content)
          1fr;
      }

      .row {
        display: grid;
        grid-column: 1 / 3;
        grid-template-columns: subgrid;
        padding: 0 12px 0 0;
      }

      .timestamp {
        /*
         * Note that this need to be 0px instead of 0, since it's used in
         * calc().
         */
        --md-focus-ring-outward-offset: 0px;
        --md-focus-ring-shape: 4px;

        font: var(--cros-body-1-font);

        /*
         * Note that compared to the spec, 2px of left/right margin is moved to
         * padding so it's included in the hover / focus ring.
         */
        margin: 12px 8px 12px 10px;
        outline: none;
        padding: 0 2px;
        place-self: start;
        position: relative;

        .seekable & {
          cursor: pointer;
        }
      }

      .paragraph {
        font: var(--cros-body-1-font);
        padding: 12px;
      }

      .highlight-word {
        text-decoration: underline 1.5px;
        text-underline-offset: 3px;
      }

      .speaker-label {
        color: var(--speaker-label-shapes-color);
        font: var(--cros-button-1-font);
        margin: 0 0 4px;

        .speaker-single & {
          display: none;
        }
      }

      .speaker-pending {
        --speaker-label-shapes-color: var(--cros-sys-on_surface_variant);
      }

      .sentence {
        border-radius: 4px;
        box-decoration-break: clone;
        -webkit-box-decoration-break: clone;

        /* "Undo" the horizontal padding so the text aligns with the design. */
        margin: 0 -2px;

        /*
         * Note that while the font size is 13px, the background height without
         * padding would be 16px. Make it full line height (20px) by adding a
         * 2px vertical padding. (horizontal padding happens to also be 2px).
         */
        padding: 2px;

        .seekable & {
          cursor: pointer;

          &:hover,
          &:focus-visible {
            background: var(--cros-sys-highlight_shape);
            outline: none;
          }
        }

        .seekable .timestamp:hover + .paragraph > &:first-of-type {
          background: var(--cros-sys-highlight_shape);
        }
      }

      #autoscroll-button {
        bottom: 16px;
        left: 0;
        margin: 0 auto;
        position: absolute;
        right: 0;

        /* TODO(pihsun): Transition between shown/hide state */
        #container.autoscroll + & {
          display: none;
        }
      }
    `]}static{this.properties={transcription:{attribute:false},currentTime:{type:Number},seekable:{type:Boolean}}}onContainerScroll(){if(!this.autoscrollEnabled.value){return}const container=assertExists(this.containerRef.value);if(container.scrollTop>=container.scrollHeight-container.offsetHeight-SCROLL_MARGIN){return}if(this.lastAutoScrollRange===null||!inBetween(container.scrollTop,this.lastAutoScrollRange)){this.autoscrollEnabled.value=false;this.lastAutoScrollRange=null}}onContainerScrollEnd(){this.lastAutoScrollRange=null}updated(changedProperties){if(!this.autoscrollEnabled.value){return}if(this.seekable&&(!changedProperties.has("currentTime")||this.currentTime===null)){return}this.runAutoScroll()}runAutoScroll(){const now=Date.now();if(this.lastAutoScrollTime!==null&&this.lastAutoScrollTime>=now-500){return}const container=assertExists(this.containerRef.value);let targetScrollTop;if(this.seekable){const autoscrollAnchor=this.shadowRoot?.querySelector(".autoscroll-anchor")??null;if(autoscrollAnchor===null){targetScrollTop=container.scrollHeight-container.offsetHeight}else{assert(autoscrollAnchor instanceof HTMLElement);targetScrollTop=clamp(autoscrollAnchor.offsetTop+autoscrollAnchor.offsetHeight/2-container.clientHeight/2,0,container.scrollHeight-container.offsetHeight)}}else{targetScrollTop=container.scrollHeight-container.offsetHeight}if(Math.abs(container.scrollTop-targetScrollTop)>=SCROLL_MARGIN){this.lastAutoScrollRange=[container.scrollTop,targetScrollTop];this.lastAutoScrollTime=now;container.scrollTo({top:targetScrollTop,behavior:"smooth"})}}renderSentence(sentence){return repeat(sentence,((_v,i)=>i),((part,i)=>{const leadingSpace=i===0?false:part.leadingSpace??true;const highlightWord=(()=>{if(this.currentTime===null||part.timeRange===null){return false}return this.currentTime>=part.timeRange.startMs/1e3&&this.currentTime<part.timeRange.endMs/1e3})();if(highlightWord){this.needAutoscrollAnchor=false;return html`${leadingSpace?" ":""}
            <span class="highlight-word autoscroll-anchor">${part.text}</span>`}const autoscrollAnchor=(()=>{if(!this.needAutoscrollAnchor||this.currentTime===null||part.timeRange===null){return false}return this.currentTime<part.timeRange.startMs/1e3})();if(autoscrollAnchor){this.needAutoscrollAnchor=false;return html`${leadingSpace?" ":""}
            <span class="autoscroll-anchor">${part.text}</span>`}return`${leadingSpace?" ":""}${part.text}`}))}renderSpeakerLabel(speakerLabels,speakerLabel,partial){if(speakerLabel===null){return nothing}let speakerLabelClass;let speakerLabelLabel;if(partial){speakerLabelClass="speaker-pending";speakerLabelLabel=i18n.transcriptionSpeakerLabelPendingLabel}else{const speakerLabelIdx=speakerLabels.indexOf(speakerLabel);assert(speakerLabelIdx!==-1);speakerLabelClass=getSpeakerLabelClass(speakerLabelIdx);speakerLabelLabel=i18n.transcriptionSpeakerLabelLabel(speakerLabel)}return html`<div class="speaker-label ${speakerLabelClass}">
      ${speakerLabelLabel}
    </div>`}renderParagraphContent(parts){if(!this.seekable){return parts.map(((part,i)=>{const leadingSpace=part.leadingSpace??i>0;return`${leadingSpace?" ":""}${part.text}`})).join("")}const sentences=sliceWhen(parts,(({text:text})=>text.endsWith(".")||text.endsWith("?")||text.endsWith("!")));return repeat(sentences,((_v,i)=>i),((sentence,i)=>{const leadingSpace=sentence[0]?.leadingSpace??i>0;return html`${leadingSpace?" ":""}<span
            class="sentence"
            data-start-ms=${ifDefined(sentence[0]?.timeRange?.startMs)}
            tabindex=${this.seekable?0:-1}
            role="button"
            >${this.renderSentence(sentence)}</span
          >`}))}renderParagraph(speakerLabels,parts){const{speakerLabel:speakerLabel,partial:partial}=assertExists(parts[0]);return[this.renderSpeakerLabel(speakerLabels,speakerLabel,partial??false),this.renderParagraphContent(parts)]}onTextClick(ev){const target=assertInstanceof(ev.target,HTMLElement);const parent=target.closest("[data-start-ms]");if(parent===null){return}const startMs=parseNumber(assertInstanceof(parent,HTMLElement).dataset["startMs"]);if(startMs===null){return}this.dispatchEvent(new CustomEvent("word-clicked",{detail:{startMs:startMs}}));this.autoscrollEnabled.value=true}onAutoScrollButtonClick(){this.autoscrollEnabled.value=true;this.runAutoScroll()}render(){if(this.transcription===null){return nothing}const speakerLabels=this.transcription.getSpeakerLabels();const paragraphs=this.transcription.getParagraphs();this.needAutoscrollAnchor=true;const content=repeat(paragraphs,((_parts,i)=>i),(parts=>{const startTimeRange=assertExists(parts[0]).timeRange;const startTimeDisplay=startTimeRange===null?"?":formatDuration({milliseconds:startTimeRange.startMs});const startTimeDisplayLabel=startTimeRange===null?"?":formatDuration({milliseconds:startTimeRange.startMs},0,true);return html`
          <div class="row">
            <span
              class="timestamp"
              tabindex=${this.seekable?0:-1}
              data-start-ms=${ifDefined(startTimeRange?.startMs)}
              role="button"
              aria-label=${startTimeDisplayLabel}
            >
              ${startTimeDisplay}
              ${this.seekable?html`<md-focus-ring></md-focus-ring>`:nothing}
            </span>
            <div class="paragraph">
              ${this.renderParagraph(speakerLabels,parts)}
            </div>
          </div>
        `}));const classes={seekable:this.seekable,autoscroll:this.autoscrollEnabled.value,[getNumSpeakerClass(speakerLabels.length)]:true};return html`<div
        id="container"
        class=${classMap(classes)}
        ${ref(this.containerRef)}
        @scroll=${this.onContainerScroll}
        @scrollend=${this.onContainerScrollEnd}
      >
        <slot></slot>
        <div
          id="transcript"
          @click=${this.seekable?this.onTextClick:nothing}
        >
          ${content}
        </div>
      </div>
      <cros-button
        button-style="secondary"
        id="autoscroll-button"
        label=${i18n.transcriptionAutoscrollButton}
        @click=${this.onAutoScrollButtonClick}
      ></cros-button>`}}window.customElements.define("transcription-view",TranscriptionView);