// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import{MAX_SPEAKER_COLORS}from"../components/styles/speaker_label.js";import{SAMPLE_RATE,SAMPLES_PER_SLICE}from"./audio_constants.js";import{computed,signal}from"./reactive/signal.js";import{transcriptionSchema}from"./soda/soda.js";import{ExportAudioFormat,ExportTranscriptionFormat}from"./state/settings.js";import{assert,assertExhaustive,assertExists}from"./utils/assert.js";import{AsyncJobQueue}from"./utils/async_job_queue.js";import{z}from"./utils/schema.js";import{ulid}from"./utils/ulid.js";import{asyncLazyInit,downloadFile}from"./utils/utils.js";const baseRecordingMetadataSchema=z.object({id:z.string(),title:z.string(),durationMs:z.number(),recordedAt:z.number()});const NO_AUDIO_POWER_THRESHOLD=10;export var TimelineSegmentKind;(function(TimelineSegmentKind){TimelineSegmentKind[TimelineSegmentKind["NO_AUDIO"]=0]="NO_AUDIO";TimelineSegmentKind[TimelineSegmentKind["AUDIO"]=1]="AUDIO";TimelineSegmentKind[TimelineSegmentKind["SPEECH"]=2]="SPEECH";TimelineSegmentKind[TimelineSegmentKind["SPEECH_SPEAKER_COLOR_1"]=3]="SPEECH_SPEAKER_COLOR_1";TimelineSegmentKind[TimelineSegmentKind["SPEECH_SPEAKER_COLOR_2"]=4]="SPEECH_SPEAKER_COLOR_2";TimelineSegmentKind[TimelineSegmentKind["SPEECH_SPEAKER_COLOR_3"]=5]="SPEECH_SPEAKER_COLOR_3";TimelineSegmentKind[TimelineSegmentKind["SPEECH_SPEAKER_COLOR_4"]=6]="SPEECH_SPEAKER_COLOR_4";TimelineSegmentKind[TimelineSegmentKind["SPEECH_SPEAKER_COLOR_5"]=7]="SPEECH_SPEAKER_COLOR_5"})(TimelineSegmentKind||(TimelineSegmentKind={}));const timelineSegmentSchema=z.tuple([z.number(),z.nativeEnum(TimelineSegmentKind)]);const TIMELINE_SEGMENT_VERSION=1;const vesionedTimelineSegmentsSchema=z.object({version:z.literal(TIMELINE_SEGMENT_VERSION),segments:z.array(timelineSegmentSchema)});const derivedRecordingMetadataSchema=z.object({numSpeakers:z.optional(z.nullable(z.number())),timelineSegments:z.catch(z.optional(vesionedTimelineSegmentsSchema),undefined),description:z.string()});const recordingMetadataSchema=z.intersection([baseRecordingMetadataSchema,derivedRecordingMetadataSchema]);const audioPowerSchema=z.object({powers:z.array(z.number()),samplesPerDataPoint:z.withDefault(z.number(),SAMPLES_PER_SLICE)});function metadataName(id){return`${id}.meta.json`}function audioPowerName(id){return`${id}.powers.json`}function transcriptionName(id){return`${id}.transcription.json`}function audioName(id){return`${id}.webm`}function calculatePowerSegments(powers,samplesPerDataPoint){const segments=[];for(const power of powers){const label=power<NO_AUDIO_POWER_THRESHOLD?TimelineSegmentKind.NO_AUDIO:TimelineSegmentKind.AUDIO;if(segments.length===0||assertExists(segments.at(-1))[1]!==label){segments.push([samplesPerDataPoint,label])}else{assertExists(segments.at(-1))[0]+=samplesPerDataPoint}}return segments}function speakerLabelToTimelineSegmentKind(speakerLabels,speakerLabel){if(speakerLabel===null){return TimelineSegmentKind.SPEECH}const speakerLabelIdx=speakerLabels.indexOf(speakerLabel);assert(speakerLabelIdx!==-1);return assertExists([TimelineSegmentKind.SPEECH_SPEAKER_COLOR_1,TimelineSegmentKind.SPEECH_SPEAKER_COLOR_2,TimelineSegmentKind.SPEECH_SPEAKER_COLOR_3,TimelineSegmentKind.SPEECH_SPEAKER_COLOR_4,TimelineSegmentKind.SPEECH_SPEAKER_COLOR_5][speakerLabelIdx%MAX_SPEAKER_COLORS])}function calculateSpeechSegments(transcription){if(transcription===null){return[]}const speakerLabels=transcription.getSpeakerLabels();let currentSampleOffset=0;const segments=[];for(const paragraph of transcription.getParagraphs()){const firstPart=assertExists(paragraph[0]);const lastPart=assertExists(paragraph.at(-1));const startMs=firstPart.timeRange?.startMs??null;const endMs=lastPart.timeRange?.endMs??null;if(startMs===null||endMs===null){continue}const start=Math.round(startMs/1e3*SAMPLE_RATE);const end=Math.round(endMs/1e3*SAMPLE_RATE);assert(currentSampleOffset<=start&&start<=end);if(end>start){segments.push({start:start,end:end,kind:speakerLabelToTimelineSegmentKind(speakerLabels,firstPart.speakerLabel)})}currentSampleOffset=end}return segments}function overlaySegments(baseSegments,overlaySegments){let overlayIdx=0;const ret=[];let time=0;function forwardTime(end,kind){assert(end>=time);const length=end-time;if(length>0){ret.push([length,kind])}time=end}for(const[baseLength,baseKind]of baseSegments){const baseEnd=time+baseLength;while(overlayIdx<overlaySegments.length){const{start:start,end:end,kind:kind}=assertExists(overlaySegments[overlayIdx]);if(start>=baseEnd){break}if(start>=time){forwardTime(start,baseKind)}if(end>=baseEnd){forwardTime(baseEnd,kind);break}forwardTime(end,kind);overlayIdx++}forwardTime(baseEnd,baseKind)}return ret}const SIMPLIFY_SEGMENT_RESOLUTION=512;function simplifySegments(segments){const totalLength=segments.map((([length])=>length)).reduce(((a,b)=>a+b),0);const threshold=Math.max(totalLength/SIMPLIFY_SEGMENT_RESOLUTION,SAMPLE_RATE);const ret=[];let currentGroup=[];let currentGroupLength=0;function endCurrentGroup(){if(currentGroup.length===0){return}const kindLengths=new Map;for(const[length,kind]of currentGroup){kindLengths.set(kind,(kindLengths.get(kind)??0)+length)}let kind=null;let maxLength=0;for(const[k,length]of kindLengths.entries()){if(length>=maxLength){maxLength=length;kind=k}}ret.push([currentGroupLength,assertExists(kind)]);currentGroup=[];currentGroupLength=0}function push(segment){currentGroup.push(segment);currentGroupLength+=segment[0]}for(const segment of segments){const[length]=segment;if(length>=threshold){endCurrentGroup()}push(segment);if(currentGroupLength>=threshold){endCurrentGroup()}}endCurrentGroup();return ret}function calculateTimelineSegments(powers,samplesPerDataPoint,transcription){const powerSegments=calculatePowerSegments(powers,samplesPerDataPoint);const speechSegments=calculateSpeechSegments(transcription);const segments=simplifySegments(overlaySegments(powerSegments,speechSegments));return{version:TIMELINE_SEGMENT_VERSION,segments:segments}}export function getDefaultFileNameWithoutExtension(meta){return meta.title.replaceAll(":",".")}export class RecordingDataManager{static async create(dataDir){async function getMetadataFromFilename(name){const file=await dataDir.read(name);const text=await file.text();const data=recordingMetadataSchema.parseJson(text);return data}const filenames=await dataDir.list();const metadataMap=Object.fromEntries((await Promise.all(filenames.filter((x=>x.endsWith(".meta.json"))).map((async x=>{try{const meta=await getMetadataFromFilename(x);return[meta.id,meta]}catch(e){console.error(`Failed to parse metadata file.`,e);return null}})))).filter((x=>x!==null)));return new RecordingDataManager(dataDir,metadataMap)}constructor(dataDir,metadata){this.dataDir=dataDir;this.cachedMetadataMap=signal({});this.metadataWriteQueues=new Map;this.cachedMetadataMap.value=metadata;for(const[id,meta]of Object.entries(metadata)){this.fillDerivedMetadata(id,meta).catch((e=>{console.error(`error while filling derived data for ${id}`,e)}))}}async fillDerivedMetadata(id,meta){const getTranscription=asyncLazyInit((()=>this.getTranscription(id)));const getPowers=asyncLazyInit((()=>this.getAudioPower(id)));let changed=false;if(meta.numSpeakers===undefined){changed=true;const transcription=await getTranscription();meta={...meta,numSpeakers:transcription?.getSpeakerLabels().length??null}}if(meta.timelineSegments===undefined){changed=true;const transcription=await getTranscription();const{powers:powers,samplesPerDataPoint:samplesPerDataPoint}=await getPowers();meta={...meta,timelineSegments:calculateTimelineSegments(powers,samplesPerDataPoint,transcription)}}if(changed){this.setMetadata(id,meta)}}getWriteQueueFor(id){const queue=this.metadataWriteQueues.get(id);if(queue!==undefined){return queue}const newQueue=new AsyncJobQueue("keepLatest");this.metadataWriteQueues.set(id,newQueue);return newQueue}async createRecording({transcription:transcription,powers:powers,samplesPerDataPoint:samplesPerDataPoint,...meta},audio){const id=ulid();const numSpeakers=transcription?.getSpeakerLabels().length??null;const description=transcription?.toShortDescription()??"";const timelineSegments=calculateTimelineSegments(powers,samplesPerDataPoint,transcription);const fullMeta={id:id,description:description,timelineSegments:timelineSegments,numSpeakers:numSpeakers,...meta};this.setMetadata(id,fullMeta);await Promise.all([this.dataDir.write(audioPowerName(id),audioPowerSchema.stringifyJson({powers:powers,samplesPerDataPoint:samplesPerDataPoint})),this.dataDir.write(transcriptionName(id),transcriptionSchema.stringifyJson(transcription)),this.dataDir.write(audioName(id),audio)]);return id}getAllMetadata(){return this.cachedMetadataMap}getMetadataRaw(id){return this.cachedMetadataMap.value[id]??null}getMetadata(id){return computed((()=>this.getMetadataRaw(id)))}setMetadata(id,meta){this.cachedMetadataMap.mutate((m=>{m[id]=meta}));this.getWriteQueueFor(id).push((async()=>{await this.dataDir.write(metadataName(id),recordingMetadataSchema.stringifyJson(meta))}))}async getAudioFile(id){const name=audioName(id);const file=await this.dataDir.read(name);return file}async getTranscription(id){const name=transcriptionName(id);const file=await this.dataDir.read(name);const text=await file.text();return transcriptionSchema.parseJson(text)}async getAudioPower(id){const name=audioPowerName(id);const file=await this.dataDir.read(name);const text=await file.text();return audioPowerSchema.parseJson(text)}async clear(){await this.dataDir.clear();this.cachedMetadataMap.value={}}remove(id){this.cachedMetadataMap.mutate((m=>{delete m[id]}));this.getWriteQueueFor(id).push((async()=>{await this.dataDir.remove(metadataName(id))}));void this.dataDir.remove(audioName(id));void this.dataDir.remove(transcriptionName(id));void this.dataDir.remove(audioPowerName(id))}async exportAudio(id,format){const metadata=this.getMetadataRaw(id);if(metadata===null){return}const file=await this.getAudioFile(id);switch(format){case ExportAudioFormat.WEBM_ORIGINAL:{const filename=getDefaultFileNameWithoutExtension(metadata)+".webm";downloadFile(filename,file);break}default:assertExhaustive(format)}}async exportTranscription(id,format){const metadata=this.getMetadataRaw(id);if(metadata===null){return}const transcription=await this.getTranscription(id);if(transcription===null||transcription.isEmpty()){return}switch(format){case ExportTranscriptionFormat.TXT:{const text=transcription.toExportText();const blob=new Blob([text],{type:"text/plain"});const filename=getDefaultFileNameWithoutExtension(metadata)+".txt";downloadFile(filename,blob);break}default:assertExhaustive(format)}}async exportRecording(id,settings){if(settings.transcription){await this.exportTranscription(id,settings.transcriptionFormat)}if(settings.audio){await this.exportAudio(id,settings.audioFormat)}}}