// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import"chrome://resources/cros_components/menu/menu_separator.js";import"chrome://resources/cros_components/slider/slider.js";import"chrome://resources/mwc/@material/web/icon/icon.js";import"chrome://resources/mwc/@material/web/iconbutton/icon-button.js";import"../components/audio-waveform.js";import"../components/cra/cra-image.js";import"../components/cra/cra-menu-item.js";import"../components/cra/cra-menu.js";import"../components/delete-recording-dialog.js";import"../components/export-dialog.js";import"../components/recording-file-list.js";import"../components/recording-info-dialog.js";import"../components/recording-title.js";import"../components/secondary-button.js";import"../components/spoken-message.js";import"../components/summarization-view.js";import"../components/transcription-view.js";import"../components/time-duration.js";import{Slider as CrosSlider}from"chrome://resources/cros_components/slider/slider.js";import{classMap,createRef,css,html,live,nothing,ref}from"chrome://resources/mwc/lit/index.js";import{withTooltip}from"../components/directives/with-tooltip.js";import{SAMPLE_RATE,SAMPLES_PER_POWER_BAR}from"../core/audio_constants.js";import{AudioPlayerController}from"../core/audio_player_controller.js";import{i18n}from"../core/i18n.js";import{usePlatformHandler,useRecordingDataManager}from"../core/lit/context.js";import{ComputedState,ReactiveLitElement,ScopedAsyncComputed,ScopedEffect}from"../core/reactive/lit.js";import{computed,signal}from"../core/reactive/signal.js";import{navigateTo}from"../core/state/route.js";import{assert,assertExists,assertInstanceof}from"../core/utils/assert.js";import{AsyncJobQueue}from"../core/utils/async_job_queue.js";import{formatDuration}from"../core/utils/datetime.js";import{InteriorMutableArray}from"../core/utils/interior_mutable_array.js";import{sleep}from"../core/utils/utils.js";const PLAYBACK_SPEED_ICON_MAP=new Map([[.25,"rate_0_25"],[.5,"rate_0_5"],[.75,"rate_0_75"],[1,"rate_1_0"],[1.25,"rate_1_25"],[1.5,"rate_1_5"],[1.75,"rate_1_75"],[2,"rate_2_0"]]);const PLAYBACK_SPEEDS=Array.from(PLAYBACK_SPEED_ICON_MAP.keys());let initialAudio=null;export function setInitialAudio(audio){initialAudio=audio}export class PlaybackPage extends ReactiveLitElement{constructor(){super(...arguments);this.recordingId=null;this.recordingIdSignal=this.propSignal("recordingId");this.menu=createRef();this.deleteRecordingDialog=createRef();this.exportDialog=createRef();this.recordingInfoDialog=createRef();this.platformHandler=usePlatformHandler();this.recordingDataManager=useRecordingDataManager();this.playbackSpeedMenu=createRef();this.playbackSpeedMenuOpened=signal(false);this.floatingVolume=createRef();this.backButton=createRef();this.playPauseButton=createRef();this.transcriptionButtonRef=createRef();this.recordingTitle=createRef();this.summarizationView=createRef();this.audioPlayer=new AudioPlayerController(this,this.recordingIdSignal);this.recordingMetadata=computed((()=>{const id=this.recordingIdSignal.value;if(id===null){return null}return this.recordingDataManager.getMetadata(id).value}));this.transcription=new ScopedAsyncComputed(this,(async()=>{if(this.recordingIdSignal.value===null){return null}return this.recordingDataManager.getTranscription(this.recordingIdSignal.value)}));this.audioPowers=new ScopedAsyncComputed(this,(async()=>{if(this.recordingIdSignal.value===null){return null}const{powers:powers,samplesPerDataPoint:samplesPerDataPoint}=await this.recordingDataManager.getAudioPower(this.recordingIdSignal.value);const downSampleStep=Math.round(SAMPLES_PER_POWER_BAR/samplesPerDataPoint);if(downSampleStep<=1){return{powers:powers,samplesPerDataPoint:samplesPerDataPoint}}const downSamplePower=[];for(let i=0;i<powers.length;i+=downSampleStep){downSamplePower.push(assertExists(powers[i]))}return{powers:downSamplePower,samplesPerDataPoint:SAMPLES_PER_POWER_BAR}}));this.showTranscription=signal(false);this.spokenAudioTime=signal(null);this.spokenTimeQueue=new AsyncJobQueue("keepLatest");this.autoOpenTranscription=new ScopedEffect(this,(()=>{if(this.transcription.state===ComputedState.DONE){const transcription=this.transcription.valueSignal.peek();this.showTranscription.value=transcription!==null&&!transcription.isEmpty()}}))}static{this.styles=css`
    :host {
      --header-padding: 8px;

      background-color: var(--cros-sys-app_base_shaded);
      box-sizing: border-box;
      display: flex;
      flex-flow: column;
      height: 100%;
      padding: 16px;
      width: 100%;

      @container style(--small-viewport: 1) {
        --header-padding: 2px;
      }
    }

    #main-area {
      border-radius: 16px;
      display: flex;
      flex: 1;
      flex-flow: column;
      gap: 4px;

      /* To have the border-radius applied to content. */
      overflow: hidden;
    }

    .sheet {
      background-color: var(--cros-sys-app_base);
      border-radius: 4px;
    }

    #header {
      align-items: center;
      display: flex;
      flex-flow: row;
      padding: var(--header-padding);

      & > recording-title {
        margin: 0 auto 0 -4px;
      }
    }

    #middle {
      display: flex;
      flex: 1;
      flex-flow: row;
      gap: 4px;
    }

    #audio-waveform-container,
    #transcription-container {
      /*
       * Makes both side full height without having these "expand" the main
       * area height when it's too high.
       */
      box-sizing: border-box;
      flex: 1;
      height: 0;
      min-height: 100%;
    }

    #transcription-container {
      display: none;
    }

    .show-transcription {
      #transcription-container {
        display: block;
      }

      @container style(--small-viewport: 1) {
        #audio-waveform-container {
          display: none;
        }
      }
    }

    #transcription-empty {
      align-items: center;
      display: flex;
      flex-flow: column;
      font: var(--cros-headline-1-font);
      gap: 16px;
      height: 100%;
      justify-content: center;
      width: 100%;
    }

    audio-waveform,
    transcription-view {
      box-sizing: border-box;
      height: 100%;
      width: 100%;
    }

    #audio-waveform-container {
      align-items: center;
      display: flex;
      flex-flow: column;
      justify-content: center;

      & > time-duration {
        color: var(--cros-sys-on_surface_variant);
        font: 440 24px/32px var(--monospace-font-family);
        letter-spacing: 0.03em;
        margin-bottom: 4px;
        position: absolute;
        position-anchor: --waveform;
        position-area: top;
      }
    }

    audio-waveform {
      height: 180px;
      anchor-name: --waveform;
    }

    #footer {
      align-items: stretch;
      display: flex;
      flex-flow: column;
      gap: 16px;
      padding: 16px 8px;
    }

    #audio-timeline {
      align-items: stretch;
      display: flex;
      flex-flow: column;

      & > div {
        display: flex;
        flex-flow: row;
        font: var(--cros-body-2-font);
        justify-content: space-between;
        padding: 0 16px;
      }
    }

    #actions {
      align-items: center;
      display: flex;
      justify-content: center;
      position: relative;
    }

    #volume-controls {
      left: 0;
      position: absolute;
    }

    #inline-slider {
      align-items: center;
      display: flex;

      & > cra-icon-button {
        margin-right: 0;
      }

      & > cros-slider {
        min-inline-size: initial;
        width: 180px;
      }

      @container style(--small-viewport: 1) {
        display: none;
      }
    }

    #floating-slider-base {
      display: none;

      @container style(--small-viewport: 1) {
        display: block;
      }

      & > cra-icon-button {
        anchor-name: --volume-button;
        margin: 0;
        padding: 4px;
      }

      /* TODO(pihsun): Animate the opening/closing */
      & > [popover]:popover-open {
        align-items: center;
        background: var(--cros-sys-base_elevated);
        border: none;
        border-radius: 8px;
        box-shadow: var(--cros-sys-app_elevation3);
        display: flex;
        flex-flow: row;
        margin: initial;
        overflow: initial;
        padding: 0;
        position: absolute;
        position-anchor: --volume-button;
        position-area: center span-right;

        & > cros-slider {
          min-inline-size: initial;
          width: 180px;
        }
      }
    }

    #middle-controls {
      align-items: center;
      display: flex;
      flex-flow: row;
      gap: 28px;
    }

    #speed-controls {
      position: absolute;
      right: 0;
    }

    #play-button {
      --cra-icon-button-container-color: var(--cros-sys-primary);
      --cra-icon-button-container-height: 96px;
      --cra-icon-button-container-width: 152px;
      --cra-icon-button-hover-state-layer-color: var(
        --cros-sys-hover_on_prominent
      );
      --cra-icon-button-pressed-state-layer-color: var(
        --cros-sys-ripple_primary
      );
      --cros-icon-button-color-override: var(--cros-sys-on_primary);
      --cros-icon-button-icon-size: 32px;

      @container style(--small-viewport: 1) {
        --cra-icon-button-container-height: 80px;
        --cra-icon-button-container-width: 136px;
      }

      margin: 0;
    }

    summarization-view {
      padding: 0 12px;
    }

    #menu {
      --cros-menu-width: 200px;
    }

    #speed-menu {
      --cros-menu-width: 160px;
    }
  `}static{this.properties={recordingId:{type:String}}}get pauseButtonForTest(){assert(this.audioPlayer.playing.value,"The playback is already paused");return assertExists(this.playPauseButton.value)}get backButtonForTest(){return assertExists(this.backButton.value)}get transcriptionToggleButtonForTest(){return assertExists(this.transcriptionButtonRef.value)}get recordingTitleForTest(){return assertExists(this.recordingTitle.value)}get summarizationViewForTest(){return assertExists(this.summarizationView.value)}firstUpdated(){const backButton=assertExists(this.backButton.value);backButton.updateComplete.then((()=>{backButton.focus()}))}connectedCallback(){super.connectedCallback();if(initialAudio!==null){this.audioPlayer.setInnerAudio(initialAudio);initialAudio=null}}onWordClick(ev){this.updateAudioTime(ev.detail.startMs/1e3)}onPlayPauseClick(){this.audioPlayer.togglePlaying()}toggleTranscription(){this.showTranscription.update((s=>!s))}renderAudioWaveform(){if(this.audioPowers.value===null){return nothing}const currentTime=this.audioPlayer.currentTime.value;const duration={seconds:currentTime};const barsPerSecond=SAMPLE_RATE/this.audioPowers.value.samplesPerDataPoint;return html`
      <time-duration
        digits=1
        .duration=${duration}
      ></time-duration>
      <audio-waveform
        .values=${new InteriorMutableArray(this.audioPowers.value.powers)}
        .currentTime=${currentTime}
        .transcription=${this.transcription.value}
        .barsPerSecond=${barsPerSecond}
      >
      </audio-waveform>
    `}renderTranscription(){const transcription=this.transcription.value;if(transcription===null){return nothing}if(transcription.isEmpty()){return html`<div id="transcription-empty">
        <cra-image name="transcription_no_speech"></cra-image>
        ${i18n.transcriptionNoSpeechText}
      </div>`}return html`<transcription-view
      .transcription=${transcription}
      @word-clicked=${this.onWordClick}
      .currentTime=${this.audioPlayer.currentTime.value}
      seekable
    >
      <summarization-view
        .transcription=${transcription}
        ${ref(this.summarizationView)}
      ></summarization-view>
    </transcription-view>`}renderPlayPauseButton(){const ariaLabel=this.audioPlayer.playing.value?i18n.playbackPauseButtonTooltip:i18n.playbackPlayButtonTooltip;return html`<cra-icon-button
      id="play-button"
      shape="circle"
      @click=${this.onPlayPauseClick}
      ${ref(this.playPauseButton)}
      aria-label=${ariaLabel}
      ${withTooltip()}
    >
      <cra-icon
        slot="icon"
        .name=${this.audioPlayer.playing.value?"pause_hero":"play_hero"}
      ></cra-icon>
    </cra-icon-button>`}onTimelineInput(ev){const target=assertInstanceof(ev.target,CrosSlider);this.audioPlayer.currentTime.value=target.value}updateAudioTime(updatedTime){this.audioPlayer.currentTime.value=updatedTime;this.spokenTimeQueue.push((async()=>{this.spokenAudioTime.value=updatedTime;await sleep(100);this.spokenAudioTime.value=null}))}onForward10Secs(){this.updateAudioTime(this.audioPlayer.currentTime.value+10)}onRewind10Secs(){this.updateAudioTime(this.audioPlayer.currentTime.value-10)}onDeleteClick(){this.deleteRecordingDialog.value?.show()}deleteRecording(){if(this.recordingId!==null){this.recordingDataManager.remove(this.recordingId);navigateTo("main")}}onExportClick(){this.exportDialog.value?.show()}onShowDetailClick(){this.recordingInfoDialog.value?.show()}renderMenu(){return html`
      <cra-menu ${ref(this.menu)} anchor="show-menu" id="menu">
        <cra-menu-item
          headline=${i18n.playbackMenuExportOption}
          @cros-menu-item-triggered=${this.onExportClick}
        ></cra-menu-item>
        <cra-menu-item
          headline=${i18n.playbackMenuShowDetailOption}
          @cros-menu-item-triggered=${this.onShowDetailClick}
        ></cra-menu-item>
        <cros-menu-separator></cros-menu-separator>
        <cra-menu-item
          headline=${i18n.playbackMenuDeleteOption}
          @cros-menu-item-triggered=${this.onDeleteClick}
        ></cra-menu-item>
      </cra-menu>
      <recording-info-dialog
        ${ref(this.recordingInfoDialog)}
        .recordingId=${this.recordingId}
      ></recording-info-dialog>
      <export-dialog ${ref(this.exportDialog)} .recordingId=${this.recordingId}>
      </export-dialog>
      <delete-recording-dialog
        ${ref(this.deleteRecordingDialog)}
        @delete=${this.deleteRecording}
      >
      </delete-recording-dialog>
    `}toggleMenu(){this.menu.value?.toggle()}renderHeader(){const transcriptionLabel=this.showTranscription.value?i18n.playbackHideTranscriptButtonTooltip:i18n.playbackShowTranscriptButtonTooltip;const transcriptionToggleButton=this.transcription.value===null?nothing:html`
            <cra-icon-button
              buttonstyle="toggle"
              .selected=${live(this.showTranscription.value)}
              @click=${this.toggleTranscription}
              ${ref(this.transcriptionButtonRef)}
              aria-label=${transcriptionLabel}
              ${withTooltip()}
            >
              <cra-icon slot="icon" name="notes"></cra-icon>
              <cra-icon slot="selectedIcon" name="notes"></cra-icon>
            </cra-icon-button>
          `;return html`
      <div id="header" class="sheet">
        <cra-icon-button
          buttonstyle="floating"
          @click=${()=>navigateTo("main")}
          ${ref(this.backButton)}
          aria-label=${i18n.backToMainButtonAriaLabel}
          ${withTooltip(i18n.backToMainButtonTooltip)}
        >
          <cra-icon slot="icon" name="arrow_back"></cra-icon>
        </cra-icon-button>
        <recording-title
          .recordingMetadata=${this.recordingMetadata.value}
          ${ref(this.recordingTitle)}
        >
        </recording-title>
        ${transcriptionToggleButton}
        <cra-icon-button
          buttonstyle="floating"
          id="show-menu"
          @click=${this.toggleMenu}
          aria-label=${i18n.playbackMenuButtonTooltip}
          ${withTooltip(i18n.playbackMenuButtonTooltip)}
        >
          <cra-icon slot="icon" name="more_vertical"></cra-icon>
        </cra-icon-button>
        ${this.renderMenu()}
      </div>
    `}renderTimeSpokenStatus(){const spokenTime=this.spokenAudioTime.value;if(spokenTime===null){return nothing}return html`<spoken-message aria-live="polite" role="status">
      ${formatDuration({seconds:spokenTime},1,true)}
    </spoken-message>`}renderAudioTimeline(){if(this.recordingMetadata.value===null){return nothing}const currentTime=this.audioPlayer.currentTime.value;const currentTimeStringLabel=formatDuration({seconds:currentTime},0,true);const currentDuration={seconds:currentTime};const totalDuration={milliseconds:this.recordingMetadata.value.durationMs};return html`<div id="audio-timeline">
      <cros-slider
        min="0"
        max=${this.recordingMetadata.value.durationMs/1e3}
        step="0.1"
        .value=${currentTime}
        @input=${this.onTimelineInput}
        aria-valuetext=${currentTimeStringLabel}
        aria-label=${i18n.playbackSeekSliderAriaLabel}
      ></cros-slider>
      <div>
        <time-duration
          .duration=${currentDuration}
        ></time-duration>
        <time-duration
          .duration=${totalDuration}
        ></time-duration>
      </div>
      ${this.renderTimeSpokenStatus()}
    </div>`}getSpeedLabel(speed){return speed===1?i18n.playbackSpeedNormalOption:speed.toString()}getPlaybackSpeedControlLabel(){return`${i18n.playbackSpeedButtonTooltip}: ${this.getSpeedLabel(this.audioPlayer.playbackSpeed.value)}`}renderSpeedControl(){const iconName=assertExists(PLAYBACK_SPEED_ICON_MAP.get(this.audioPlayer.playbackSpeed.value));const menuItems=PLAYBACK_SPEEDS.map((speed=>{const label=this.getSpeedLabel(speed);const onClick=()=>{this.audioPlayer.playbackSpeed.value=speed;this.platformHandler.eventsSender.sendChangePlaybackSpeedEvent({playbackSpeed:speed})};return html`<cra-menu-item
        headline=${label}
        ?checked=${this.audioPlayer.playbackSpeed.value===speed}
        @cros-menu-item-triggered=${onClick}
        data-role="menuitemradio"
      ></cra-menu-item>`}));const onMenuOpen=()=>{this.playbackSpeedMenuOpened.value=true};const onMenuClose=()=>{this.playbackSpeedMenuOpened.value=false};const togglePlaybackSpeedMenu=()=>{this.playbackSpeedMenu.value?.toggle()};const classes={selected:this.playbackSpeedMenuOpened.value};return html`
      <cra-menu
        ${ref(this.playbackSpeedMenu)}
        anchor="show-speed-menu"
        id="speed-menu"
        @opened=${onMenuOpen}
        @closed=${onMenuClose}
      >
        ${menuItems}
      </cra-menu>
      <cra-icon-button
        buttonstyle="filled"
        class="with-toggle-style ${classMap(classes)}"
        id="show-speed-menu"
        @click=${togglePlaybackSpeedMenu}
        aria-haspopup="true"
        aria-label=${this.getPlaybackSpeedControlLabel()}
        ${withTooltip(i18n.playbackSpeedButtonTooltip)}
      >
        <cra-icon slot="icon" .name=${iconName}></cra-icon>
      </cra-icon-button>
    `}updateVolume(muted,volume){this.audioPlayer.muted.value=muted;this.audioPlayer.volume.value=volume/100;this.platformHandler.eventsSender.sendChangePlaybackVolumeEvent({muted:muted,volume:volume})}onVolumeInput(ev){const slider=assertInstanceof(ev.target,CrosSlider);this.updateVolume(false,slider.value);this.requestUpdate()}toggleMuted(){const muted=!this.audioPlayer.muted.value;const volume=Math.round(this.audioPlayer.volume.value*100);this.updateVolume(muted,volume);this.requestUpdate()}renderVolumeIcon(){const{muted:muted,volume:volume}=this.audioPlayer;const iconName=(()=>{if(muted.value){return"volume_mute"}if(volume.value===0){return"volume_off"}return volume.value<.5?"volume_down":"volume_up"})();return html`<cra-icon slot="icon" .name=${iconName}></cra-icon>`}renderVolumeSlider(){const{muted:muted,volume:volume}=this.audioPlayer;const volumeDisplay=muted.value?0:Math.round(volume.value*100);return html`
      <cros-slider
        withlabel
        .value=${volumeDisplay}
        min="0"
        max="100"
        @input=${this.onVolumeInput}
        aria-label=${i18n.playbackVolumeSliderAriaLabel}
      ></cros-slider>
    `}showFloatingVolume(){this.floatingVolume.value?.showPopover()}hideFloatingVolume(ev){const newTarget=ev.relatedTarget;if(newTarget!==null&&newTarget instanceof Node&&this.floatingVolume.value?.contains(newTarget)){return}this.floatingVolume.value?.hidePopover()}renderVolumeControl(){const volumeButtonLabel=this.audioPlayer.muted.value?i18n.playbackUnmuteButtonTooltip:i18n.playbackMuteButtonTooltip;return html`
      <div id="inline-slider">
        <cra-icon-button
          buttonstyle="toggle"
          @click=${this.toggleMuted}
          aria-label=${volumeButtonLabel}
          class="with-floating-style"
          .selected=${this.audioPlayer.muted.value}
          ${withTooltip()}
        >
          ${this.renderVolumeIcon()}
          <cra-icon slot="selectedIcon" name="volume_mute"></cra-icon>
        </cra-icon-button>
        ${this.renderVolumeSlider()}
      </div>
      <div id="floating-slider-base">
        <cra-icon-button
          buttonstyle="floating"
          @click=${this.showFloatingVolume}
          aria-label=${i18n.playbackFloatingVolumeShowButtonAriaLabel}
          ${withTooltip()}
        >
          ${this.renderVolumeIcon()}
        </cra-icon-button>
        <div
          popover
          ${ref(this.floatingVolume)}
          @focusout=${this.hideFloatingVolume}
        >
          <cra-icon-button
            buttonstyle="floating"
            @click=${this.toggleMuted}
            aria-label=${volumeButtonLabel}
            ${withTooltip()}
          >
            ${this.renderVolumeIcon()}
          </cra-icon-button>
          ${this.renderVolumeSlider()}
          <cra-icon-button
            buttonstyle="floating"
            @click=${this.hideFloatingVolume}
            aria-label=${i18n.playbackFloatingVolumeCloseButtonAriaLabel}
            ${withTooltip()}
          >
            <cra-icon slot="icon" name="close"></cra-icon>
          </cra-icon-button>
        </div>
      </div>
    `}render(){const mainSectionClasses={"show-transcription":this.showTranscription.value};return html`
      <div id="main-area">
        ${this.renderHeader()}
        <div id="middle" class=${classMap(mainSectionClasses)}>
          <div id="audio-waveform-container" class="sheet">
            ${this.renderAudioWaveform()}
          </div>
          <div
            id="transcription-container"
            class="sheet"
            aria-label=${i18n.playbackTranscriptLandmarkAriaLabel}
            role="region"
          >
            ${this.renderTranscription()}
          </div>
        </div>
      </div>
      <div
        id="footer"
        aria-label=${i18n.playbackControlsLandmarkAriaLabel}
        role="complementary"
      >
        ${this.renderAudioTimeline()}
        <div id="actions">
          <div id="volume-controls">${this.renderVolumeControl()}</div>
          <div id="middle-controls">
            <secondary-button
              @click=${this.onRewind10Secs}
              aria-label=${i18n.playbackBackwardButtonTooltip}
              ${withTooltip()}
            >
              <cra-icon slot="icon" name="replay_10"></cra-icon>
            </secondary-button>
            ${this.renderPlayPauseButton()}
            <secondary-button
              @click=${this.onForward10Secs}
              aria-label=${i18n.playbackForwardButtonTooltip}
              ${withTooltip()}
            >
              <cra-icon slot="icon" name="forward_10"></cra-icon>
            </secondary-button>
          </div>
          <div id="speed-controls">${this.renderSpeedControl()}</div>
        </div>
      </div>
    `}}window.customElements.define("playback-page",PlaybackPage);