// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { classMap, css, html, nothing, repeat, svg, } from 'chrome://resources/mwc/lit/index.js';
import { POWER_BARS_PER_SECOND, POWER_SCALE_FACTOR, } from '../core/audio_constants.js';
import { i18n } from '../core/i18n.js';
import { ReactiveLitElement } from '../core/reactive/lit.js';
import { computed } from '../core/reactive/signal.js';
import { assert, assertExists, assertInstanceof, } from '../core/utils/assert.js';
import { InteriorMutableArray } from '../core/utils/interior_mutable_array.js';
import { getNumSpeakerClass, getSpeakerLabelClass, SPEAKER_LABEL_COLORS, } from './styles/speaker_label.js';
const BAR_WIDTH = 4;
const BAR_GAP = 5;
const BAR_MIN_HEIGHT = 4.5;
const BAR_MAX_HEIGHT = 100;
const SPEAKER_LABEL_LINE_HEIGHT = 128;
// TODO(pihsun): Is there some way to set .viewBox.baseVal?
function toViewBoxString(viewBox) {
    if (viewBox === null) {
        return nothing;
    }
    const { x, y, width, height } = viewBox;
    return `${x} ${y} ${width} ${height}`;
}
/*
 * There are multiple different coordinate system for the "timestamp" of the
 * waveform used in this component:
 * (1) Time (in seconds). Each second contains `barsPerSecond` bars.
 * (2) Index of the "bar" in the waveform, starting from 0.
 * (3) The x coordinate that is rendered in the SVG. Time 0 always corresponds
 *     to x = 0, and the viewBox of the whole SVG is set to show around the
 *     current time.
 *
 * `timestampToBarIndex` converts from (1) to (2), `getBarX` converts from
 * (2) to (3), and `xCoordinateToRoughIdx` converts from (3) to (2).
 *
 * Since the whole waveform looks better when things are aligned to bar, most
 * variables (ended in BarIdx) are in the coordinate of (2).
 *
 * Also note that to render separator between bars, sometimes the "index" in (2)
 * have 0.5 in fractions, but those values should only be used to convert to
 * rendered x coordinate (3) and doesn't corresponds to actual slice of audio
 * samples.
 */
function timestampToBarIndex(seconds, barsPerSecond) {
    return Math.floor(seconds * barsPerSecond);
}
function getBarX(barIdx) {
    return barIdx * (BAR_WIDTH + BAR_GAP);
}
function xCoordinateToRoughIdx(x) {
    return Math.floor(x / (BAR_WIDTH + BAR_GAP));
}
/**
 * Component for showing audio waveform.
 */
export class AudioWaveform extends ReactiveLitElement {
    constructor() {
        super(...arguments);
        // Values to be shown as bars. Should be in range [0, POWER_SCALE_FACTOR - 1].
        this.values = new InteriorMutableArray([]);
        this.currentTime = null;
        this.barsPerSecond = POWER_BARS_PER_SECOND;
        this.currentTimeSignal = this.propSignal('currentTime');
        this.currentTimeBarIdx = computed(() => {
            if (this.currentTimeSignal.value === null) {
                return null;
            }
            return timestampToBarIndex(this.currentTimeSignal.value, this.barsPerSecond);
        });
        this.size = null;
        this.transcription = null;
        this.transcriptionSignal = this.propSignal('transcription');
        this.speakerLabelInfo = computed(() => {
            const transcription = this.transcriptionSignal.value;
            if (transcription === null) {
                return {
                    speakerLabels: [],
                    ranges: [],
                };
            }
            const paragraphs = transcription.getParagraphs();
            const speakerLabels = transcription.getSpeakerLabels();
            const ranges = [];
            for (const paragraph of paragraphs) {
                const firstPart = assertExists(paragraph[0]);
                const lastPart = assertExists(paragraph.at(-1));
                const speakerLabel = firstPart.speakerLabel;
                if (speakerLabel === null) {
                    // The paragraph doesn't have speaker label.
                    continue;
                }
                const speakerLabelIndex = speakerLabels.indexOf(speakerLabel);
                assert(speakerLabelIndex !== -1);
                const startMs = firstPart.timeRange?.startMs ?? null;
                const endMs = lastPart.timeRange?.endMs ?? null;
                if (startMs === null || endMs === null) {
                    // TODO(pihsun): Check if there's any possibility that the timestamp is
                    // missing.
                    continue;
                }
                // The timestamps should be increasing.
                assert(startMs <= endMs);
                const startBarIdx = timestampToBarIndex(startMs / 1000, this.barsPerSecond);
                const endBarIdx = timestampToBarIndex(endMs / 1000, this.barsPerSecond);
                assert(ranges.length === 0 ||
                    assertExists(ranges.at(-1)).endBarIdx <= startBarIdx);
                // These can be equal if there's a very short paragraph with speaker
                // label.
                if (startBarIdx !== endBarIdx) {
                    ranges.push({
                        speakerLabelIndex,
                        startBarIdx,
                        endBarIdx,
                    });
                }
            }
            return {
                speakerLabels,
                ranges,
            };
        });
        this.resizeObserver = new ResizeObserver(() => {
            this.size = this.getBoundingClientRect();
        });
    }
    static { this.styles = [
        SPEAKER_LABEL_COLORS,
        css `
      :host {
        display: block;
        position: relative;
      }

      #chart {
        inset: 0;
        position: absolute;
      }

      .speaker-single {
        & .range {
          display: none;
        }
      }

      .speaker-duo,
      .speaker-multiple {
        & .no-speaker {
          --speaker-label-shapes-color: var(--cros-sys-primary_container);
        }
      }

      .speaker-range-start {
        /* The dash and space looks equal length with rounded linecap. */
        stroke-dasharray: 2, 6;
        stroke-linecap: round;
        stroke-width: 2;
        stroke: var(--speaker-label-shapes-color);

        .speaker-single & {
          display: none;
        }

        &.future {
          opacity: var(--cros-disabled-opacity);
        }

        .range:hover & {
          stroke-dasharray: none;
        }
      }

      .bar {
        /* Don't block hover on the background. */
        pointer-events: none;

        .speaker-single & {
          fill: var(--cros-sys-primary);

          &.future {
            fill: var(--cros-sys-primary_container);
          }
        }

        :is(.speaker-duo, .speaker-multiple) & {
          fill: var(--speaker-label-shapes-color);

          &.future {
            opacity: var(--cros-disabled-opacity);
          }
        }
      }

      .background {
        /* fill: none prevents :hover state, so we set opacity: 0 instead. */
        opacity: 0;
        fill: var(--speaker-label-container-color);

        .range:hover & {
          opacity: 1;

          &.future {
            opacity: var(--cros-disabled-opacity);
          }
        }
      }

      .speaker-label {
        align-items: center;
        background: var(--speaker-label-shapes-color);
        border-radius: 10px 10px 10px 0;
        bottom: 0;
        box-sizing: border-box;
        color: var(--speaker-label-label-color);
        display: flex;
        font: var(--cros-label-1-font);
        height: 20px;
        justify-content: center;
        left: 0;
        min-width: 20px;
        padding: 4px;
        position: absolute;
        width: fit-content;

        &.outside {
          display: none;
        }

        & > .full {
          display: none;
        }

        .range:hover & {
          display: block;

          /* TODO: b/336963138 - Animation on hover? */
          height: 26px;
          padding: 8px;

          & > .full {
            display: inline;
          }

          & > .short {
            display: none;
          }
        }
      }

      .playhead {
        fill: var(--cros-sys-on_surface_variant);

        /* Don't block hover on the background. */
        pointer-events: none;
      }
    `,
    ]; }
    static { this.properties = {
        values: { attribute: false },
        size: { state: true },
        currentTime: { type: Number },
        barsPerSecond: { attribute: false },
        transcription: { attribute: false },
    }; }
    get chart() {
        return assertInstanceof(assertExists(this.shadowRoot).querySelector('#chart'), SVGElement);
    }
    // TODO(pihsun): Check if we can use ResizeObserver in @lit-labs/observers.
    connectedCallback() {
        super.connectedCallback();
        this.resizeObserver.observe(this);
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.resizeObserver.disconnect();
    }
    getBarLocation(idx, val, minHeight, maxHeight) {
        const width = BAR_WIDTH;
        const height = minHeight + (maxHeight - minHeight) * (val / (POWER_SCALE_FACTOR - 1));
        const x = getBarX(idx) - width / 2;
        const y = -height / 2;
        return { x, y, width, height };
    }
    isAfterCurrentTime(idx) {
        return (this.currentTimeBarIdx.value !== null &&
            idx >= this.currentTimeBarIdx.value);
    }
    renderSpeakerRangeStart({ startBarIdx, speakerLabelIndex, }) {
        const startX = getBarX(startBarIdx - 0.5);
        const classes = {
            [getSpeakerLabelClass(speakerLabelIndex)]: true,
            future: this.isAfterCurrentTime(startBarIdx),
        };
        const height = SPEAKER_LABEL_LINE_HEIGHT;
        // clang-format off
        return svg `<line
      x1=${startX}
      x2=${startX}
      y1=${-height / 2}
      y2=${height / 2}
      class="speaker-range-start ${classMap(classes)} "
    />`;
        // clang-format on
    }
    renderSpeakerRangeLabel(speakerLabels, { startBarIdx, speakerLabelIndex }, viewBox) {
        // minus one so it aligns with the left edge of the speaker label range
        // start.
        const startX = getBarX(startBarIdx - 0.5) - 1;
        const classes = {
            [getSpeakerLabelClass(speakerLabelIndex)]: true,
            outside: startX < viewBox.x,
        };
        const maxHeight = 26;
        // Always render the label in view. It'll be hidden until hover if it's
        // originally outside of the view. Note that only the label that has some
        // corresponding bar inside view will be rendered (see renderSvgContent).
        const x = Math.max(startX, viewBox.x);
        const y = -SPEAKER_LABEL_LINE_HEIGHT / 2 - maxHeight;
        const shortLabel = assertExists(speakerLabels[speakerLabelIndex]);
        const fullLabel = i18n.transcriptionSpeakerLabelLabel(shortLabel);
        // clang-format off
        // The width/height on foreignObject is necessary for the div to be shown,
        // but the actual label size can be smaller than that.
        // TODO(pihsun): This introduce a bit more hover space than the visible
        // labels. Check if there's a better way to do this.
        return svg `<foreignObject
      x=${x}
      y=${y}
      width="100"
      height=${maxHeight}
    >
      <div class="speaker-label ${classMap(classes)}" aria-label=${fullLabel}>
        <span class="short" aria-hidden="true">${shortLabel}</span>
        <span class="full" aria-hidden="true">${fullLabel}</span>
      </div>
    </foreignObject>`;
        // clang-format on
    }
    /**
     * Returns the background path with the top-right and bottom-right corner
     * rounded.
     */
    getBackgroundPath(startX, endX) {
        const height = SPEAKER_LABEL_LINE_HEIGHT;
        const radius = 12;
        // clang-format off
        return `
      M ${startX} ${-height / 2}
      v ${height}
      H ${endX - radius}
      a ${radius} ${radius} 0 0 0 ${radius} ${-radius}
      V ${-height / 2 + radius}
      a ${radius} ${radius} 0 0 0 ${-radius} ${-radius}
      H ${startX}
    `;
        // clang-format on
    }
    renderSpeakerRangeBackground({ startBarIdx, endBarIdx, speakerLabelIndex, }) {
        const startX = getBarX(startBarIdx - 0.5);
        const endX = getBarX(endBarIdx - 0.5);
        const classes = {
            [getSpeakerLabelClass(speakerLabelIndex)]: true,
        };
        const currentTimeIdx = this.currentTimeBarIdx.value;
        if (currentTimeIdx !== null && startBarIdx <= currentTimeIdx &&
            currentTimeIdx < endBarIdx) {
            // Part of the background are before and part are after. Need to cut the
            // background in half.
            const centerX = getBarX(currentTimeIdx) - BAR_WIDTH / 2;
            const height = SPEAKER_LABEL_LINE_HEIGHT;
            const y = -height / 2;
            return [
                svg `<rect
          x=${startX}
          y=${y}
          width=${centerX - startX}
          height=${height}
          class="background ${classMap(classes)}"
        />`,
                svg `<path
          d=${this.getBackgroundPath(centerX, endX)}
          class="background future ${classMap(classes)}"
        />`,
            ];
        }
        else {
            classes['future'] = this.isAfterCurrentTime(startBarIdx);
            return svg `<path
        d=${this.getBackgroundPath(startX, endX)}
        class="background ${classMap(classes)}"
      />`;
        }
    }
    renderSpeakerRange(speakerLabels, range, viewBox) {
        return svg `<g class="range">
      ${this.renderSpeakerRangeBackground(range)}
      ${this.renderSpeakerRangeStart(range)}
      ${this.renderSpeakerRangeLabel(speakerLabels, range, viewBox)}
    </g>`;
    }
    renderCurrentTimeBar(viewBox) {
        if (this.currentTimeBarIdx.value === null) {
            return nothing;
        }
        const width = 2;
        // Add the progress indicator at the current time. Draw on the left side
        // of the current bar so it looks more "correct" when jumping to the start
        // of a paragraph with speaker label.
        const x = getBarX(this.currentTimeBarIdx.value) - BAR_WIDTH / 2 - width;
        const y = viewBox.y;
        return svg `<rect
      x=${x}
      y=${y}
      width=${width}
      height=${viewBox.height}
      rx="1"
      class="playhead"
    />`;
    }
    renderAudioBars(viewBox) {
        if (this.values.length === 0) {
            return nothing;
        }
        const speakerLabelRanges = this.speakerLabelInfo.value.ranges;
        let currentSpeakerLabelRangeIdx = 0;
        let currentSpeakerLabelRangeRendered = false;
        /**
         * Gets the speaker label index of a bar index.
         *
         * The `barIdx` given to this function needs to be increasing across
         * multiple calls, since this is implemented by scanning through the
         * speakerLabelRanges.
         */
        function getSpeakerLabelRange(barIdx) {
            while (currentSpeakerLabelRangeIdx < speakerLabelRanges.length) {
                const range = assertExists(speakerLabelRanges[currentSpeakerLabelRangeIdx]);
                if (barIdx < range.startBarIdx) {
                    return null;
                }
                if (barIdx < range.endBarIdx) {
                    return range;
                }
                currentSpeakerLabelRangeIdx += 1;
                currentSpeakerLabelRangeRendered = false;
            }
            return null;
        }
        // This is an optimization to not goes through the whole values array, and
        // directly calculate the part that needs to be rendered instead. To
        // simplify the logic we calculate the rough range and just extend it a bit
        // to make sure we covers the whole range.
        const startIdx = Math.max(xCoordinateToRoughIdx(viewBox.x) - 5, 0);
        const endIdx = Math.min(xCoordinateToRoughIdx(viewBox.x + viewBox.width) + 5, this.values.length - 1);
        if (endIdx < startIdx) {
            return nothing;
        }
        const idxRange = Array.from({ length: endIdx - startIdx + 1 }, (_, i) => i + startIdx);
        const toRenderBars = [];
        const toRenderSpeakerLabelRanges = [];
        for (const i of idxRange) {
            const val = assertExists(this.values.array[i]);
            const rect = this.getBarLocation(i, val, BAR_MIN_HEIGHT, Math.min(viewBox.height, BAR_MAX_HEIGHT));
            if (rect.x + rect.width < viewBox.x ||
                rect.x > viewBox.x + viewBox.width) {
                continue;
            }
            const classes = {
                future: this.isAfterCurrentTime(i),
            };
            const range = getSpeakerLabelRange(i);
            if (range !== null) {
                if (!currentSpeakerLabelRangeRendered) {
                    toRenderSpeakerLabelRanges.push(range);
                    currentSpeakerLabelRangeRendered = true;
                }
                classes[getSpeakerLabelClass(range.speakerLabelIndex)] = true;
            }
            else {
                classes['no-speaker'] = true;
            }
            toRenderBars.push({ idx: i, rect, classes });
        }
        return [
            repeat(toRenderSpeakerLabelRanges, ({ startBarIdx }) => startBarIdx, (range) => this.renderSpeakerRange(this.speakerLabelInfo.value.speakerLabels, range, viewBox)),
            repeat(toRenderBars, ({ idx }) => idx, ({ rect, classes }) => {
                return svg `<rect
          x=${rect.x}
          y=${rect.y}
          width=${rect.width}
          height=${rect.height}
          rx=${rect.width / 2}
          class="bar ${classMap(classes)}"
        />`;
            }),
        ];
    }
    renderSvgContent(viewBox) {
        if (viewBox === null) {
            return nothing;
        }
        return [this.renderAudioBars(viewBox), this.renderCurrentTimeBar(viewBox)];
    }
    getViewBox() {
        if (this.size === null) {
            return null;
        }
        const { width, height } = this.size;
        const x = (() => {
            if (this.currentTimeBarIdx.value !== null) {
                const x = getBarX(this.currentTimeBarIdx.value);
                // Put the current time in the center.
                // TODO(pihsun): Should this be controlled by a separate property?
                // TODO(pihsun): Should we use the real time offset, instead of
                // aligning to the bar?
                return x - width / 2;
            }
            else {
                return this.values.length * (BAR_WIDTH + BAR_GAP) - width;
            }
        })();
        const y = -height / 2;
        return { x, y, width, height };
    }
    render() {
        if (this.size === null) {
            return nothing;
        }
        const numSpeakerClass = getNumSpeakerClass(this.speakerLabelInfo.value.speakerLabels.length);
        const viewBox = this.getViewBox();
        // TODO(pihsun): Performance doesn't seem to be ideal for rendering this
        // with svg. Measure it for longer recording and see if there's other way to
        // do it. (Draw on canvas directly?)
        return html `<svg
      id="chart"
      viewBox=${toViewBoxString(viewBox)}
      class=${numSpeakerClass}
    >
      ${this.renderSvgContent(viewBox)}
    </svg>`;
    }
}
window.customElements.define('audio-waveform', AudioWaveform);
