// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import 'chrome://resources/cros_components/tooltip/tooltip.js';
import { Tooltip } from 'chrome://resources/cros_components/tooltip/tooltip.js';
import { AsyncDirective, directive, nothing, PartType, } from 'chrome://resources/mwc/lit/index.js';
import { assert, assertExists, assertInstanceof, } from '../../core/utils/assert.js';
class WithTooltip extends AsyncDirective {
    constructor(partInfo) {
        super(partInfo);
        this.tooltip = null;
        this.parentElement = document.body;
        assert(partInfo.type === PartType.ELEMENT, 'The `withTooltip` directive must be used in element tag');
    }
    createTooltip(element) {
        const tooltip = new Tooltip();
        tooltip.ariaHidden = 'true';
        tooltip.anchorElement = element;
        if (element.parentElement !== null) {
            this.parentElement = element.parentElement;
        }
        return tooltip;
    }
    update(part, [label]) {
        if (!this.isConnected) {
            return this.render(label);
        }
        const element = assertInstanceof(part.element, HTMLElement);
        if (this.tooltip === null) {
            this.tooltip = this.createTooltip(element);
            this.parentElement.appendChild(this.tooltip);
        }
        if (label === undefined) {
            this.tooltip.label = assertExists(element.ariaLabel, '`aria-label` must be set when using `withTooltip()` without argument');
        }
        else {
            this.tooltip.label = label;
        }
        return this.render(label);
    }
    disconnected() {
        if (this.tooltip !== null) {
            this.parentElement.removeChild(this.tooltip);
        }
    }
    reconnected() {
        if (this.tooltip !== null) {
            this.parentElement.appendChild(this.tooltip);
        }
    }
    render(_ = '') {
        return nothing;
    }
}
/**
 * The directive sets up tooltip when the element is focused or hovered. If not
 * specified, the tooltip string is set to the aria-label of the element.
 */
export const withTooltip = directive(WithTooltip);
