// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { useRecordingDataManager } from './lit/context.js';
import { ScopedAsyncEffect, ScopedEffect } from './reactive/lit.js';
import { computed, signal } from './reactive/signal.js';
import { AnimationFrameController } from './utils/animation_frame_controller.js';
import { assertInstanceof } from './utils/assert.js';
class ReactiveAudioImpl {
    get playing() {
        return this.playingImpl;
    }
    constructor() {
        this.latestAudioSrc = null;
        this.currentTimeImpl = signal(0);
        this.playingImpl = signal(false);
        this.mutedImpl = signal(false);
        this.volumeImpl = signal(1);
        this.audio = new Audio();
        this.playbackSpeedImpl = signal(1);
        this.recordingId = null;
        this.currentTime = computed({
            get: () => {
                return this.currentTimeImpl.value;
            },
            set: (currentTime) => {
                this.audio.currentTime = currentTime;
            },
        });
        this.playbackSpeed = computed({
            get: () => {
                return this.playbackSpeedImpl.value;
            },
            set: (speed) => {
                // The speed can be changed while playback is paused, and the ratechange
                // event won't be immediately fired in this case. Since we want the
                // speed change to be immediately effective on UI and don't expect it to
                // fail for the list of speeds on the UI, we optimistic update the
                // `playbackSpeedImpl` here and log an error on the ratechange callback
                // if the actual speed is different from the set value.
                this.playbackSpeedImpl.value = speed;
                this.audio.playbackRate = speed;
            },
        });
        this.muted = computed({
            get: () => {
                return this.mutedImpl.value;
            },
            set: (muted) => {
                this.audio.muted = muted;
            },
        });
        this.volume = computed({
            get: () => {
                return this.volumeImpl.value;
            },
            set: (volume) => {
                this.audio.volume = volume;
            },
        });
        this.audio.addEventListener('ratechange', () => {
            if (this.audio.playbackRate !== this.playbackSpeedImpl.value) {
                // TODO(pihsun): Integrate with error reporting.
                // TODO(pihsun): Check if this will be fired on pause.
                console.warn('Audio playback speed mismatch', this.audio.playbackRate, this.playbackSpeedImpl.value);
            }
        });
        this.audio.addEventListener('volumechange', () => {
            this.volumeImpl.value = this.audio.volume;
            this.mutedImpl.value = this.audio.muted;
        });
        this.audio.addEventListener('timeupdate', () => {
            this.currentTimeImpl.value = this.audio.currentTime;
        });
        // While audio is playing, we also use AnimationFrameController in addition
        // of the timeupdate event, since timeupdate fires infrequently and doesn't
        // look smooth while playing.
        this.animationFrameController = new AnimationFrameController(() => {
            this.currentTimeImpl.value = this.audio.currentTime;
        });
        // Only run the animationFrameController when the audio is playing, to save
        // CPU cycle.
        this.audio.addEventListener('play', () => {
            this.playingImpl.value = true;
            this.animationFrameController.start();
        });
        this.audio.addEventListener('pause', () => {
            this.playingImpl.value = false;
            this.animationFrameController.stop();
        });
    }
    revokeAudio() {
        if (this.latestAudioSrc !== null) {
            URL.revokeObjectURL(this.latestAudioSrc);
            this.latestAudioSrc = null;
        }
    }
    revoke() {
        this.audio.pause();
        this.revokeAudio();
        this.audio.src = '';
        this.animationFrameController.stop();
        this.recordingId = null;
    }
    play() {
        return this.audio.play();
    }
    togglePlaying() {
        if (this.audio.paused) {
            // TODO(pihsun): This is async, should we await it?
            void this.audio.play();
        }
        else {
            this.audio.pause();
        }
    }
    loadAudioSrc(recordingId, src) {
        this.recordingId = recordingId;
        this.revokeAudio();
        this.latestAudioSrc = src;
        this.audio.src = this.latestAudioSrc;
        this.audio.load();
    }
}
export class AudioPlayerController {
    constructor(host, recordingId, autoPlay = false) {
        this.recordingId = recordingId;
        this.recordingDataManager = useRecordingDataManager();
        this.audio = new ReactiveAudioImpl();
        host.addController(this);
        this.loadAudioData = new ScopedAsyncEffect(host, async (signal) => {
            const id = this.recordingId.value;
            if (id === null) {
                this.audio.revoke();
                return;
            }
            const data = await this.recordingDataManager.getAudioFile(id);
            signal.throwIfAborted();
            if (id === this.audio.recordingId) {
                // The inner audio is from the same recording, likely set by
                // setInnerAudio. Don't load the audio src and use the inner audio
                // state in this case.
                return;
            }
            this.audio.loadAudioSrc(id, URL.createObjectURL(data));
            if (autoPlay) {
                await this.audio.play();
            }
        });
        this.setMediaSessionTitle = new ScopedEffect(host, () => {
            const id = this.recordingId.value;
            if (id === null) {
                return;
            }
            const metadata = this.recordingDataManager.getMetadata(id).value;
            if (metadata === null) {
                return;
            }
            navigator.mediaSession.metadata = new MediaMetadata({
                title: metadata.title,
            });
        });
    }
    hostDisconnected() {
        this.audio.revoke();
    }
    get currentTime() {
        return this.audio.currentTime;
    }
    get playbackSpeed() {
        return this.audio.playbackSpeed;
    }
    get muted() {
        return this.audio.muted;
    }
    get volume() {
        return this.audio.volume;
    }
    get playing() {
        return this.audio.playing;
    }
    togglePlaying() {
        this.audio.togglePlaying();
    }
    // Note that caller needs to make sure that the `.revoke()` on the returning
    // ReactiveAudio is eventually called, otherwise the AnimationFrameController
    // would leak.
    takeInnerAudio() {
        const audio = this.audio;
        // TODO(pihsun): Creating a new inner object to prevent revoking on
        // hostDisconnected is a bit wasteful.
        this.audio = new ReactiveAudioImpl();
        return audio;
    }
    setInnerAudio(audio) {
        const impl = assertInstanceof(audio, ReactiveAudioImpl);
        this.audio.revoke();
        this.audio = impl;
    }
}
