// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * A data directory to read and write recording files.
 */
export class DataDir {
    constructor(root) {
        this.root = root;
    }
    async write(name, data) {
        const handle = await this.root.getFileHandle(name, { create: true });
        const writable = await handle.createWritable();
        await writable.write(data);
        await writable.close();
    }
    async read(name) {
        const handle = await this.root.getFileHandle(name);
        const file = await handle.getFile();
        return file;
    }
    async list() {
        const names = [];
        for await (const name of this.root.keys()) {
            names.push(name);
        }
        return names;
    }
    async clear() {
        const names = await this.list();
        await Promise.all(names.map(async (name) => {
            await this.root.removeEntry(name);
        }));
    }
    async remove(name) {
        await this.root.removeEntry(name);
    }
    // TODO(shik): Support creating from other directories.
    static async createFromOpfs() {
        const root = await navigator.storage.getDirectory();
        return new DataDir(root);
    }
}
