// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { signal } from './reactive/signal.js';
import { assert, assertExists } from './utils/assert.js';
import { AsyncJobQueue } from './utils/async_job_queue.js';
const DEFAULT_MIC_ID = 'default';
async function listAllMicrophones(infoCallback) {
    const allDevices = await navigator.mediaDevices.enumerateDevices();
    // Use only audioinput, and remove the device with the deviceId "default"
    // since it's a duplicated entry and can't be used to get the internal info.
    const devices = allDevices.filter((d) => d.kind === 'audioinput' && d.deviceId !== DEFAULT_MIC_ID);
    // Retrieve the internal info from mojo.
    const devicesWithInfo = await Promise.all(devices.map(async ({ deviceId, label }) => {
        const internalMicInfo = await infoCallback(deviceId);
        return { ...internalMicInfo, deviceId, label };
    }));
    // Microphones sorting order: Default mic, Internal mics, then by label.
    const sortedDevices = devicesWithInfo.sort((a, b) => {
        if (a.isDefault !== b.isDefault) {
            return a.isDefault ? -1 : 1;
        }
        else if (a.isInternal !== b.isInternal) {
            return a.isInternal ? -1 : 1;
        }
        return a.label.localeCompare(b.label);
    });
    return sortedDevices;
}
/**
 * Returns whether the given `micId` is a valid microphone deviceId.
 *
 * @param microphones List of all connected microphones.
 * @param micId DeviceId of the microphone.
 * @return Whether the given `micId` is a valid microphone deviceId.
 */
function isValidMicId(microphones, micId) {
    return microphones.some((device) => device.deviceId === micId);
}
/**
 * Return the deviceId of the selected microphone.
 *
 * Returns `currentMicId` if the latest selected microphone is still connected.
 * Otherwise, returns `deviceId` of the default microphone or null if no
 * microphone is connected.
 *
 * @param microphoneList List of all connected microphones.
 * @param currentMicId DeviceId of the current selected microphone.
 * @return DeviceId of the selected microphone.
 */
function getSelectedMicId(microphoneList, currentMicId) {
    if (isValidMicId(microphoneList, currentMicId)) {
        return currentMicId;
    }
    if (microphoneList.length === 0) {
        console.error('There are no connected microphones.');
        return null;
    }
    // In case the microphone is unplugged, fall back to the default device.
    return assertExists(microphoneList[0]).deviceId;
}
export class MicrophoneManager {
    static async create(infoCallback) {
        const microphoneList = await listAllMicrophones(infoCallback);
        return new MicrophoneManager(microphoneList, infoCallback);
    }
    constructor(microphoneList, infoCallback) {
        this.infoCallback = infoCallback;
        this.cachedMicrophoneList = signal([]);
        this.selectedMicId = signal(null);
        this.updateMicListQueue = new AsyncJobQueue('keepLatest');
        this.cachedMicrophoneList.value = microphoneList;
        this.selectedMicId.value = getSelectedMicId(microphoneList, null);
        navigator.mediaDevices.addEventListener('devicechange', () => {
            this.updateMicListQueue.push(() => this.updateActiveMicrophones());
        });
    }
    getMicrophoneList() {
        return this.cachedMicrophoneList;
    }
    getSelectedMicId() {
        return this.selectedMicId;
    }
    setSelectedMicId(micId) {
        assert(isValidMicId(this.cachedMicrophoneList.value, micId), `Invalid microphone deviceId: ${micId}`);
        this.selectedMicId.value = micId;
    }
    async updateActiveMicrophones() {
        const microphones = await listAllMicrophones(this.infoCallback);
        this.cachedMicrophoneList.value = microphones;
        this.selectedMicId.value = getSelectedMicId(microphones, this.selectedMicId.value);
    }
}
