// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { assertExists } from '../utils/assert.js';
/**
 * Smaller number indicates that the state will appear earlier in the UI states.
 */
const uiOrderMap = {
    unavailable: 0,
    notInstalled: 1,
    installing: 2,
    needsReboot: 3,
    error: 4,
    installed: 5,
};
/**
 * Maps model state to the UI order.
 */
export function getModelUiOrder(state) {
    return assertExists(uiOrderMap[state.kind]);
}
/**
 * Possible error types of model execution.
 */
export var ModelExecutionError;
(function (ModelExecutionError) {
    // General error.
    ModelExecutionError["GENERAL"] = "GENERAL";
    // The transcription language is not supported.
    ModelExecutionError["UNSUPPORTED_LANGUAGE"] = "UNSUPPORTED_LANGUAGE";
    // The transcription word count is less than the minimum length.
    ModelExecutionError["UNSUPPORTED_TRANSCRIPTION_IS_TOO_SHORT"] = "UNSUPPORTED_TRANSCRIPTION_IS_TOO_SHORT";
    // The transcription word count is more than the maximum length.
    ModelExecutionError["UNSUPPORTED_TRANSCRIPTION_IS_TOO_LONG"] = "UNSUPPORTED_TRANSCRIPTION_IS_TOO_LONG";
    // Filtered by T&S on the request or response string.
    ModelExecutionError["UNSAFE"] = "UNSAFE";
})(ModelExecutionError || (ModelExecutionError = {}));
/**
 * Possible error types of model load.
 */
export var ModelLoadError;
(function (ModelLoadError) {
    // Unspecified load error.
    ModelLoadError["LOAD_FAILURE"] = "LOAD_FAILURE";
    // Model requires OS reboot.
    ModelLoadError["NEEDS_REBOOT"] = "NEEDS_REBOOT";
})(ModelLoadError || (ModelLoadError = {}));
export var GenaiResultType;
(function (GenaiResultType) {
    GenaiResultType["SUMMARY"] = "SUMMARY";
    GenaiResultType["TITLE_SUGGESTION"] = "TITLE_SUGGESTION";
})(GenaiResultType || (GenaiResultType = {}));
export class ModelLoader {
    /**
     * Requests download of the given model.
     */
    download() {
        // TODO(pihsun): There's currently no way of requesting download of the
        // model but not load it, so we load the model (which downloads the model)
        // and then immediately unloads it. Check the performance overhead and
        // consider adding another API for only downloading the model if the
        // overhead is large.
        void this.load().then((result) => {
            if (result.kind === 'success') {
                result.model.close();
            }
        });
    }
}
