// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
export class PerfLogger {
    constructor(eventsSender) {
        this.eventsSender = eventsSender;
        this.perfEventMap = new Map();
    }
    start(event) {
        // TODO: b/327538356 - Use assert when verified that there is no error.
        if (this.perfEventMap.has(event.kind)) {
            console.error(`Perf event ${event.kind} already exists.`);
        }
        this.perfEventMap.set(event.kind, {
            event,
            startTime: performance.now(),
        });
    }
    /**
     * Finishes the event if it has started.
     *
     * This avoids `finish` error when we only record `eventType` initiated from
     * particular sources, e.g. UI.
     */
    tryFinish(eventType) {
        if (!this.perfEventMap.has(eventType)) {
            return;
        }
        this.finish(eventType);
    }
    /**
     * Terminates and sends the perf event.
     */
    finish(eventType) {
        const eventValue = this.perfEventMap.get(eventType);
        if (eventValue === undefined) {
            console.error(`Perf event ${eventType} has not started logging.`);
            return;
        }
        const { event, startTime } = eventValue;
        const duration = Math.round(performance.now() - startTime);
        this.eventsSender.sendPerfEvent(event, duration);
        this.perfEventMap.delete(eventType);
    }
}
