// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { createTranscriptionModelDownloadPerf, } from './events_sender.js';
import { getModelUiOrder, } from './on_device_model/types.js';
import { effect } from './reactive/signal.js';
import { LanguageCode } from './soda/language_info.js';
import { settings } from './state/settings.js';
export class PlatformHandler {
    /**
     * Returns the formatted localized string by given `id` and `args`.
     *
     * This is the lower level function that is used to implement the `i18n`
     * helper in core/i18n.ts, and shouldn't be directly used.
     * The `i18n` helper provides better typing and should be used instead.
     *
     * This is declared as `static` so it can be directly use at module import
     * time, and all implementations should ensure that it can be called at
     * module import time.
     */
    static getStringF(_id, ..._args) {
        throw new Error('getStringF not implemented');
    }
    /**
     * Returns device type.
     *
     * This is the lower level function that is used to replace get device type
     * string in core/i18n.ts, and shouldn't be directly used.
     *
     * This is declared as `static` so it can be directly use at module import
     * time, and all implementations should ensure that it can be called at
     * module import time.
     */
    static getDeviceType() {
        throw new Error('getDeviceType not implemented');
    }
    /**
     * Gets integrated model state for title suggestion and summarization.
     *
     * Model state with smaller UI order will be returned. If both models are
     * downloading, then return state with smaller progress.
     */
    getGenAiModelState() {
        const summaryModelState = this.summaryModelLoader.state.value;
        const summaryUiOrder = getModelUiOrder(summaryModelState);
        const titleSuggestionModelState = this.titleSuggestionModelLoader.state.value;
        const titleSuggestionUiOrder = getModelUiOrder(titleSuggestionModelState);
        if (summaryModelState.kind === 'installing' &&
            titleSuggestionModelState.kind === 'installing') {
            if (summaryModelState.progress < titleSuggestionModelState.progress) {
                return summaryModelState;
            }
            return titleSuggestionModelState;
        }
        if (summaryUiOrder < titleSuggestionUiOrder) {
            return summaryModelState;
        }
        return titleSuggestionModelState;
    }
    isGenAiAvailable() {
        return this.getGenAiModelState().kind !== 'unavailable';
    }
    /**
     * Wrapper to download GenAI-related model.
     */
    downloadGenAiModel() {
        this.summaryModelLoader.download();
        this.titleSuggestionModelLoader.download();
    }
    /**
     * Returns the currently selected language.
     *
     * Returns null when there are multiple available languages, and no language
     * is selected.
     */
    getSelectedLanguage() {
        let selectedLanguage = settings.value.transcriptionLanguage;
        if (selectedLanguage !== null &&
            this.getSodaState(selectedLanguage).value.kind === 'unavailable') {
            // Unselect the language if the selected language pack is unavailable.
            selectedLanguage = null;
        }
        if (selectedLanguage === null && !this.isMultipleLanguageAvailable()) {
            // Use the default language (en-us) when there's no multiple language
            // pack available. Note that the language state may be unavailable.
            selectedLanguage = LanguageCode.EN_US;
        }
        return selectedLanguage;
    }
    /**
     * Returns information of the selected language.
     *
     * Returns null when no language is selected.
     */
    getSelectedLangPackInfo() {
        const selectedLanguage = this.getSelectedLanguage();
        return selectedLanguage === null ? null :
            this.getLangPackInfo(selectedLanguage);
    }
    /**
     * Returns the SODA installation state of the selected language.
     *
     * Returns null when there are multiple languages available, and no language
     * is selected.
     */
    getSelectedLanguageState() {
        const selectedLanguage = this.getSelectedLanguage();
        return selectedLanguage === null ? null :
            this.getSodaState(selectedLanguage);
    }
    /**
     * Returns the locale used to format various date/time string.
     *
     * The locale returned should be in the format that is compatible with the
     * locales argument for `Intl`.
     * (https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Intl#locales_argument).
     *
     * Returns undefined to use the system locale from Chrome.
     */
    getLocale() {
        return undefined;
    }
    /**
     * Adds model state watchers for perf events.
     */
    initPerfEventWatchers() {
        // Watcher for summarization model download.
        effect(() => {
            const state = this.summaryModelLoader.state.value;
            const summaryEventType = 'summaryModelDownload';
            if (state.kind === 'installed') {
                // Records perf event only if the download has been initiated from UI.
                this.perfLogger.tryFinish(summaryEventType);
            }
        });
        // Watchers for transcription model download.
        const languageList = this.getLangPackList();
        for (const language of languageList) {
            effect(() => {
                const state = this.getSodaState(language.languageCode).value;
                if (state.kind === 'installed') {
                    // Records perf event only if the download has been initiated from UI.
                    this.perfLogger.tryFinish(createTranscriptionModelDownloadPerf(language.languageCode).kind);
                }
            });
        }
    }
}
