// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Extended LitElement class to automatically rerender on signal
 * change.
 *
 * This is similar to SignalWatcher in lit-labs/preact-signal.
 */
import { LitElement, } from 'chrome://resources/mwc/lit/index.js';
import { batch, effect, signal, } from './signal.js';
export class ReactiveLitElement extends LitElement {
    constructor() {
        super(...arguments);
        this.disposeUpdateEffect = null;
        this.inPerformUpdate = false;
        this.updateTrigger = signal(false);
        // TODO(pihsun): Use a key to signal map if performance is an issue.
        this.propertySignalBindings = [];
    }
    performUpdate() {
        // We deviates with default lit semantic here and don't update while
        // disconnected. This ensures proper teardown of the update effect.
        if (!this.isUpdatePending || !this.isConnected) {
            return;
        }
        this.inPerformUpdate = true;
        if (this.disposeUpdateEffect !== null) {
            // Effect had already been set up, trigger it.
            // This lines triggers a rerun of the effect.
            this.updateTrigger.update((x) => !x);
            return;
        }
        this.disposeUpdateEffect = effect(() => {
            // Read the value of updateTrigger, so changing the value of
            // updateTrigger will triggers this effect.
            void this.updateTrigger.value;
            if (this.inPerformUpdate) {
                this.inPerformUpdate = false;
                super.performUpdate();
            }
            else {
                // This is being triggered by signal dependency changes in
                // super.performUpdate().
                this.requestUpdate();
            }
        });
    }
    connectedCallback() {
        super.connectedCallback();
        // Request render on connect, so we know the signals used in render.
        this.requestUpdate();
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.disposeUpdateEffect?.();
        this.disposeUpdateEffect = null;
    }
    registerPropertySignal(signal, prop) {
        this.propertySignalBindings.push({ signal, prop });
    }
    willUpdate(changedProperties) {
        batch(() => {
            for (const { signal, prop } of this.propertySignalBindings) {
                if (changedProperties.has(prop)) {
                    // TODO(pihsun): Do we need to check changedProperties since signal
                    // already does change check internally?
                    signal.value = Reflect.get(this, prop);
                }
            }
        });
    }
    propSignal(prop) {
        const sig = signal(Reflect.get(this, prop));
        this.registerPropertySignal(sig, prop);
        return sig;
    }
}
export var ComputedState;
(function (ComputedState) {
    ComputedState["DONE"] = "DONE";
    ComputedState["ERROR"] = "ERROR";
    ComputedState["RUNNING"] = "RUNNING";
    ComputedState["UNINITIALIZED"] = "UNINITIALIZED";
})(ComputedState || (ComputedState = {}));
export class ScopedAsyncComputed {
    // Note that only the signal values before the first async point of callback
    // is tracked.
    constructor(host, callback) {
        this.callback = callback;
        this.dispose = null;
        this.stateInternal = signal(ComputedState.UNINITIALIZED);
        // Note that this retains the latest successfully computed value even when
        // the value is being recomputing / the latest compute failed.
        // TODO(pihsun): Check if this behavior is expected.
        // TODO(pihsun): Save latest error.
        this.valueInternal = signal(null);
        this.forceRerunToggle = signal(false);
        host.addController(this);
    }
    rerun() {
        this.forceRerunToggle.update((x) => !x);
    }
    get state() {
        return this.stateInternal.value;
    }
    get value() {
        return this.valueInternal.value;
    }
    get valueSignal() {
        return this.valueInternal;
    }
    hostConnected() {
        let latestRun;
        let abortController = null;
        this.dispose = effect(() => {
            void this.forceRerunToggle.value;
            abortController?.abort();
            abortController = new AbortController();
            const thisRun = Symbol();
            latestRun = thisRun;
            this.stateInternal.value = ComputedState.RUNNING;
            this.callback(abortController.signal)
                .then((val) => {
                if (latestRun === thisRun) {
                    this.valueInternal.value = val;
                    this.stateInternal.value = ComputedState.DONE;
                }
            }, (e) => {
                if (latestRun === thisRun) {
                    console.error(e);
                    // TODO(pihsun): Save the latest error.
                    this.stateInternal.value = ComputedState.ERROR;
                }
            });
        });
    }
    hostDisconnected() {
        this.dispose?.();
        this.dispose = null;
        this.stateInternal.value = ComputedState.UNINITIALIZED;
        this.valueInternal.value = null;
    }
}
// This is for exporting the class alias.
// eslint-disable-next-line @typescript-eslint/naming-convention
export const ScopedAsyncEffect = (ScopedAsyncComputed);
export class ScopedEffect {
    constructor(host, callback) {
        this.callback = callback;
        this.dispose = null;
        host.addController(this);
    }
    hostConnected() {
        if (this.dispose === null) {
            this.dispose = effect(this.callback);
        }
    }
    hostDisconnected() {
        this.dispose?.();
        this.dispose = null;
    }
}
