// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// TODO(pihsun): Consider using the polyfill of
// https://github.com/tc39/proposal-signals or at least align our public API
// with the proposal.
import { assert } from '../../utils/assert.js';
import { IterableWeakSet } from '../../utils/iterable_weak_set.js';
import { forceCast } from '../../utils/type_utils.js';
import { Signal } from './types.js';
// All descendents of DIRTY should be DIRTY or TO_CHECK.
// All descendents of TO_CHECK should be TO_CHECK.
// Nothing should have a descendent of DISPOSED.
// TODO(pihsun): Really check what checks are needed for DISPOSED case.
var DirtyState;
(function (DirtyState) {
    DirtyState[DirtyState["CLEAN"] = 0] = "CLEAN";
    DirtyState[DirtyState["TO_CHECK"] = 1] = "TO_CHECK";
    DirtyState[DirtyState["DIRTY"] = 2] = "DIRTY";
    DirtyState[DirtyState["DISPOSED"] = 3] = "DISPOSED";
})(DirtyState || (DirtyState = {}));
let currentComputing = null;
export class SignalImpl extends Signal {
    constructor(valueInternal) {
        super();
        this.valueInternal = valueInternal;
        this.children = new IterableWeakSet();
    }
    get value() {
        if (currentComputing !== null) {
            this.children.add(currentComputing);
            currentComputing.addParent(this);
        }
        return this.valueInternal;
    }
    set value(newValue) {
        if (newValue !== this.valueInternal) {
            this.valueInternal = newValue;
            // This is a micro-optimization that we never mark Signal as "dirty", but
            // instead always directly push dirty a level down, since we need to mark
            // children as TO_CHECK recursively anyway.
            for (const child of this.children) {
                child.markDirtyRecursive();
            }
            Effect.processBatchedEffect();
        }
    }
    peek() {
        return this.valueInternal;
    }
    maybeUpdate() {
        // Since we already mark children dirty when setting different value, we
        // don't need to do anything here.
        return;
    }
    removeChild(child) {
        this.children.delete(child);
    }
    numChildrenForTesting() {
        return this.children.sizeForTesting();
    }
}
const uninitialized = Symbol('uninitialized');
// Note that this doesn't really always implements all interface of Signal when
// `set` is not given. This is enforced at type level in the exported
// `computed` function in lib.ts.
export class ComputedImpl extends Signal {
    // Note that the `set` function should actually change the "source" value
    // depend by `get`, so the next call to `get` will get the correct value.
    constructor(get, set) {
        super();
        this.get = get;
        this.set = set;
        // We always start computed in dirty state, so the first value is only
        // computed when client first call to .value. Since we check for value change
        // when re-evaluating, the initial value should be something distinct to
        // possible values of the computed variable, so we can't use null/undefined
        // here and use a unique symbol as initial value instead.
        this.valueInternal = forceCast(uninitialized);
        this.state = DirtyState.DIRTY;
        this.parents = new Set();
        this.children = new IterableWeakSet();
    }
    get value() {
        assert(this.state !== DirtyState.DISPOSED);
        if (currentComputing !== null) {
            this.children.add(currentComputing);
            currentComputing.addParent(this);
        }
        return this.peek();
    }
    set value(val) {
        assert(this.set !== undefined, 'value setter called on computed without set');
        this.set(val);
    }
    peek() {
        assert(this.state !== DirtyState.DISPOSED);
        this.maybeUpdate();
        return this.valueInternal;
    }
    markDirty() {
        this.state = DirtyState.DIRTY;
    }
    markDirtyRecursive() {
        if (this.state !== DirtyState.DIRTY) {
            this.state = DirtyState.DIRTY;
            for (const child of this.children) {
                child.markToCheckRecursive();
            }
        }
    }
    markToCheckRecursive() {
        if (this.state !== DirtyState.TO_CHECK && this.state !== DirtyState.DIRTY) {
            this.state = DirtyState.TO_CHECK;
            for (const child of this.children) {
                child.markToCheckRecursive();
            }
        }
    }
    addParent(parent) {
        this.parents.add(parent);
    }
    dispose() {
        for (const parent of this.parents) {
            parent.removeChild(this);
        }
        this.parents.clear();
        this.state = DirtyState.DISPOSED;
    }
    maybeUpdate() {
        if (this.state === DirtyState.CLEAN) {
            return;
        }
        if (this.state === DirtyState.TO_CHECK) {
            for (const parent of this.parents) {
                parent.maybeUpdate();
                // Typescript assumes that maybeUpdate won't change this.state, but it
                // can.
                /* eslint-disable-next-line
                   @typescript-eslint/consistent-type-assertions */
                if (this.state === DirtyState.DIRTY) {
                    // Someone already pass dirty state down, no need to check further.
                    break;
                }
            }
            if (this.state === DirtyState.TO_CHECK) {
                // All parents clean, yay!
                this.state = DirtyState.CLEAN;
                return;
            }
        }
        // Needs update. Since the parents might change we need to dispose here.
        this.dispose();
        const oldComputing = currentComputing;
        // eslint-disable-next-line @typescript-eslint/no-this-alias
        currentComputing = this;
        const newValue = this.get();
        currentComputing = oldComputing;
        if (newValue !== this.valueInternal) {
            // Value changed, mark child as dirty.
            this.valueInternal = newValue;
            for (const child of this.children) {
                // Since this node is originally in either DIRTY or TO_CHECK, all
                // descendents should already be in DIRTY or TO_CHECK, so we don't need
                // to mark recursively here.
                child.markDirty();
            }
        }
        this.state = DirtyState.CLEAN;
    }
    removeChild(child) {
        this.children.delete(child);
    }
    numChildrenForTesting() {
        return this.children.sizeForTesting();
    }
}
// Set of all effects to prevent effect getting garbage collected. Effect needs
// to be cancelled explicitly with .dispose() when not in used.
const allEffects = new Set();
export class Effect {
    static { this.batchedEffect = new Set(); }
    static { this.batchDepth = 0; }
    // TODO(pihsun): Have some test to ensure that there's no effect leaked.
    constructor(callback) {
        this.callback = callback;
        this.parents = new Set();
        this.state = DirtyState.CLEAN;
        this.dispose = () => {
            allEffects.delete(this);
            this.state = DirtyState.DISPOSED;
            this.disconnect();
        };
        // TODO(pihsun): Warning when allEffects is growing / have too many items?
        allEffects.add(this);
        this.execute();
    }
    markDirty() {
        if (this.state !== DirtyState.DIRTY) {
            this.state = DirtyState.DIRTY;
            Effect.batchedEffect.add(this);
        }
    }
    markDirtyRecursive() {
        this.markDirty();
    }
    markToCheckRecursive() {
        if (this.state !== DirtyState.DIRTY && this.state !== DirtyState.TO_CHECK) {
            this.state = DirtyState.TO_CHECK;
            Effect.batchedEffect.add(this);
        }
    }
    disconnect() {
        for (const parent of this.parents) {
            parent.removeChild(this);
        }
        this.parents.clear();
    }
    execute() {
        this.disconnect();
        const oldComputing = currentComputing;
        // eslint-disable-next-line @typescript-eslint/no-this-alias
        currentComputing = this;
        this.callback({ dispose: this.dispose });
        currentComputing = oldComputing;
        // The effect might have been disposed in it's callback.
        if (this.state !== DirtyState.DISPOSED) {
            this.state = DirtyState.CLEAN;
        }
    }
    maybeExecute() {
        if (this.state === DirtyState.TO_CHECK) {
            for (const parent of this.parents) {
                parent.maybeUpdate();
                // Typescript assumes that maybeUpdate won't change this.state, but it
                // can.
                /* eslint-disable-next-line
                   @typescript-eslint/consistent-type-assertions */
                if (this.state === DirtyState.DIRTY) {
                    // Someone already pass dirty state down, no need to check further.
                    break;
                }
            }
            if (this.state === DirtyState.TO_CHECK) {
                // All parents clean, yay!
                this.state = DirtyState.CLEAN;
                return;
            }
        }
        if (this.state === DirtyState.DIRTY) {
            this.execute();
        }
    }
    addParent(parent) {
        assert(this.state !== DirtyState.DISPOSED, 'addParent called after the effect had been disposed');
        this.parents.add(parent);
    }
    static processBatchedEffect() {
        if (Effect.batchDepth !== 0) {
            return;
        }
        let cnt = 0;
        while (Effect.batchedEffect.size > 0) {
            cnt++;
            if (cnt === 10000) {
                throw new Error('Effect recurse for more than 10000 times');
            }
            const effects = Effect.batchedEffect;
            Effect.batchedEffect = new Set();
            for (const effect of effects) {
                effect.maybeExecute();
            }
        }
    }
}
