// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview This file contains all public type interface of signal.
 *
 * Note that all types are re-exported by signal.ts, and user outside of the
 * signal implementation should import from that file instead.
 */
import { produce } from '../../utils/draft.js';
export class ReadonlySignal {
}
export class Signal extends ReadonlySignal {
    /**
     * Updates the signal value based on the old value.
     *
     * This is a shortcut of `this.value = updater(this.value);`.
     *
     * Note that the updater should returns a new value without updating the
     * value in place, since current implementation relies on object identity for
     * change detection.
     */
    update(updater) {
        this.value = updater(this.peek());
    }
    /**
     * "Mutates" the signal value.
     *
     * This is a shortcut of `this.value = produce(this.value, recipe);`.
     *
     * Note that the value isn't actually mutated in place but immutably updated
     * via draft.ts.
     */
    mutate(recipe) {
        this.value = produce(this.peek(), recipe);
    }
}
