// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { POWER_SCALE_FACTOR, SAMPLE_RATE, SAMPLES_PER_POWER_BAR, SAMPLES_PER_SLICE, } from './audio_constants.js';
import { computed, signal } from './reactive/signal.js';
import { SodaEventTransformer, Transcription } from './soda/soda.js';
import { assert, assertExists, } from './utils/assert.js';
import { AsyncJobQueue } from './utils/async_job_queue.js';
import { InteriorMutableArray } from './utils/interior_mutable_array.js';
import { clamp } from './utils/utils.js';
const AUDIO_MIME_TYPE = 'audio/webm;codecs=opus';
const TIME_SLICE_MS = 100;
// Logical group size for each aggregated power data point.
// Note that in implementation we only take one slice from each group to avoid
// calculation overhead.
const POWER_GROUP_SIZE = Math.round(SAMPLES_PER_POWER_BAR / SAMPLES_PER_SLICE);
// The number of samples to calculate one power slice.
const SAMPLE_SIZE = 16;
function getMicrophoneStream(micId, echoCancellation) {
    return navigator.mediaDevices.getUserMedia({
        audio: {
            autoGainControl: { exact: false },
            deviceId: { exact: micId },
            echoCancellation: { exact: echoCancellation },
        },
    });
}
let audioCtxGlobal = null;
async function getAudioContext() {
    if (audioCtxGlobal === null) {
        // Set null output device when recording.
        audioCtxGlobal = new AudioContext({
            sampleRate: SAMPLE_RATE,
            sinkId: { type: 'none' },
        });
        await audioCtxGlobal.audioWorklet.addModule('./static/audio_worklet.js');
    }
    return audioCtxGlobal;
}
/**
 * A recording session to retrieve audio input and produce an audio blob output.
 */
export class RecordingSession {
    constructor(audioCtx, config) {
        this.audioCtx = audioCtx;
        this.config = config;
        this.dataChunks = [];
        this.currentSodaSession = null;
        this.sodaEnableQueue = new AsyncJobQueue('keepLatest');
        this.powers = signal(new InteriorMutableArray([]));
        this.transcription = signal(null);
        this.processedSamples = 0;
        this.micAudioSourceNode = null;
        this.systemAudioSourceNode = null;
        this.micMuted = false;
        this.everPausedInternal = false;
        this.everMutedInternal = false;
        this.powerCounts = 0;
        this.progress = computed(() => {
            const powers = this.powers.value;
            const length = powers.length * SAMPLES_PER_POWER_BAR / SAMPLE_RATE;
            return {
                length,
                powers,
                transcription: this.transcription.value,
            };
        });
        this.sodaEventTransformer = new SodaEventTransformer(config.speakerLabelEnabled);
        this.combinedInputNode = audioCtx.createMediaStreamDestination();
        this.audioProcessor = new AudioWorkletNode(audioCtx, 'audio-processor');
        this.mediaRecorder = new MediaRecorder(this.combinedInputNode.stream, {
            mimeType: AUDIO_MIME_TYPE,
        });
        this.mediaRecorder.addEventListener('dataavailable', (e) => {
            this.onDataAvailable(e);
        });
        this.mediaRecorder.addEventListener('error', (e) => {
            this.onError(e);
        });
        this.audioProcessor.port.addEventListener('message', (ev) => {
            this.powerCounts = (this.powerCounts + 1) % POWER_GROUP_SIZE;
            const samples = ev.data;
            if (this.powerCounts === 0) {
                // Calculates the power of the slice. The value range is [0, 1].
                const squaredSum = samples
                    .slice(0, SAMPLE_SIZE)
                    .map((v) => v * v)
                    .reduce((x, y) => x + y, 0);
                const power = Math.sqrt(squaredSum / SAMPLE_SIZE);
                // Takes another `sqrt` to apply non-linear distortion, making small
                // gain easier to be seen.
                const scaledPower = clamp(Math.floor(Math.sqrt(power) * POWER_SCALE_FACTOR), 0, POWER_SCALE_FACTOR - 1);
                this.powers.value = this.powers.value.push(scaledPower);
            }
            this.currentSodaSession?.session.addAudio(samples);
            this.processedSamples += samples.length;
        });
    }
    /**
     * Sets the mute state of the mic stream.
     *
     * Note that this doesn't change the state of the system audio stream, as the
     * mute button is intended to only mute the mic stream.
     */
    setMicMuted(muted) {
        this.micMuted = muted;
        if (muted) {
            this.everMutedInternal = true;
        }
        if (this.micAudioSourceNode !== null) {
            for (const track of this.micAudioSourceNode.mediaStream.getTracks()) {
                track.enabled = !muted;
            }
        }
    }
    setSpeakerLabelEnabled(enabled) {
        this.sodaEventTransformer.speakerLabelEnabled = enabled;
    }
    get everPaused() {
        return this.everPausedInternal;
    }
    get everMuted() {
        return this.everMutedInternal;
    }
    connectSourceNode(node) {
        node.connect(this.combinedInputNode);
        node.connect(this.audioProcessor);
    }
    async initMicAudioSourceNode() {
        if (this.micAudioSourceNode !== null) {
            return;
        }
        // Turn on AEC when capturing system audio via getDisplayMedia.
        const micStream = await getMicrophoneStream(this.config.micId, this.config.canCaptureSystemAudioWithLoopback);
        this.micAudioSourceNode = this.audioCtx.createMediaStreamSource(micStream);
        this.connectSourceNode(this.micAudioSourceNode);
        // Set the mic muted setting again onto the new mic stream.
        this.setMicMuted(this.micMuted);
    }
    async initSystemAudioSourceNode() {
        if (this.systemAudioSourceNode !== null) {
            return;
        }
        if (!this.config.includeSystemAudio ||
            !this.config.canCaptureSystemAudioWithLoopback) {
            return;
        }
        const systemAudioStream = await this.config.platformHandler.getSystemAudioMediaStream();
        this.systemAudioSourceNode =
            this.audioCtx.createMediaStreamSource(systemAudioStream);
        this.connectSourceNode(this.systemAudioSourceNode);
    }
    closeAudioSourceNode(node) {
        for (const track of node.mediaStream.getTracks()) {
            track.stop();
        }
        node.disconnect();
    }
    closeMicAudioSourceNode() {
        if (this.micAudioSourceNode !== null) {
            this.closeAudioSourceNode(this.micAudioSourceNode);
            this.micAudioSourceNode = null;
        }
    }
    closeSystemAudioSourceNode() {
        if (this.systemAudioSourceNode !== null) {
            this.closeAudioSourceNode(this.systemAudioSourceNode);
            this.systemAudioSourceNode = null;
        }
    }
    async setPaused(paused) {
        if (paused) {
            await this.audioCtx.suspend();
            // We still need to explicitly pause the media recorder, otherwise the
            // exported webm will have wrong timestamps.
            this.mediaRecorder.pause();
            // Close the mic when paused, so the "mic in use" indicator would go away.
            this.closeMicAudioSourceNode();
            this.everPausedInternal = true;
        }
        else {
            await this.initMicAudioSourceNode();
            this.mediaRecorder.resume();
            await this.audioCtx.resume();
        }
    }
    onDataAvailable(event) {
        // TODO(shik): Save the data to file system while recording.
        this.dataChunks.push(event.data);
    }
    onError(event) {
        // TODO(shik): Proper error handling.
        console.error(event);
    }
    async isSodaInstalled(language) {
        const { platformHandler } = this.config;
        const sodaState = platformHandler.getSodaState(language);
        assert(sodaState.value.kind !== 'unavailable', `Trying to install SODA when it's unavailable`);
        // Because there's no `OnSodaUninstalled` event, `installed` state may be
        // outdated when other process removes the library. Wait for status update
        // to avoid state inconsistency.
        // TODO: b/375306309 - Remove "await" when soda states are always consistent
        // after the `OnSodaUninstalled` event is implemented.
        await platformHandler.installSoda(language);
        return sodaState.value.kind === 'installed';
    }
    tryStartSodaSession(language) {
        return this.sodaEnableQueue.push(async () => {
            if (!await this.isSodaInstalled(language)) {
                return;
            }
            if (this.currentSodaSession !== null) {
                return;
            }
            if (this.transcription.value === null) {
                this.transcription.value = new Transcription([], language);
            }
            // Abort current running job if there's a new enable/disable request.
            if (this.sodaEnableQueue.hasPendingJob()) {
                return;
            }
            const session = await this.config.platformHandler.newSodaSession(language);
            const unsubscribe = session.subscribeEvent((ev) => {
                this.sodaEventTransformer.addEvent(ev, assertExists(this.currentSodaSession).startOffsetMs);
                this.transcription.value =
                    this.sodaEventTransformer.getTranscription(language);
            });
            this.currentSodaSession = {
                language,
                session,
                unsubscribe,
                startOffsetMs: (this.processedSamples / SAMPLE_RATE) * 1000,
            };
            await session.start();
        });
    }
    stopSodaSession() {
        return this.sodaEnableQueue.push(async () => {
            if (this.currentSodaSession === null) {
                return;
            }
            await this.currentSodaSession.session.stop();
            this.currentSodaSession.unsubscribe();
            // TODO: b/369277555 - Investigate why SODA does not convert all results
            // to final.
            this.sodaEventTransformer.finalizeTokens();
            this.transcription.value = this.sodaEventTransformer.getTranscription(this.currentSodaSession.language);
            this.currentSodaSession = null;
        });
    }
    /**
     * Starts the recording session.
     *
     * Note that each recording session is intended to only be started once.
     */
    async start(transcriptionEnabled, language = null) {
        // Suspend the context while initializing the source nodes.
        await this.audioCtx.suspend();
        if (transcriptionEnabled && language !== null) {
            // Do not wait for session start to avoid SODA failure hangs recording.
            // TODO(hsuanling): Have the audio buffered and send to recognizer later?
            this.tryStartSodaSession(language);
        }
        await Promise.all([
            this.initMicAudioSourceNode(),
            this.initSystemAudioSourceNode(),
        ]);
        // Resume the context and start the recorder & audio processor after we've
        // initialized all sources.
        await this.audioCtx.resume();
        this.audioProcessor.port.start();
        this.mediaRecorder.start(TIME_SLICE_MS);
    }
    async finish() {
        const stopped = new Promise((resolve) => {
            this.mediaRecorder.addEventListener('stop', resolve, { once: true });
        });
        this.mediaRecorder.stop();
        this.audioProcessor.port.close();
        await this.stopSodaSession().result;
        await stopped;
        this.closeMicAudioSourceNode();
        this.closeSystemAudioSourceNode();
        return new Blob(this.dataChunks, { type: AUDIO_MIME_TYPE });
    }
    static async create(config) {
        const audioCtx = await getAudioContext();
        return new RecordingSession(audioCtx, config);
    }
}
