// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import './shimless_rma_shared.css.js';
import './base_page.js';
import 'chrome://resources/ash/common/cr_elements/cr_dialog/cr_dialog.js';
import 'chrome://resources/ash/common/cr_elements/cr_input/cr_input.js';
import { I18nMixin } from 'chrome://resources/ash/common/cr_elements/i18n_mixin.js';
import { assert } from 'chrome://resources/js/assert.js';
import { afterNextRender, PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { getShimlessRmaService } from './mojo_interface_provider.js';
import { getTemplate } from './onboarding_enter_rsu_wp_disable_code_page.html.js';
import { RmadErrorCode } from './shimless_rma.mojom-webui.js';
import { dispatchNextButtonClick, enableNextButton } from './shimless_rma_util.js';
// The number of characters in an RSU code.
const RSU_CODE_EXPECTED_LENGTH = 8;
/**
 * @fileoverview
 * 'onboarding-enter-rsu-wp-disable-code-page' asks the user for the RSU disable
 * code.
 */
const OnboardingEnterRsuWpDisableCodePageBase = I18nMixin(PolymerElement);
export class OnboardingEnterRsuWpDisableCodePage extends OnboardingEnterRsuWpDisableCodePageBase {
    constructor() {
        super(...arguments);
        this.shimlessRmaService = getShimlessRmaService();
    }
    static get is() {
        return 'onboarding-enter-rsu-wp-disable-code-page';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            /**
             * Set by shimless_rma.ts.
             */
            allButtonsDisabled: Boolean,
            /**
             * Set by shimless_rma.ts.
             */
            errorCode: {
                type: Object,
                observer: OnboardingEnterRsuWpDisableCodePage.prototype.onErrorCodeChanged,
            },
            canvasSize: {
                type: Number,
                value: 0,
            },
            rsuChallenge: {
                type: String,
                value: '',
            },
            rsuHwid: {
                type: String,
                value: '',
            },
            rsuCode: {
                type: String,
                value: '',
                observer: OnboardingEnterRsuWpDisableCodePage.prototype.onRsuCodeChanged,
            },
            rsuCodeExpectedLength: {
                type: Number,
                value: RSU_CODE_EXPECTED_LENGTH,
                readOnly: true,
            },
            rsuInstructionsText: {
                type: String,
                value: '',
            },
            qrCodeUrl: {
                type: String,
                value: '',
            },
            rsuChallengeLinkText: {
                type: String,
                value: '',
                computed: 'computeRsuChallengeLinkText(rsuHwid, rsuChallenge)',
            },
            rsuCodeValidationRegex: {
                type: String,
                value: '.{1,8}',
                readOnly: true,
            },
            rsuCodeInvalid: {
                type: Boolean,
                value: false,
                reflectToAttribute: true,
            },
        };
    }
    ready() {
        super.ready();
        this.getRsuChallengeAndHwid();
        this.setRsuInstructionsText();
        enableNextButton(this);
        afterNextRender(this, () => {
            const codeInput = this.shadowRoot.querySelector('#rsuCode');
            assert(codeInput);
            codeInput.focus();
        });
    }
    getRsuChallengeAndHwid() {
        this.shimlessRmaService.getRsuDisableWriteProtectChallenge().then((result) => this.rsuChallenge = result.challenge);
        this.shimlessRmaService.getRsuDisableWriteProtectHwid().then((result) => {
            this.rsuHwid = result.hwid;
        });
        this.shimlessRmaService.getRsuDisableWriteProtectChallengeQrCode().then(this.updateQrCode.bind(this));
    }
    updateQrCode(response) {
        const blob = new Blob([Uint8Array.from(response.qrCodeData)], { 'type': 'image/png' });
        this.qrCodeUrl = URL.createObjectURL(blob);
    }
    rsuCodeIsPlausible() {
        return !!this.rsuCode && this.rsuCode.length === RSU_CODE_EXPECTED_LENGTH;
    }
    onRsuCodeChanged() {
        // Set to false whenever the user changes the code to remove the red invalid
        // warning.
        this.rsuCodeInvalid = false;
        this.rsuCode = this.rsuCode.toUpperCase();
    }
    onKeyDown(event) {
        if (event.key === 'Enter') {
            dispatchNextButtonClick(this);
        }
    }
    onNextButtonClick() {
        if (this.rsuCode.length !== this.rsuCodeExpectedLength) {
            this.rsuCodeInvalid = true;
            return Promise.reject(new Error('No RSU code set'));
        }
        return this.shimlessRmaService.setRsuDisableWriteProtectCode(this.rsuCode);
    }
    setRsuInstructionsText() {
        this.rsuInstructionsText =
            this.i18nAdvanced('rsuCodeInstructionsText', { attrs: ['id'] });
        const linkElement = this.shadowRoot.querySelector('#rsuCodeDialogLink');
        assert(linkElement);
        linkElement.setAttribute('href', '#');
        linkElement.addEventListener('click', () => {
            if (this.allButtonsDisabled) {
                return;
            }
            const dialog = this.shadowRoot.querySelector('#rsuChallengeDialog');
            assert(dialog);
            dialog.showModal();
        });
    }
    computeRsuChallengeLinkText() {
        const unlockPageUrl = 'https://chromeos.google.com/partner/console/cr50reset?challenge=';
        return unlockPageUrl + this.rsuChallenge + '&hwid=' + this.rsuHwid;
    }
    closeDialog() {
        const dialog = this.shadowRoot.querySelector('#rsuChallengeDialog');
        assert(dialog);
        dialog.close();
    }
    onErrorCodeChanged() {
        if (this.errorCode === RmadErrorCode.kWriteProtectDisableRsuCodeInvalid) {
            this.rsuCodeInvalid = true;
        }
    }
    getRsuCodeLabelText() {
        return this.rsuCodeInvalid ? this.i18n('rsuCodeErrorLabelText') :
            this.i18n('rsuCodeLabelText');
    }
    getRsuAriaDescription() {
        return `${this.getRsuCodeLabelText()} ${this.i18n('rsuCodeInstructionsAriaText')}`;
    }
}
customElements.define(OnboardingEnterRsuWpDisableCodePage.is, OnboardingEnterRsuWpDisableCodePage);
