// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import './shimless_rma_shared.css.js';
import './base_page.js';
import './icons.html.js';
import 'chrome://resources/ash/common/cr_elements/icons.html.js';
import 'chrome://resources/polymer/v3_0/iron-icon/iron-icon.js';
import { CrContainerShadowMixin } from 'chrome://resources/ash/common/cr_elements/cr_container_shadow_mixin.js';
import { I18nMixin } from 'chrome://resources/ash/common/cr_elements/i18n_mixin.js';
import { loadTimeData } from 'chrome://resources/ash/common/load_time_data.m.js';
import { assert } from 'chrome://resources/js/assert.js';
import { afterNextRender, PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { getShimlessRmaService } from './mojo_interface_provider.js';
import { getTemplate } from './reimaging_device_information_page.html.js';
import { FeatureLevel } from './shimless_rma.mojom-webui.js';
import { disableNextButton, enableNextButton, focusPageTitle } from './shimless_rma_util.js';
/**
 * @fileoverview
 * 'reimaging-device-information-page' allows the user to update important
 * device information if necessary.
 */
const ReimagingDeviceInformationPageBase = I18nMixin(CrContainerShadowMixin(PolymerElement));
/**
 * Supported options for IsChassisBranded and HwComplianceVersion questions.
 */
export var BooleanOrDefaultOptions;
(function (BooleanOrDefaultOptions) {
    BooleanOrDefaultOptions["DEFAULT"] = "default";
    BooleanOrDefaultOptions["YES"] = "yes";
    BooleanOrDefaultOptions["NO"] = "no";
})(BooleanOrDefaultOptions || (BooleanOrDefaultOptions = {}));
export class ReimagingDeviceInformationPage extends ReimagingDeviceInformationPageBase {
    constructor() {
        super(...arguments);
        this.shimlessRmaService = getShimlessRmaService();
    }
    static get is() {
        return 'reimaging-device-information-page';
    }
    static get template() {
        return getTemplate();
    }
    static get observers() {
        return [
            'updateNextButtonDisabledState(serialNumber, skuIndex, regionIndex,' +
                ' customLabelIndex, isChassisBranded, hwComplianceVersion,' +
                ' featureLevel)',
        ];
    }
    static get properties() {
        return {
            /**
             * Set by shimless_rma.ts.
             */
            allButtonsDisabled: Boolean,
            disableResetSerialNumber: {
                type: Boolean,
                computed: 'getDisableResetSerialNumber(originalSerialNumber,' +
                    'serialNumber, allButtonsDisabled)',
            },
            disableResetRegion: {
                type: Boolean,
                computed: 'getDisableResetRegion(originalRegionIndex, regionIndex,' +
                    'allButtonsDisabled)',
            },
            disableResetSku: {
                type: Boolean,
                computed: 'getDisableResetSku(originalSkuIndex, skuIndex,' +
                    'allButtonsDisabled)',
            },
            disableResetCustomLabel: {
                type: Boolean,
                computed: 'getDisableResetCustomLabel(' +
                    'originalCustomLabelIndex, customLabelIndex, allButtonsDisabled)',
            },
            disableResetDramPartNumber: {
                type: Boolean,
                computed: 'getDisableResetDramPartNumber(' +
                    'originalDramPartNumber, dramPartNumber, allButtonsDisabled)',
            },
            disableModifySerialNumber: {
                type: Boolean,
                value: false,
            },
            disableModifyRegion: {
                type: Boolean,
                value: false,
            },
            disableModifySku: {
                type: Boolean,
                value: false,
            },
            disableModifyCustomLabel: {
                type: Boolean,
                value: false,
            },
            disableModifyDramPartNumber: {
                type: Boolean,
                value: false,
            },
            disableModifyFeatureLevel: {
                type: Boolean,
                value: false,
            },
            originalSerialNumber: {
                type: String,
                value: '',
            },
            serialNumber: {
                type: String,
                value: '',
            },
            regions: {
                type: Array,
                value: () => [],
            },
            originalRegionIndex: {
                type: Number,
                value: -1,
            },
            regionIndex: {
                type: Number,
                value: -1,
            },
            skus: {
                type: Array,
                value: () => [],
            },
            originalSkuIndex: {
                type: Number,
                value: -1,
            },
            skuIndex: {
                type: Number,
                value: -1,
            },
            customLabels: {
                type: Array,
                value: () => [],
            },
            originalCustomLabelIndex: {
                type: Number,
                value: 0,
            },
            customLabelIndex: {
                type: Number,
                value: 0,
            },
            originalDramPartNumber: {
                type: String,
                value: '',
            },
            dramPartNumber: {
                type: String,
                value: '',
            },
            featureLevel: {
                type: Number,
                value: FeatureLevel.kRmadFeatureLevelUnsupported,
            },
            /**
             * Used to refer to the enum values in the HTML file.
             */
            booleanOrDefaultOptions: {
                type: Object,
                value: BooleanOrDefaultOptions,
                readOnly: true,
            },
            isChassisBranded: {
                type: String,
                value: BooleanOrDefaultOptions.DEFAULT,
            },
            hwComplianceVersion: {
                type: String,
                value: BooleanOrDefaultOptions.DEFAULT,
            },
        };
    }
    ready() {
        super.ready();
        this.getOriginalSerialNumber();
        this.getOriginalRegionAndRegionList();
        this.getOriginalSkuAndSkuList();
        this.getOriginalCustomLabelAndCustomLabelList();
        this.getOriginalDramPartNumber();
        this.getOriginalFeatureLevel();
        this.updateInputFieldModifiabilities();
        focusPageTitle(this);
    }
    allInformationIsValid() {
        const complianceQuestionsHaveDefaultValues = this.isChassisBranded === BooleanOrDefaultOptions.DEFAULT ||
            this.hwComplianceVersion === BooleanOrDefaultOptions.DEFAULT;
        if (this.areComplianceQuestionsShown() &&
            complianceQuestionsHaveDefaultValues) {
            return false;
        }
        return (this.serialNumber !== '') && (this.skuIndex >= 0) &&
            (this.regionIndex >= 0) && (this.customLabelIndex >= 0);
    }
    updateNextButtonDisabledState() {
        const disabled = !this.allInformationIsValid();
        if (disabled) {
            disableNextButton(this);
        }
        else {
            enableNextButton(this);
        }
    }
    getOriginalSerialNumber() {
        this.shimlessRmaService.getOriginalSerialNumber().then((result) => {
            this.originalSerialNumber = result.serialNumber;
            this.serialNumber = this.originalSerialNumber;
        });
    }
    getOriginalRegionAndRegionList() {
        this.shimlessRmaService.getOriginalRegion()
            .then((result) => {
            this.originalRegionIndex = result.regionIndex;
            return this.shimlessRmaService.getRegionList();
        })
            .then((result) => {
            this.regions = result.regions;
            this.regionIndex = this.originalRegionIndex;
            // Need to wait for the select options to render before setting the
            // selected index.
            afterNextRender(this, () => {
                const regionSelect = this.shadowRoot.querySelector('#regionSelect');
                assert(regionSelect);
                regionSelect.selectedIndex = this.regionIndex;
            });
        });
    }
    getOriginalSkuAndSkuList() {
        this.shimlessRmaService.getOriginalSku()
            .then((result) => {
            this.originalSkuIndex = result.skuIndex;
            return this.shimlessRmaService.getSkuList();
        })
            .then((result) => {
            this.skus = result.skus;
            this.skuIndex = this.originalSkuIndex;
            return this.shimlessRmaService.getSkuDescriptionList();
        })
            .then((result) => {
            // The SKU description list can be empty.
            if (this.skus.length === result.skuDescriptions.length) {
                this.skus = this.skus.map((sku, index) => `${sku}: ${result.skuDescriptions[index]}`);
            }
            // Need to wait for the select options to render before setting the
            // selected index.
            afterNextRender(this, () => {
                const skuSelect = this.shadowRoot.querySelector('#skuSelect');
                assert(skuSelect);
                skuSelect.selectedIndex = this.skuIndex;
            });
        });
    }
    getOriginalCustomLabelAndCustomLabelList() {
        this.shimlessRmaService.getOriginalCustomLabel()
            .then((result) => {
            this.originalCustomLabelIndex = result.customLabelIndex;
            return this.shimlessRmaService.getCustomLabelList();
        })
            .then((result) => {
            this.customLabels = result.customLabels;
            const blankIndex = this.customLabels.indexOf('');
            if (blankIndex >= 0) {
                this.customLabels[blankIndex] =
                    this.i18n('confirmDeviceInfoEmptyCustomLabelLabel');
                if (this.originalCustomLabelIndex < 0) {
                    this.originalCustomLabelIndex = blankIndex;
                }
            }
            this.customLabelIndex = this.originalCustomLabelIndex;
            // Need to wait for the select options to render before setting the
            // selected index.
            afterNextRender(this, () => {
                const customLabelSelect = this.shadowRoot.querySelector('#customLabelSelect');
                assert(customLabelSelect);
                customLabelSelect.selectedIndex = this.customLabelIndex;
            });
        });
    }
    getOriginalDramPartNumber() {
        this.shimlessRmaService.getOriginalDramPartNumber().then((result) => {
            this.originalDramPartNumber = result.dramPartNumber;
            this.dramPartNumber = this.originalDramPartNumber;
        });
    }
    getOriginalFeatureLevel() {
        this.shimlessRmaService.getOriginalFeatureLevel().then((result) => {
            this.featureLevel = result.originalFeatureLevel;
        });
    }
    getDisableResetSerialNumber() {
        return this.originalSerialNumber === this.serialNumber ||
            this.allButtonsDisabled;
    }
    getDisableResetRegion() {
        return this.originalRegionIndex === this.regionIndex ||
            this.allButtonsDisabled;
    }
    getDisableResetSku() {
        return this.originalSkuIndex === this.skuIndex || this.allButtonsDisabled;
    }
    getDisableResetCustomLabel() {
        return this.originalCustomLabelIndex === this.customLabelIndex ||
            this.allButtonsDisabled;
    }
    getDisableResetDramPartNumber() {
        return this.originalDramPartNumber === this.dramPartNumber ||
            this.allButtonsDisabled;
    }
    onSelectedRegionChange(_e) {
        const regionSelect = this.shadowRoot.querySelector('#regionSelect');
        assert(regionSelect);
        this.regionIndex = regionSelect.selectedIndex;
    }
    onSelectedSkuChange(_e) {
        const skuSelect = this.shadowRoot.querySelector('#skuSelect');
        assert(skuSelect);
        this.skuIndex = skuSelect.selectedIndex;
    }
    onSelectedCustomLabelChange(_e) {
        const customLabelSelect = this.shadowRoot.querySelector('#customLabelSelect');
        assert(customLabelSelect);
        this.customLabelIndex = customLabelSelect.selectedIndex;
    }
    onResetSerialNumberButtonClicked(_e) {
        this.serialNumber = this.originalSerialNumber;
    }
    onResetRegionButtonClicked(_e) {
        const regionSelect = this.shadowRoot.querySelector('#regionSelect');
        assert(regionSelect);
        this.regionIndex = this.originalRegionIndex;
        regionSelect.selectedIndex = this.regionIndex;
    }
    onResetSkuButtonClicked(_e) {
        const skuSelect = this.shadowRoot.querySelector('#skuSelect');
        assert(skuSelect);
        this.skuIndex = this.originalSkuIndex;
        skuSelect.selectedIndex = this.skuIndex;
    }
    onResetCustomLabelButtonClicked(_e) {
        const customLabelSelect = this.shadowRoot.querySelector('#customLabelSelect');
        assert(customLabelSelect);
        this.customLabelIndex = this.originalCustomLabelIndex;
        customLabelSelect.selectedIndex = this.customLabelIndex;
    }
    onResetDramPartNumberButtonClicked(_e) {
        this.dramPartNumber = this.originalDramPartNumber;
    }
    onIsChassisBrandedChange(_e) {
        const isChassisBranded = this.shadowRoot.querySelector('#isChassisBranded');
        assert(isChassisBranded);
        this.isChassisBranded = isChassisBranded.value;
    }
    onDoesMeetRequirementsChange(_e) {
        const doesMeetRequirements = this.shadowRoot.querySelector('#doesMeetRequirements');
        assert(doesMeetRequirements);
        this.hwComplianceVersion = doesMeetRequirements.value;
    }
    onNextButtonClick() {
        if (!this.allInformationIsValid()) {
            return Promise.reject(new Error('Some required information is not set'));
        }
        else {
            let isChassisBranded = false;
            let hwComplianceVersion = 0;
            if (this.areComplianceQuestionsShown()) {
                // Convert isChassisBranded to boolean value for mojo.
                isChassisBranded =
                    this.isChassisBranded === BooleanOrDefaultOptions.YES;
                // Convert hwComplianceVersion_ to correct value for mojo.
                const HARDWARE_COMPLIANT = 1;
                const HARDWARE_NOT_COMPLIANT = 0;
                hwComplianceVersion =
                    this.hwComplianceVersion === BooleanOrDefaultOptions.YES ?
                        HARDWARE_COMPLIANT :
                        HARDWARE_NOT_COMPLIANT;
            }
            return this.shimlessRmaService.setDeviceInformation(this.serialNumber, this.regionIndex, this.skuIndex, this.customLabelIndex, this.dramPartNumber, isChassisBranded, hwComplianceVersion);
        }
    }
    shouldShowComplianceSection() {
        return this.featureLevel !== FeatureLevel.kRmadFeatureLevelUnsupported;
    }
    isComplianceStatusKnown() {
        return this.featureLevel !== FeatureLevel.kRmadFeatureLevelUnsupported &&
            this.featureLevel !== FeatureLevel.kRmadFeatureLevelUnknown;
    }
    areComplianceQuestionsShown() {
        return this.shouldShowComplianceSection() &&
            !this.isComplianceStatusKnown();
    }
    getComplianceStatusString() {
        const deviceIsCompliant = this.featureLevel >= FeatureLevel.kRmadFeatureLevel1;
        return deviceIsCompliant ? this.i18n('confirmDeviceInfoDeviceCompliant') :
            this.i18n('confirmDeviceInfoDeviceNotCompliant');
    }
    async updateInputFieldModifiabilities() {
        if (!loadTimeData.getBoolean('dynamicDeviceInfoInputsEnabled')) {
            return;
        }
        const result = await this.shimlessRmaService.getStateProperties();
        if (result?.statePropertyResult.property?.updateDeviceInfoStateProperty ===
            undefined) {
            return;
        }
        const properties = result.statePropertyResult.property.updateDeviceInfoStateProperty;
        this.disableModifySerialNumber = !properties.serialNumberModifiable;
        this.disableModifyRegion = !properties.regionModifiable;
        this.disableModifySku = !properties.skuModifiable;
        this.disableModifyCustomLabel = !properties.customLabelModifiable;
        this.disableModifyDramPartNumber = !properties.dramPartNumberModifiable;
        this.disableModifyFeatureLevel = !properties.featureLevelModifiable;
    }
    isInputDisabled(inputDisabled) {
        return inputDisabled || this.allButtonsDisabled;
    }
}
customElements.define(ReimagingDeviceInformationPage.is, ReimagingDeviceInformationPage);
