// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import './accelerator_view.js';
import './text_accelerator.js';
import '/strings.m.js';
import '../css/shortcut_customization_shared.css.js';
import 'chrome://resources/ash/common/cr_elements/cr_input/cr_input.js';
import 'chrome://resources/ash/common/cr_elements/cr_shared_style.css.js';
import 'chrome://resources/polymer/v3_0/iron-icon/iron-icon.js';
import 'chrome://resources/polymer/v3_0/paper-tooltip/paper-tooltip.js';
import { I18nMixin } from 'chrome://resources/ash/common/cr_elements/i18n_mixin.js';
import { strictQuery } from 'chrome://resources/ash/common/typescript_utils/strict_query.js';
import { PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { AcceleratorLookupManager } from './accelerator_lookup_manager.js';
import { getTemplate } from './accelerator_row.html.js';
import { getShortcutProvider } from './mojo_interface_provider.js';
import { LayoutStyle } from './shortcut_types.js';
import { getAriaLabelForStandardAccelerators, getAriaLabelForTextAccelerators, getTextAcceleratorParts, isCustomizationAllowed } from './shortcut_utils.js';
/**
 * @fileoverview
 * 'accelerator-row' is a wrapper component for one shortcut. It features a
 * description of the shortcut along with a list of accelerators.
 */
const AcceleratorRowElementBase = I18nMixin(PolymerElement);
export class AcceleratorRowElement extends AcceleratorRowElementBase {
    constructor() {
        super(...arguments);
        this.lookupManager = AcceleratorLookupManager.getInstance();
        this.shortcutInterfaceProvider = getShortcutProvider();
    }
    static get is() {
        return 'accelerator-row';
    }
    static get properties() {
        return {
            description: {
                type: String,
                value: '',
            },
            acceleratorInfos: {
                type: Array,
                value: () => [],
            },
            layoutStyle: {
                type: Object,
            },
            isLocked: {
                type: Boolean,
                value: false,
            },
            action: {
                type: Number,
                value: 0,
                reflectToAttribute: true,
            },
            source: {
                type: Number,
                value: 0,
                observer: AcceleratorRowElement.prototype.onSourceChanged,
            },
            isEllipsisActive_: {
                type: Boolean,
                value: false,
            },
        };
    }
    connectedCallback() {
        super.connectedCallback();
        this.subcategoryIsLocked = this.lookupManager.isSubcategoryLocked(this.lookupManager.getAcceleratorSubcategory(this.source, this.action));
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        if (!this.isLocked) {
            this.removeEventListener('edit-icon-clicked', () => this.showDialog());
        }
    }
    onSourceChanged() {
        this.shortcutInterfaceProvider.isMutable(this.source)
            .then(({ isMutable }) => {
            this.isLocked = !isMutable;
            if (!this.isLocked) {
                this.addEventListener('edit-icon-clicked', () => this.showDialog());
            }
        });
    }
    isDefaultLayout() {
        return this.layoutStyle === LayoutStyle.kDefault;
    }
    isTextLayout() {
        return this.layoutStyle === LayoutStyle.kText;
    }
    showDialog() {
        if (!isCustomizationAllowed() || this.isTextLayout()) {
            return;
        }
        this.dispatchEvent(new CustomEvent('show-edit-dialog', {
            bubbles: true,
            composed: true,
            detail: {
                description: this.description,
                accelerators: this.acceleratorInfos,
                action: this.action,
                source: this.source,
            },
        }));
    }
    getTextAcceleratorParts(infos) {
        return getTextAcceleratorParts(infos);
    }
    isEmptyList(infos) {
        return infos.length === 0;
    }
    // Returns true if it is the first accelerator in the list.
    isFirstAccelerator(index) {
        return index === 0;
    }
    onEditIconClicked() {
        this.dispatchEvent(new CustomEvent('edit-icon-clicked', { bubbles: true, composed: true }));
    }
    onFocusOrMouseEnter() {
        if (this.lookupManager.getSearchResultRowFocused()) {
            return;
        }
        strictQuery('#container', this.shadowRoot, HTMLTableRowElement).focus();
    }
    onBlur() {
        this.lookupManager.setSearchResultRowFocused(false);
    }
    rowIsLocked() {
        // Accelerator row is locked if the subcategory or the source is locked or
        // it is text accelerator or if all accelerator infos are locked.
        return this.subcategoryIsLocked || this.isLocked || this.isTextLayout() ||
            (this.acceleratorInfos.length > 0 &&
                this.acceleratorInfos.every(info => info.locked));
    }
    getAcceleratorText() {
        // No shortcut assigned case:
        if (this.acceleratorInfos.length === 0) {
            return this.i18n('noShortcutAssigned');
        }
        return this.isDefaultLayout() ?
            getAriaLabelForStandardAccelerators(this.acceleratorInfos, this.i18n('acceleratorTextDivider')) :
            getAriaLabelForTextAccelerators(this.acceleratorInfos);
    }
    getAriaLabel() {
        if (!isCustomizationAllowed()) {
            return this.i18n('acceleratorRowAriaLabelReadOnly', this.description, this.getAcceleratorText());
        }
        else {
            const rowStatus = this.rowIsLocked() ? this.i18n('locked') : this.i18n('editable');
            return this.i18n('acceleratorRowAriaLabel', this.description, this.getAcceleratorText(), rowStatus);
        }
    }
    getEditButtonAriaLabel() {
        return this.i18n('editButtonForRow', this.description);
    }
    onMouseEnterDescriptionText_() {
        const descriptionText = this.$.descriptionText;
        const container = this.$.container;
        this.isEllipsisActive_ =
            container.clientHeight < descriptionText.scrollHeight;
    }
    static get template() {
        return getTemplate();
    }
}
customElements.define(AcceleratorRowElement.is, AcceleratorRowElement);
