// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview
 * 'bruschetta-subpage' is the settings subpage for managing Bruschetta
 * (third-party VMs).
 */
import 'chrome://resources/ash/common/cr_elements/cr_link_row/cr_link_row.js';
import '../settings_shared.css.js';
import '../guest_os/guest_os_confirmation_dialog.js';
import { PrefsMixin } from '/shared/settings/prefs/prefs_mixin.js';
import { PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { castExists } from '../assert_extras.js';
import { DeepLinkingMixin } from '../common/deep_linking_mixin.js';
import { RouteOriginMixin } from '../common/route_origin_mixin.js';
import { Setting } from '../mojom-webui/setting.mojom-webui.js';
import { Router, routes } from '../router.js';
import { getTemplate } from './bruschetta_subpage.html.js';
import { CrostiniBrowserProxyImpl } from './crostini_browser_proxy.js';
const BruschettaSubpageElementBase = DeepLinkingMixin(RouteOriginMixin(PrefsMixin(PolymerElement)));
export class BruschettaSubpageElement extends BruschettaSubpageElementBase {
    static get is() {
        return 'settings-bruschetta-subpage';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            showBruschettaMicPermissionDialog_: {
                type: Boolean,
                value: false,
            },
        };
    }
    static get observers() {
        return [
            'onInstalledChanged_(prefs.bruschetta.installed.value)',
        ];
    }
    constructor() {
        super();
        // DeepLinkingMixin override
        this.supportedSettingIds = new Set([
            Setting.kBruschettaMicAccess,
        ]);
        /** RouteOriginMixin override */
        this.route = routes.BRUSCHETTA_DETAILS;
        // For now we reuse the Crostini browser proxy, we're both part of
        // crostini_page. At some point we may want to split them apart (or make
        // something for all guest OSs).
        this.browserProxy_ = CrostiniBrowserProxyImpl.getInstance();
    }
    ready() {
        super.ready();
        this.addFocusConfig(routes.BRUSCHETTA_SHARED_USB_DEVICES, '#bruschettaSharedUsbDevicesRow');
        this.addFocusConfig(routes.BRUSCHETTA_SHARED_PATHS, '#bruschettaSharedPathsRow');
    }
    onSharedUsbDevicesClick_() {
        Router.getInstance().navigateTo(routes.BRUSCHETTA_SHARED_USB_DEVICES);
    }
    onSharedPathsClick_() {
        Router.getInstance().navigateTo(routes.BRUSCHETTA_SHARED_PATHS);
    }
    onRemoveClick_() {
        this.browserProxy_.requestBruschettaUninstallerView();
    }
    onInstalledChanged_(installed) {
        if (!installed &&
            Router.getInstance().currentRoute === routes.BRUSCHETTA_DETAILS) {
            Router.getInstance().navigateToPreviousRoute();
        }
    }
    getMicToggle_() {
        return castExists(this.shadowRoot.querySelector('#bruschetta-mic-permission-toggle'));
    }
    /**
     * If a change to the mic settings requires Bruschetta to be restarted, a
     * dialog is shown.
     */
    async onMicPermissionChange_() {
        if (await this.browserProxy_.checkBruschettaIsRunning()) {
            this.showBruschettaMicPermissionDialog_ = true;
        }
        else {
            this.getMicToggle_().sendPrefChange();
        }
    }
    onBruschettaMicPermissionDialogClose_(e) {
        const toggle = this.getMicToggle_();
        if (e.detail.accepted) {
            toggle.sendPrefChange();
            this.browserProxy_.shutdownBruschetta();
        }
        else {
            toggle.resetToPrefValue();
        }
        this.showBruschettaMicPermissionDialog_ = false;
    }
}
customElements.define(BruschettaSubpageElement.is, BruschettaSubpageElement);
