// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview
 * 'audio-settings' allow users to configure their audio settings in system
 * settings.
 */
import '../settings_shared.css.js';
import '//resources/ash/common/cr_elements/cr_shared_style.css.js';
import 'chrome://resources/ash/common/cr_elements/cr_icon_button/cr_icon_button.js';
import 'chrome://resources/ash/common/cr_elements/cr_slider/cr_slider.js';
import 'chrome://resources/ash/common/cr_elements/localized_link/localized_link.js';
import 'chrome://resources/ash/common/cr_elements/policy/cr_policy_indicator.js';
import { PrefsMixin } from '/shared/settings/prefs/prefs_mixin.js';
import { CrToggleElement } from 'chrome://resources/ash/common/cr_elements/cr_toggle/cr_toggle.js';
import { I18nMixin } from 'chrome://resources/ash/common/cr_elements/i18n_mixin.js';
import { WebUiListenerMixin } from 'chrome://resources/ash/common/cr_elements/web_ui_listener_mixin.js';
import { strictQuery } from 'chrome://resources/ash/common/typescript_utils/strict_query.js';
import { assert } from 'chrome://resources/js/assert.js';
import { loadTimeData } from 'chrome://resources/js/load_time_data.js';
import { PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { DeepLinkingMixin } from '../common/deep_linking_mixin.js';
import { RouteObserverMixin } from '../common/route_observer_mixin.js';
import { SettingsRadioGroupElement } from '../controls/settings_radio_group.js';
import { AudioDeviceType, AudioEffectState, AudioEffectType, AudioSystemPropertiesObserverReceiver, MuteState } from '../mojom-webui/cros_audio_config.mojom-webui.js';
import { Setting } from '../mojom-webui/setting.mojom-webui.js';
import { AudioAndCaptionsPageBrowserProxyImpl } from '../os_a11y_page/audio_and_captions_page_browser_proxy.js';
import { routes } from '../router.js';
import { getTemplate } from './audio.html.js';
import { getCrosAudioConfig } from './cros_audio_config.js';
import { DevicePageBrowserProxyImpl } from './device_page_browser_proxy.js';
import { FakeCrosAudioConfig } from './fake_cros_audio_config.js';
/** Utility for keeping percent in inclusive range of [0,100].  */
function clampPercent(percent) {
    return Math.max(0, Math.min(percent, 100));
}
const SettingsAudioElementBase = WebUiListenerMixin(DeepLinkingMixin(PrefsMixin(RouteObserverMixin(I18nMixin(PolymerElement)))));
const VOLUME_ICON_OFF_LEVEL = 0;
// TODO(b/271871947): Match volume icon logic to QS revamp sliders.
// Matches level calculated in unified_volume_view.cc.
const VOLUME_ICON_LOUD_LEVEL = 34;
export class SettingsAudioElement extends SettingsAudioElementBase {
    static get is() {
        return 'settings-audio';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            crosAudioConfig_: {
                type: Object,
            },
            audioSystemProperties_: {
                type: Object,
            },
            isOutputMuted_: {
                type: Boolean,
                reflectToAttribute: true,
            },
            isInputMuted_: {
                type: Boolean,
                reflectToAttribute: true,
            },
            showVoiceIsolationSubsection_: {
                type: Boolean,
            },
            /**
             * Enum values for the
             * 'ash.input_voice_isolation_preferred_effect' preference. These
             * values map to cras::AudioEffectType, and are written to prefs.
             */
            voiceIsolationEffectModePrefValues_: {
                readOnly: true,
                type: Object,
                value: {
                    STYLE_TRANSFER: AudioEffectType.kStyleTransfer,
                    BEAMFORMING: AudioEffectType.kBeamforming,
                },
            },
            outputVolume_: {
                type: Number,
            },
            powerSoundsHidden_: {
                type: Boolean,
                computed: 'computePowerSoundsHidden_(batteryStatus_)',
            },
            startupSoundEnabled_: {
                type: Boolean,
                value: false,
            },
            showAllowAGC: {
                type: Boolean,
                value: loadTimeData.getBoolean('enableForceRespectUiGainsToggle'),
                readonly: true,
            },
            isAllowAGCEnabled: {
                type: Boolean,
                value: true,
            },
            isHfpMicSrEnabled: {
                type: Boolean,
            },
            showSpatialAudio: {
                type: Boolean,
            },
            isSpatialAudioEnabled_: {
                type: Boolean,
                value: true,
            },
        };
    }
    constructor() {
        super();
        // DeepLinkingMixin override
        this.supportedSettingIds = new Set([
            Setting.kChargingSounds,
            Setting.kLowBatterySound,
        ]);
        this.crosAudioConfig_ = getCrosAudioConfig();
        this.audioSystemPropertiesObserverReceiver_ =
            new AudioSystemPropertiesObserverReceiver(this);
        this.audioAndCaptionsBrowserProxy_ =
            AudioAndCaptionsPageBrowserProxyImpl.getInstance();
        this.devicePageBrowserProxy_ = DevicePageBrowserProxyImpl.getInstance();
    }
    ready() {
        super.ready();
        this.observeAudioSystemProperties_();
        this.addWebUiListener('startup-sound-setting-retrieved', (startupSoundEnabled) => {
            this.startupSoundEnabled_ = startupSoundEnabled;
        });
        this.addWebUiListener('battery-status-changed', this.set.bind(this, 'batteryStatus_'));
        // Manually call updatePowerStatus to ensure batteryStatus_ is initialized
        // and up to date.
        this.devicePageBrowserProxy_.updatePowerStatus();
    }
    /**
     * AudioSystemPropertiesObserverInterface override
     */
    onPropertiesUpdated(properties) {
        this.audioSystemProperties_ = properties;
        this.isOutputMuted_ =
            this.audioSystemProperties_.outputMuteState !== MuteState.kNotMuted;
        this.isInputMuted_ =
            this.audioSystemProperties_.inputMuteState !== MuteState.kNotMuted;
        const activeInputDevice = this.audioSystemProperties_.inputDevices.find((device) => device.isActive);
        const toggleType = this.audioSystemProperties_.voiceIsolationUiAppearance.toggleType;
        this.showVoiceIsolationSubsection_ = toggleType !== AudioEffectType.kNone;
        this.isAllowAGCEnabled =
            (activeInputDevice?.forceRespectUiGainsState ===
                AudioEffectState.kNotEnabled);
        this.outputVolume_ = this.audioSystemProperties_.outputVolumePercent;
        this.isHfpMicSrEnabled =
            (activeInputDevice?.hfpMicSrState === AudioEffectState.kEnabled);
        this.isHfpMicSrSupported = activeInputDevice !== undefined &&
            activeInputDevice?.hfpMicSrState !== AudioEffectState.kNotSupported;
        const activeOutputDevice = this.audioSystemProperties_.outputDevices.find((device) => device.isActive);
        this.isSpatialAudioEnabled_ = activeOutputDevice !== undefined &&
            activeOutputDevice?.spatialAudioState === AudioEffectState.kEnabled;
        this.isSpatialAudioSupported_ = activeOutputDevice !== undefined &&
            activeOutputDevice?.spatialAudioState !==
                AudioEffectState.kNotSupported;
        this.showSpatialAudio =
            (this.isSpatialAudioSupported_ &&
                loadTimeData.getBoolean('enableSpatialAudioToggle'));
    }
    getIsOutputMutedForTest() {
        return this.isOutputMuted_;
    }
    getIsInputMutedForTest() {
        return this.isInputMuted_;
    }
    observeAudioSystemProperties_() {
        // Use fake observer implementation to access additional properties not
        // available on mojo interface.
        if (this.crosAudioConfig_ instanceof FakeCrosAudioConfig) {
            this.crosAudioConfig_.observeAudioSystemProperties(this);
            return;
        }
        this.crosAudioConfig_.observeAudioSystemProperties(this.audioSystemPropertiesObserverReceiver_.$
            .bindNewPipeAndPassRemote());
    }
    /** Determines if audio output is muted by policy. */
    isOutputMutedByPolicy_() {
        return this.audioSystemProperties_.outputMuteState ===
            MuteState.kMutedByPolicy;
    }
    onInputMuteClicked() {
        this.crosAudioConfig_.setInputMuted(!this.isInputMuted_);
    }
    /** Handles updating active input device. */
    onInputDeviceChanged() {
        const inputDeviceSelect = this.shadowRoot.querySelector('#audioInputDeviceDropdown');
        assert(!!inputDeviceSelect);
        this.crosAudioConfig_.setActiveDevice(BigInt(inputDeviceSelect.value));
    }
    /**
     * Handles the event where the input volume slider is being changed.
     */
    onInputVolumeSliderChanged() {
        const sliderValue = this.shadowRoot
            .querySelector('#audioInputGainVolumeSlider').value;
        this.crosAudioConfig_.setInputGainPercent(clampPercent(sliderValue));
    }
    /**
     * Handles the event where the output volume slider is being changed.
     */
    onOutputVolumeSliderChanged_() {
        const sliderValue = this.shadowRoot.querySelector('#outputVolumeSlider').value;
        this.crosAudioConfig_.setOutputVolumePercent(clampPercent(sliderValue));
    }
    /** Handles updating active output device. */
    onOutputDeviceChanged() {
        const outputDeviceSelect = this.shadowRoot.querySelector('#audioOutputDeviceDropdown');
        assert(!!outputDeviceSelect);
        this.crosAudioConfig_.setActiveDevice(BigInt(outputDeviceSelect.value));
    }
    /** Handles updating outputMuteState. */
    onOutputMuteButtonClicked() {
        this.crosAudioConfig_.setOutputMuted(!this.isOutputMuted_);
    }
    currentRouteChanged(route) {
        // Does not apply to this page.
        // TODO(crbug.com/1092970): Add DeepLinkingMixin and attempt deep link.
        if (route !== routes.AUDIO) {
            return;
        }
        this.audioAndCaptionsBrowserProxy_.getStartupSoundEnabled();
    }
    /** Handles updating the mic icon depending on the input mute state. */
    getInputIcon_() {
        return this.isInputMuted_ ? 'os-settings:mic-off' : 'cr:mic';
    }
    /**
     * Handles updating the output icon depending on the output mute state and
     * volume.
     */
    getOutputIcon_() {
        if (this.isOutputMuted_) {
            return 'os-settings:volume-up-off';
        }
        if (this.outputVolume_ === VOLUME_ICON_OFF_LEVEL) {
            return 'os-settings:volume-zero';
        }
        if (this.outputVolume_ < VOLUME_ICON_LOUD_LEVEL) {
            return 'os-settings:volume-down';
        }
        return 'os-settings:volume-up';
    }
    /**
     * Handles the case when there are no output devices. The output section
     * should be hidden in this case.
     */
    getOutputHidden_() {
        return this.audioSystemProperties_.outputDevices.length === 0;
    }
    /**
     * Handles the case when there are no input devices. The input section should
     * be hidden in this case.
     */
    getInputHidden_() {
        return this.audioSystemProperties_.inputDevices.length === 0;
    }
    /**
     * Returns true if input is muted by physical switch; otherwise, return false.
     */
    shouldDisableInputGainControls() {
        return this.audioSystemProperties_.inputMuteState ===
            MuteState.kMutedExternally;
    }
    /** Translates the device name if applicable. */
    getDeviceName_(audioDevice) {
        switch (audioDevice.deviceType) {
            case AudioDeviceType.kHeadphone:
                return this.i18n('audioDeviceHeadphoneLabel');
            case AudioDeviceType.kMic:
                return this.i18n('audioDeviceMicJackLabel');
            case AudioDeviceType.kUsb:
                return this.i18n('audioDeviceUsbLabel', audioDevice.displayName);
            case AudioDeviceType.kBluetooth:
            case AudioDeviceType.kBluetoothNbMic:
                return this.i18n('audioDeviceBluetoothLabel', audioDevice.displayName);
            case AudioDeviceType.kHdmi:
                return this.i18n('audioDeviceHdmiLabel', audioDevice.displayName);
            case AudioDeviceType.kInternalSpeaker:
                return this.i18n('audioDeviceInternalSpeakersLabel');
            case AudioDeviceType.kInternalMic:
                return this.i18n('audioDeviceInternalMicLabel');
            case AudioDeviceType.kFrontMic:
                return this.i18n('audioDeviceFrontMicLabel');
            case AudioDeviceType.kRearMic:
                return this.i18n('audioDeviceRearMicLabel');
            default:
                return audioDevice.displayName;
        }
    }
    /**
     * Returns the appropriate tooltip for output and input device mute buttons
     * based on `muteState`.
     */
    getMuteTooltip_(muteState) {
        switch (muteState) {
            case MuteState.kNotMuted:
                return this.i18n('audioToggleToMuteTooltip');
            case MuteState.kMutedByUser:
                return this.i18n('audioToggleToUnmuteTooltip');
            case MuteState.kMutedByPolicy:
                return this.i18n('audioMutedByPolicyTooltip');
            case MuteState.kMutedExternally:
                return this.i18n('audioMutedExternallyTooltip');
            default:
                return '';
        }
    }
    /** Returns the appropriate aria-label for input mute button. */
    getInputMuteButtonAriaLabel() {
        if (this.audioSystemProperties_.inputMuteState ===
            MuteState.kMutedExternally) {
            return this.i18n('audioInputMuteButtonAriaLabelMutedByHardwareSwitch');
        }
        return this.isInputMuted_ ?
            this.i18n('audioInputMuteButtonAriaLabelMuted') :
            this.i18n('audioInputMuteButtonAriaLabelNotMuted');
    }
    /** Returns the appropriate aria-label for output mute button. */
    getOutputMuteButtonAriaLabel() {
        return this.isOutputMuted_ ?
            this.i18n('audioOutputMuteButtonAriaLabelMuted') :
            this.i18n('audioOutputMuteButtonAriaLabelNotMuted');
    }
    getVoiceIsolationToggleTitle_(voiceIsolationUIAppearance) {
        if (voiceIsolationUIAppearance === undefined) {
            return '';
        }
        switch (voiceIsolationUIAppearance.toggleType) {
            case AudioEffectType.kNoiseCancellation:
                return this.i18n('audioInputNoiseCancellationTitle');
            case AudioEffectType.kStyleTransfer:
                return this.i18n('audioInputStyleTransferTitle');
            case AudioEffectType.kBeamforming:
                return this.i18n('audioInputBeamformingTitle');
            default:
                return '';
        }
    }
    getVoiceIsolationToggleDescription_(voiceIsolationUIAppearance) {
        if (voiceIsolationUIAppearance === undefined) {
            return '';
        }
        switch (voiceIsolationUIAppearance.toggleType) {
            case AudioEffectType.kStyleTransfer:
                return this.i18n('audioInputStyleTransferDescription');
            case AudioEffectType.kBeamforming:
                return this.i18n('audioInputBeamformingDescription');
            default:
                return '';
        }
    }
    shouldShowVoiceIsolationEffectModeOptions_(effectModeOptions, voiceIsolationEnabled) {
        return effectModeOptions !== 0 && voiceIsolationEnabled;
    }
    shouldShowVoiceIsolationFallbackMessage_(crasShowEffectFallbackMessage, voiceIsolationEnabled) {
        return crasShowEffectFallbackMessage && voiceIsolationEnabled;
    }
    toggleHfpMicSrEnabled_(e) {
        this.crosAudioConfig_.setHfpMicSrEnabled(e.detail);
    }
    toggleStartupSoundEnabled_(e) {
        this.audioAndCaptionsBrowserProxy_.setStartupSoundEnabled(e.detail);
    }
    toggleAllowAgcEnabled_(e) {
        this.crosAudioConfig_.setForceRespectUiGainsEnabled(!e.detail);
    }
    toggleSpatialAudioEnabled_(e) {
        this.crosAudioConfig_.setSpatialAudioEnabled(e.detail);
    }
    computePowerSoundsHidden_() {
        return !this.batteryStatus_?.present;
    }
    onDeviceStartupSoundRowClicked_() {
        this.startupSoundEnabled_ = !this.startupSoundEnabled_;
        this.audioAndCaptionsBrowserProxy_.setStartupSoundEnabled(this.startupSoundEnabled_);
    }
    onVoiceIsolationRowClicked_() {
        // The pref change will be handled by
        // CrasAudioHandler::OnVoiceIsolationPrefChanged().
        // See also AudioDevicesPrefHandlerImpl::InitializePrefObservers().
        this.crosAudioConfig_.recordVoiceIsolationEnabledChange();
    }
    onVoiceIsolationEffectModeChanged_() {
        // The pref change will be handled by
        // CrasAudioHandler::OnVoiceIsolationPrefChanged().
        // See also AudioDevicesPrefHandlerImpl::InitializePrefObservers().
        // The pref value might not be updated yet, so only getting
        // the UI value here to record to UMA.
        const voiceIsolationEffectModes = strictQuery('#voiceIsolationEffectModeOptions', this.shadowRoot, SettingsRadioGroupElement);
        const selected = +voiceIsolationEffectModes.selected;
        this.crosAudioConfig_.recordVoiceIsolationPreferredEffectChange(selected);
    }
    onSpatialAudioRowClicked_() {
        const spatialAudioToggle = strictQuery('#audioOutputSpatialAudioToggle', this.shadowRoot, CrToggleElement);
        this.crosAudioConfig_.setSpatialAudioEnabled(!spatialAudioToggle.checked);
    }
}
customElements.define(SettingsAudioElement.is, SettingsAudioElement);
