// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import 'chrome://resources/polymer/v3_0/iron-icon/iron-icon.js';
import 'chrome://resources/ash/common/cr_elements/md_select.css.js';
import './customize_button_select.js';
import '../settings_shared.css.js';
import '../controls/settings_dropdown_menu.js';
import '../os_settings_icons.html.js';
import { I18nMixin } from 'chrome://resources/ash/common/cr_elements/i18n_mixin.js';
import { strictQuery } from 'chrome://resources/ash/common/typescript_utils/strict_query.js';
import { assert } from 'chrome://resources/js/assert.js';
import { PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { ButtonPressObserverReceiver } from '../mojom-webui/input_device_settings_provider.mojom-webui.js';
import { getTemplate } from './customize_button_row.html.js';
import { setDataTransferOriginIndex } from './drag_and_drop_manager.js';
import { FakeInputDeviceSettingsProvider } from './fake_input_device_settings_provider.js';
import { getInputDeviceSettingsProvider } from './input_device_mojo_interface_provider.js';
import { MetaKey } from './input_device_settings_types.js';
import { buttonsAreEqual } from './input_device_settings_utils.js';
const CustomizeButtonRowElementBase = I18nMixin(PolymerElement);
export class CustomizeButtonRowElement extends CustomizeButtonRowElementBase {
    constructor() {
        super(...arguments);
        this.metaKey = MetaKey.kSearch;
        this.inputDeviceSettingsProvider_ = getInputDeviceSettingsProvider();
    }
    static get is() {
        return 'customize-button-row';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            buttonRemappingList: {
                type: Array,
            },
            buttonRemapping_: {
                type: Object,
            },
            remappingIndex: {
                type: Number,
            },
            actionList: {
                type: Array,
            },
            /**
             * Name of the remapping.
             */
            buttonRemappingName_: {
                type: String,
                value: '',
                computed: 'getButtonRemappingName_(buttonRemappingList.*, remappingIndex)',
            },
            /**
             * True if this element is being dragged by its handle.
             * This property is used to apply custom styling to the element when it's
             * being dragged.
             */
            isBeingDragged_: {
                type: Boolean,
                reflectToAttribute: true,
            },
            /**
             * Reference to the HTML element that was last pressed
             * on. This property is used to determine if a drag event was started
             * from this element's drag handle or elsewhere on the element..
             */
            lastMouseDownTarget_: {
                type: Object,
                value: null,
            },
            metaKey: Object,
        };
    }
    static get observers() {
        return [
            'initializeButtonRow_(buttonRemappingList.*, remappingIndex)',
        ];
    }
    connectedCallback() {
        super.connectedCallback();
        this.observeButtonPresses();
        // Focus dropdown right away as this button was just pressed.
        this.$.remappingActionDropdown.focus();
        this.$.reorderButton.addEventListener('keydown', this.handleKeyDownReorderButton_);
        this.addEventListener('reorder-button-direction', this.onButtonReorderDirectEvent_);
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.$.reorderButton.removeEventListener('keydown', this.handleKeyDownReorderButton_);
        this.removeEventListener('reorder-button-direction', this.onButtonReorderDirectEvent_);
    }
    /**
     * Focuses the reordering button for this row.
     */
    focusReorderingButton() {
        this.$.reorderButton.focus();
    }
    focus() {
        assert(this.$.remappingActionDropdown);
        this.$.remappingActionDropdown.focus();
    }
    observeButtonPresses() {
        if (this.inputDeviceSettingsProvider_ instanceof
            FakeInputDeviceSettingsProvider) {
            this.inputDeviceSettingsProvider_.observeButtonPresses(this);
            return;
        }
        this.buttonPressObserverReceiver = new ButtonPressObserverReceiver(this);
        this.inputDeviceSettingsProvider_.observeButtonPresses(this.buttonPressObserverReceiver.$.bindNewPipeAndPassRemote());
    }
    /**
     * Initialize the button remapping content and set up fake pref.
     */
    initializeButtonRow_() {
        if (!this.buttonRemappingList ||
            !this.buttonRemappingList[this.remappingIndex]) {
            return;
        }
        if (this.remappingIndex === 0) {
            this.$.container.classList.add('first');
        }
        this.buttonRemapping_ = this.buttonRemappingList[this.remappingIndex];
    }
    /**
     * Pops out the dialog to edit button label.
     */
    onEditButtonLabelClicked_() {
        this.dispatchEvent(new CustomEvent('show-renaming-dialog', {
            bubbles: true,
            composed: true,
            detail: { buttonIndex: this.remappingIndex },
        }));
    }
    /**
     * Get the button remapping name when initializing or users updated it.
     */
    getButtonRemappingName_() {
        if (!!this.buttonRemappingList &&
            !!this.buttonRemappingList[this.remappingIndex]) {
            return this.buttonRemappingList[this.remappingIndex].name;
        }
        return '';
    }
    onButtonPressed(button) {
        if (buttonsAreEqual(button, this.buttonRemapping_.button)) {
            this.$.remappingActionDropdown.focus();
        }
    }
    onContainerMouseDown_(event) {
        this.lastMouseDownTarget_ = event.target;
    }
    onDragStart_(event) {
        const dragHandle = strictQuery('.move-button', this.shadowRoot, HTMLElement);
        // Check if the drag event started from the drag handle.
        if (!dragHandle.contains(this.lastMouseDownTarget_)) {
            // Drag didn't start from the handle, so don't allow the drag.
            event.preventDefault();
            return;
        }
        this.isBeingDragged_ = true;
        // Create the grey rectangle used as the drag image.
        // It's necessary to create a canvas element this way, so that we have
        // full control over the rendering of the drag image.
        const canvas = document.createElement('canvas');
        document.body.append(canvas);
        // We won't need the canvas after this function finishes, so
        // remove it from the DOM.
        setTimeout(() => canvas.remove(), 100);
        canvas.width = this.offsetWidth;
        canvas.height = this.offsetHeight;
        // Position the canvas offscreen.
        canvas.style.position = 'absolute';
        canvas.style.top = '-200px';
        // Using the canvas context, draw the grey rectangle that will be displayed
        // while the user is dragging.
        const context = canvas.getContext('2d');
        if (context) {
            // Using getComputedStyle and getPropertyValue is the only
            // way to use CSS variables with canvas painting.
            context.fillStyle = getComputedStyle(canvas).getPropertyValue('--cros-sys-ripple_neutral_on_subtle');
            context.fillRect(0, 0, canvas.width, canvas.height);
            context.strokeStyle = getComputedStyle(canvas).getPropertyValue('--cros-sys-highlight_shape');
            context.lineWidth = 1;
            context.strokeRect(0, 0, canvas.width, canvas.height);
        }
        if (event.dataTransfer) {
            event.dataTransfer.setDragImage(canvas, event.offsetX, event.offsetY);
            // dropEffect and effectAllowed affect the cursor that's
            // displayed during the drag.
            event.dataTransfer.dropEffect = 'move';
            event.dataTransfer.effectAllowed = 'move';
            // Set data on the event that allows the drop receiver to determine
            // the index of this row.
            setDataTransferOriginIndex(event, this.remappingIndex);
        }
    }
    onDragEnd_() {
        this.isBeingDragged_ = false;
    }
    isDropdownDisabled_() {
        return this.isBeingDragged_;
    }
    getReorderButtonLabel_() {
        return this.i18n('buttonReorderingAriaLabel', this.buttonRemappingName_);
    }
    onButtonReorderDirectEvent_(e) {
        const destinationIndex = this.remappingIndex + (e.detail.direction ? -1 : 1);
        this.dispatchEvent(new CustomEvent('reorder-button', {
            bubbles: true,
            composed: true,
            detail: { originIndex: this.remappingIndex, destinationIndex },
        }));
    }
    handleKeyDownReorderButton_(e) {
        if (!e.ctrlKey) {
            return;
        }
        const isArrowUp = e.key === 'ArrowUp';
        const isArrowDown = e.key === 'ArrowDown';
        if (isArrowUp || isArrowDown) {
            this.dispatchEvent(new CustomEvent('reorder-button-direction', {
                bubbles: true,
                composed: true,
                detail: { direction: isArrowUp ? true : false },
            }));
        }
    }
}
customElements.define(CustomizeButtonRowElement.is, CustomizeButtonRowElement);
