// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { AudioOutputCapability, DeviceConnectionState, DeviceType } from 'chrome://resources/mojo/chromeos/ash/services/bluetooth_config/public/mojom/cros_bluetooth_config.mojom-webui.js';
import { PolicyStatus } from './input_device_settings_types.js';
function objectsAreEqual(obj1, obj2) {
    const keys1 = Object.keys(obj1);
    const keys2 = Object.keys(obj2);
    if (keys1.length !== keys2.length) {
        return false;
    }
    for (let i = 0; i < keys1.length; i++) {
        const key = keys1[i];
        const val1 = obj1[key];
        const val2 = obj2[key];
        if (val1 instanceof Object || val2 instanceof Object) {
            if (!(val1 instanceof Object) || !(val2 instanceof Object) ||
                !objectsAreEqual(val1, val2)) {
                return false;
            }
        }
        else if (val1 !== val2) {
            return false;
        }
    }
    return true;
}
function deviceInList(deviceId, deviceList) {
    for (const device of deviceList) {
        if (device.id === deviceId) {
            return true;
        }
    }
    return false;
}
export function settingsAreEqual(settings1, settings2) {
    return objectsAreEqual(settings1, settings2);
}
export function buttonsAreEqual(button1, button2) {
    return objectsAreEqual(button1, button2);
}
export function keyEventsAreEqual(keyEvent1, keyEvent2) {
    return objectsAreEqual(keyEvent1, keyEvent2);
}
export function getPrefPolicyFields(policy) {
    if (policy) {
        const enforcement = policy.policyStatus === PolicyStatus.kManaged ?
            chrome.settingsPrivate.Enforcement.ENFORCED :
            chrome.settingsPrivate.Enforcement.RECOMMENDED;
        return {
            controlledBy: chrome.settingsPrivate.ControlledBy.USER_POLICY,
            enforcement,
            recommendedValue: policy.value,
        };
    }
    // These fields must be set back to undefined so the html badge is properly
    // removed from the UI.
    return {
        controlledBy: undefined,
        enforcement: undefined,
        recommendedValue: undefined,
    };
}
export function getDeviceStateChangesToAnnounce(newDeviceList, prevDeviceList) {
    let msgId;
    let devices;
    if (newDeviceList.length > prevDeviceList.length) {
        devices = newDeviceList.filter((device) => !deviceInList(device.id, prevDeviceList));
        msgId = 'deviceConnectedA11yLabel';
    }
    else {
        msgId = 'deviceDisconnectedA11yLabel';
        devices = prevDeviceList.filter((device) => !deviceInList(device.id, newDeviceList));
    }
    return { msgId, deviceNames: devices.map(device => device.name) };
}
export function createBluetoothDeviceProperties(id, publicName, batteryPercentage) {
    return {
        id: id,
        address: id,
        publicName: publicName,
        deviceType: DeviceType.kMouse,
        audioCapability: AudioOutputCapability.kNotCapableOfAudioOutput,
        connectionState: DeviceConnectionState.kConnected,
        isBlockedByPolicy: false,
        batteryInfo: {
            defaultProperties: { batteryPercentage },
            leftBudInfo: null,
            rightBudInfo: null,
            caseInfo: null,
        },
        imageInfo: null,
    };
}
