// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import 'chrome://resources/ash/common/cr_elements/cr_button/cr_button.js';
import 'chrome://resources/ash/common/cr_elements/cr_dialog/cr_dialog.js';
import 'chrome://resources/ash/common/cr_elements/cr_radio_button/cr_radio_button.js';
import 'chrome://resources/ash/common/cr_elements/cr_radio_group/cr_radio_group.js';
import 'chrome://resources/ash/common/cr_elements/icons.html.js';
import 'chrome://resources/polymer/v3_0/iron-icon/iron-icon.js';
import '../settings_shared.css.js';
import '../os_settings_icons.html.js';
import { PrefsMixin } from '/shared/settings/prefs/prefs_mixin.js';
import { PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { assertExhaustive } from '../assert_extras.js';
import { DevicePageBrowserProxyImpl, OptimizedChargingStrategy } from './device_page_browser_proxy.js';
import { SettingsPowerElement } from './power.js';
import { getTemplate } from './power_optimized_charging_dialog.html.js';
/**
 * Names of the radio buttons which allow the user to choose their optimized
 * charging strategy.
 */
var OptimizedChargingButtons;
(function (OptimizedChargingButtons) {
    OptimizedChargingButtons["ADAPTIVE_CHARGING"] = "adaptive-charging";
    OptimizedChargingButtons["CHARGE_LIMIT"] = "charge-limit";
})(OptimizedChargingButtons || (OptimizedChargingButtons = {}));
const PowerOptimizedChargingDialogElementBase = PrefsMixin(PolymerElement);
export class PowerOptimizedChargingDialogElement extends PowerOptimizedChargingDialogElementBase {
    static get is() {
        return 'power-optimized-charging-dialog';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            previousStrategy: { type: Number, computed: `getPreviousPrefValue_()` },
            selectedOption_: {
                type: String,
                computed: 'computeButtonName_(previousStrategy)',
            },
        };
    }
    constructor() {
        super();
        this.browserProxy_ = DevicePageBrowserProxyImpl.getInstance();
    }
    onDoneClick_() {
        const uiSelectedButton = this.$.radioGroup.selected;
        const newlySelectedStrategy = this.convertRadioNameToStrategyEnum_(uiSelectedButton);
        if (this.previousStrategy !== newlySelectedStrategy) {
            this.setPrefValue(SettingsPowerElement.OPTIMIZED_CHARGING_STRATEGY_PREF_NAME, newlySelectedStrategy);
        }
        this.$.dialog.close();
    }
    onCancelClick_() {
        this.$.dialog.cancel();
    }
    computeButtonName_(strategy) {
        switch (strategy) {
            case OptimizedChargingStrategy.STRATEGY_ADAPTIVE_CHARGING:
                return OptimizedChargingButtons.ADAPTIVE_CHARGING;
            case OptimizedChargingStrategy.STRATEGY_CHARGE_LIMIT:
                return OptimizedChargingButtons.CHARGE_LIMIT;
            default:
                assertExhaustive(strategy);
        }
    }
    getPreviousPrefValue_() {
        return this.getPref(SettingsPowerElement.OPTIMIZED_CHARGING_STRATEGY_PREF_NAME)
            .value;
    }
    convertRadioNameToStrategyEnum_(radioButtonName) {
        switch (radioButtonName) {
            case OptimizedChargingButtons.ADAPTIVE_CHARGING:
                return OptimizedChargingStrategy.STRATEGY_ADAPTIVE_CHARGING;
            case OptimizedChargingButtons.CHARGE_LIMIT:
                return OptimizedChargingStrategy.STRATEGY_CHARGE_LIMIT;
            default:
                assertExhaustive(radioButtonName);
        }
    }
}
customElements.define(PowerOptimizedChargingDialogElement.is, PowerOptimizedChargingDialogElement);
